@extends('layouts.app')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <!-- Header -->
    <div class="mb-6">
        <h1 class="text-2xl font-bold text-gray-900 dark:text-white">👥 Seller KPI Performance</h1>
        <p class="text-gray-600 dark:text-gray-400">Track individual seller performance and achievements</p>
    </div>

    @include('forecasting.partials.navigation')

    <!-- Filters -->
    <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-4 mb-6">
        <form method="GET" action="{{ route('forecasting.seller-kpi.index') }}" class="flex flex-wrap gap-4">
            <div>
                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Year</label>
                <select name="year" class="form-select rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white">
                    @for($y = date('Y'); $y >= 2022; $y--)
                        <option value="{{ $y }}" {{ $year == $y ? 'selected' : '' }}>{{ $y }}</option>
                    @endfor
                </select>
            </div>

            <div>
                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Month</label>
                <select name="month" class="form-select rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white">
                    @foreach(['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'] as $m => $monthName)
                        <option value="{{ $m + 1 }}" {{ $month == ($m + 1) ? 'selected' : '' }}>{{ $monthName }}</option>
                    @endforeach
                </select>
            </div>

            <div>
                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Grade Filter</label>
                <select name="grade" class="form-select rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white">
                    <option value="">All Grades</option>
                    <option value="A" {{ request('grade') == 'A' ? 'selected' : '' }}>A (Exceeding)</option>
                    <option value="B" {{ request('grade') == 'B' ? 'selected' : '' }}>B (Good)</option>
                    <option value="C" {{ request('grade') == 'C' ? 'selected' : '' }}>C (Average)</option>
                    <option value="D" {{ request('grade') == 'D' ? 'selected' : '' }}>D (Below)</option>
                    <option value="F" {{ request('grade') == 'F' ? 'selected' : '' }}>F (Needs Attention)</option>
                </select>
            </div>

            <div class="flex items-end">
                <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-md">
                    Filter
                </button>
            </div>

            <div class="flex items-end ml-auto gap-2">
                <a href="{{ route('forecasting.seller-kpi.set-targets') }}" class="bg-purple-600 hover:bg-purple-700 text-white px-4 py-2 rounded-md">
                    Set Targets
                </a>
                <button type="button" onclick="syncSales()" class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded-md">
                    Sync Sales Data
                </button>
            </div>
        </form>
    </div>

    <!-- Summary Cards -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-4">
            <div class="text-sm text-gray-600 dark:text-gray-400 mb-1">Total Sellers</div>
            <div class="text-2xl font-bold text-gray-800 dark:text-white">{{ $summary['total_sellers'] }}</div>
        </div>

        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-4">
            <div class="text-sm text-gray-600 dark:text-gray-400 mb-1">
                Total Sales (Direct Sales Only)
                <span class="text-xs block text-gray-500 dark:text-gray-500 mt-1">Sum of all sellers' direct sales</span>
            </div>
            <div class="text-2xl font-bold text-gray-800 dark:text-white">
                RM {{ number_format($summary['total_sales'], 2) }}
            </div>
        </div>

        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-4">
            <div class="text-sm text-gray-600 dark:text-gray-400 mb-1">Avg Achievement</div>
            <div class="text-2xl font-bold text-gray-800 dark:text-white">
                {{ number_format($summary['avg_achievement'], 1) }}%
            </div>
        </div>

        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-4">
            <div class="text-sm text-gray-600 dark:text-gray-400 mb-1">Grade Distribution</div>
            <div class="flex gap-2 text-sm mt-2">
                <span class="bg-green-100 text-green-800 px-2 py-1 rounded">A: {{ $summary['grades']['A'] }}</span>
                <span class="bg-blue-100 text-blue-800 px-2 py-1 rounded">B: {{ $summary['grades']['B'] }}</span>
                <span class="bg-yellow-100 text-yellow-800 px-2 py-1 rounded">C: {{ $summary['grades']['C'] }}</span>
                <span class="bg-orange-100 text-orange-800 px-2 py-1 rounded">D: {{ $summary['grades']['D'] }}</span>
                <span class="bg-red-100 text-red-800 px-2 py-1 rounded">F: {{ $summary['grades']['F'] }}</span>
            </div>
        </div>
    </div>

    <!-- Sellers Table -->
    <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm overflow-hidden">
        <div class="overflow-x-auto">
            <table class="w-full">
                <thead class="bg-gray-50 dark:bg-gray-700">
                    <tr>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Seller</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Target (RM)</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Actual (RM)</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Achievement</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Grade</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Trend</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Orders</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">AOV (RM)</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Actions</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-gray-200 dark:divide-gray-700">
                    @forelse($enrichedForecasts as $forecast)
                        <tr class="hover:bg-gray-50 dark:hover:bg-gray-700">
                            <td class="px-4 py-3 whitespace-nowrap">
                                <div class="flex items-center gap-2">
                                    <span class="text-sm font-medium text-gray-900 dark:text-white">{{ $forecast->user->name }}</span>
                                    @if($forecast->user->isManager())
                                        <span class="text-xs bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-300 px-2 py-0.5 rounded">Manager</span>
                                    @endif
                                </div>
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-600 dark:text-gray-400">
                                {{ number_format($forecast->target_sales, 2) }}
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap">
                                <div class="text-sm font-semibold text-gray-900 dark:text-white">
                                    {{ number_format($forecast->cumulative_sales, 2) }}
                                </div>
                                @if($forecast->user->isManager() && $forecast->team_sales > 0)
                                    <div class="text-xs text-gray-500 dark:text-gray-400">
                                        Direct: {{ number_format($forecast->actual_sales, 2) }} | Team: {{ number_format($forecast->team_sales, 2) }}
                                    </div>
                                @endif
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap">
                                <span class="text-sm font-semibold {{ $forecast->cumulative_achievement >= 100 ? 'text-green-600' : ($forecast->cumulative_achievement >= 80 ? 'text-blue-600' : 'text-red-600') }}">
                                    {{ number_format($forecast->cumulative_achievement, 1) }}%
                                </span>
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap">
                                @php
                                    $gradeColors = [
                                        'A' => 'bg-green-100 text-green-800',
                                        'B' => 'bg-blue-100 text-blue-800',
                                        'C' => 'bg-yellow-100 text-yellow-800',
                                        'D' => 'bg-orange-100 text-orange-800',
                                        'F' => 'bg-red-100 text-red-800',
                                    ];
                                @endphp
                                <span class="px-2 py-1 inline-flex text-xs leading-5 font-semibold rounded-full {{ $gradeColors[$forecast->cumulative_grade] ?? 'bg-gray-100 text-gray-800' }}">
                                    {{ $forecast->cumulative_grade }}
                                </span>
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap">
                                @if($forecast->trend == 'growing')
                                    <span class="text-green-600">↗ Growing</span>
                                @elseif($forecast->trend == 'declining')
                                    <span class="text-red-600">↘ Declining</span>
                                @else
                                    <span class="text-gray-600">→ Stable</span>
                                @endif
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-600 dark:text-gray-400">
                                {{ number_format($forecast->orders_count) }}
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-600 dark:text-gray-400">
                                {{ number_format($forecast->aov, 2) }}
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap text-sm">
                                <a href="{{ route('forecasting.seller-kpi.show', $forecast->user_id) }}" class="text-blue-600 hover:text-blue-800">
                                    View Details
                                </a>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="9" class="px-4 py-8 text-center text-gray-500 dark:text-gray-400">
                                No seller data found for this period. Click "Sync Sales Data" to load from orders.
                            </td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
function syncSales() {
    if (!confirm('This will sync sales data from the orders table. Continue?')) return;

    const year = document.querySelector('select[name="year"]').value;
    const month = document.querySelector('select[name="month"]').value;

    fetch('{{ route("forecasting.seller-kpi.sync") }}', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        },
        body: JSON.stringify({ year, month })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert(data.message);
            window.location.reload();
        } else {
            alert('Error: ' + data.message);
        }
    })
    .catch(error => {
        alert('Error syncing data: ' + error);
    });
}
</script>
@endsection
