@extends('layouts.app')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <!-- Header -->
    <div class="mb-6 flex items-center justify-between">
        <div>
            <h1 class="text-2xl font-bold text-gray-900 dark:text-white">🎯 Marketing Strategy Dashboard</h1>
            <p class="text-gray-600 dark:text-gray-400">ROI Analysis & AI-Powered Recommendations</p>
        </div>
        <div>
            <a href="{{ route('forecasting.marketing-strategy.input') }}" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-md">
                Input Marketing Data
            </a>
        </div>
    </div>

    @include('forecasting.partials.navigation')

    <!-- Year Selector -->
    <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-4 mb-6">
        <form method="GET" action="{{ route('forecasting.marketing-strategy.index') }}" class="flex gap-4">
            <div>
                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Year</label>
                <select name="year" class="form-select rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white" onchange="this.form.submit()">
                    @for($y = date('Y'); $y >= 2022; $y--)
                        <option value="{{ $y }}" {{ $year == $y ? 'selected' : '' }}>{{ $y }}</option>
                    @endfor
                </select>
            </div>
        </form>
    </div>

    <!-- Revenue Mode Toggle -->
    <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-4 mb-4">
        <div class="flex items-center justify-between">
            <div>
                <h3 class="text-sm font-semibold text-gray-800 dark:text-white">Revenue Display Mode</h3>
                <p class="text-xs text-gray-600 dark:text-gray-400 mt-1">Toggle to see marketing effectiveness on HQ sales only</p>
            </div>
            <div class="flex items-center gap-3">
                <span class="text-xs font-medium text-gray-600 dark:text-gray-400" id="dashboardToggleLabel">All Sales (HQ + Stokis + Agent)</span>
                <button type="button" onclick="toggleDashboardRevenue()"
                        class="relative inline-flex h-6 w-11 items-center rounded-full bg-gray-300 dark:bg-gray-600 transition-colors focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2"
                        id="dashboardRevenueToggle">
                    <span class="inline-block h-4 w-4 transform rounded-full bg-white transition-transform translate-x-1" id="dashboardToggleThumb"></span>
                </button>
            </div>
        </div>
    </div>

    <!-- Year Summary Cards -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-4">
            <div class="text-sm text-gray-600 dark:text-gray-400 mb-1">Total Marketing Spend</div>
            <div class="text-2xl font-bold text-gray-800 dark:text-white">
                RM {{ number_format($yearSummary['total_spend'], 2) }}
            </div>
        </div>

        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-4">
            <div class="text-sm text-gray-600 dark:text-gray-400 mb-1">Total Revenue</div>
            <div id="dashboardAllRevenue">
                <div class="text-2xl font-bold text-gray-800 dark:text-white">
                    RM {{ number_format($yearSummary['total_revenue'], 2) }}
                </div>
                <div class="text-xs text-gray-500 dark:text-gray-400 mt-1">All Sales</div>
            </div>
            <div id="dashboardHqRevenue" class="hidden">
                <div class="text-2xl font-bold text-purple-800 dark:text-purple-300">
                    RM {{ number_format($yearSummary['hq_revenue'], 2) }}
                </div>
                <div class="text-xs text-purple-600 dark:text-purple-400 mt-1">HQ Sales Only</div>
            </div>
        </div>

        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-4">
            <div class="text-sm text-gray-600 dark:text-gray-400 mb-1">Average ROI</div>
            <div id="dashboardAllRoi">
                <div class="text-2xl font-bold {{ $yearSummary['avg_roi'] >= 100 ? 'text-green-600' : 'text-orange-600' }}">
                    {{ number_format($yearSummary['avg_roi'], 1) }}%
                </div>
                @php
                    $allRatio = $yearSummary['total_spend'] > 0 ? $yearSummary['total_revenue'] / $yearSummary['total_spend'] : 0;
                @endphp
                <div class="text-sm text-gray-700 dark:text-gray-300 font-semibold mt-1">
                    Ratio: {{ number_format($allRatio, 2) }}x
                </div>
                <div class="text-xs text-gray-500 dark:text-gray-400 mt-1">Based on All Sales</div>
            </div>
            <div id="dashboardHqRoi" class="hidden">
                <div class="text-2xl font-bold {{ $yearSummary['hq_roi'] >= 100 ? 'text-green-600' : 'text-orange-600' }}">
                    {{ number_format($yearSummary['hq_roi'], 1) }}%
                </div>
                @php
                    $hqRatio = $yearSummary['total_spend'] > 0 ? $yearSummary['hq_revenue'] / $yearSummary['total_spend'] : 0;
                @endphp
                <div class="text-sm text-purple-700 dark:text-purple-300 font-semibold mt-1">
                    Ratio: {{ number_format($hqRatio, 2) }}x
                </div>
                <div class="text-xs text-purple-600 dark:text-purple-400 mt-1">Based on HQ Sales</div>
            </div>
        </div>

        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-4">
            <div class="text-sm text-gray-600 dark:text-gray-400 mb-1">Profit from Marketing</div>
            <div id="dashboardAllProfit">
                <div class="text-2xl font-bold text-green-600">
                    RM {{ number_format($yearSummary['total_revenue'] - $yearSummary['total_spend'], 2) }}
                </div>
                <div class="text-xs text-gray-500 dark:text-gray-400 mt-1">All Sales</div>
            </div>
            <div id="dashboardHqProfit" class="hidden">
                <div class="text-2xl font-bold {{ ($yearSummary['hq_revenue'] - $yearSummary['total_spend']) >= 0 ? 'text-green-600' : 'text-red-600' }}">
                    RM {{ number_format($yearSummary['hq_revenue'] - $yearSummary['total_spend'], 2) }}
                </div>
                <div class="text-xs text-purple-600 dark:text-purple-400 mt-1">HQ Sales Only</div>
            </div>
        </div>
    </div>

    <!-- Best/Worst Month - All Sales View -->
    <div id="bestWorstAllSales" class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
        @if($yearSummary['best_month'])
        <div class="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-4">
            <div class="text-sm font-medium text-green-800 dark:text-green-400 mb-2">Best Performing Month (All Sales)</div>
            <div class="text-lg font-bold text-green-900 dark:text-green-300">
                {{ $yearSummary['best_month']->month_name }}: {{ number_format($yearSummary['best_month']->roi_percentage, 1) }}% ROI
            </div>
            @php
                $bestRatio = $yearSummary['best_month']->total_marketing_spend > 0
                    ? $yearSummary['best_month']->actual_revenue / $yearSummary['best_month']->total_marketing_spend
                    : 0;
            @endphp
            <div class="text-sm font-semibold text-green-800 dark:text-green-300">
                Ratio: {{ number_format($bestRatio, 2) }}x
            </div>
            <div class="text-sm text-green-700 dark:text-green-400 mt-1">
                Spend: RM {{ number_format($yearSummary['best_month']->total_marketing_spend, 2) }} →
                Revenue: RM {{ number_format($yearSummary['best_month']->actual_revenue, 2) }}
            </div>
        </div>
        @endif

        @if($yearSummary['worst_month'])
        <div class="bg-orange-50 dark:bg-orange-900/20 border border-orange-200 dark:border-orange-800 rounded-lg p-4">
            <div class="text-sm font-medium text-orange-800 dark:text-orange-400 mb-2">Needs Optimization (All Sales)</div>
            <div class="text-lg font-bold text-orange-900 dark:text-orange-300">
                {{ $yearSummary['worst_month']->month_name }}: {{ number_format($yearSummary['worst_month']->roi_percentage, 1) }}% ROI
            </div>
            @php
                $worstRatio = $yearSummary['worst_month']->total_marketing_spend > 0
                    ? $yearSummary['worst_month']->actual_revenue / $yearSummary['worst_month']->total_marketing_spend
                    : 0;
            @endphp
            <div class="text-sm font-semibold text-orange-800 dark:text-orange-300">
                Ratio: {{ number_format($worstRatio, 2) }}x
            </div>
            <div class="text-sm text-orange-700 dark:text-orange-400 mt-1">
                Spend: RM {{ number_format($yearSummary['worst_month']->total_marketing_spend, 2) }} →
                Revenue: RM {{ number_format($yearSummary['worst_month']->actual_revenue, 2) }}
            </div>
        </div>
        @endif
    </div>

    <!-- Best/Worst Month - HQ Only View -->
    <div id="bestWorstHqOnly" class="hidden grid-cols-1 md:grid-cols-2 gap-4 mb-6">
        @if($yearSummary['best_month_hq'])
        <div class="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-4">
            <div class="text-sm font-medium text-green-800 dark:text-green-400 mb-2">Best Performing Month (HQ Only)</div>
            <div class="text-lg font-bold text-green-900 dark:text-green-300">
                {{ $yearSummary['best_month_hq']->month_name }}: {{ number_format($yearSummary['best_month_hq']->hq_roi_percentage, 1) }}% ROI
            </div>
            @php
                $bestHqRatio = $yearSummary['best_month_hq']->total_marketing_spend > 0
                    ? $yearSummary['best_month_hq']->hq_revenue / $yearSummary['best_month_hq']->total_marketing_spend
                    : 0;
            @endphp
            <div class="text-sm font-semibold text-green-800 dark:text-green-300">
                Ratio: {{ number_format($bestHqRatio, 2) }}x
            </div>
            <div class="text-sm text-green-700 dark:text-green-400 mt-1">
                Spend: RM {{ number_format($yearSummary['best_month_hq']->total_marketing_spend, 2) }} →
                HQ Revenue: RM {{ number_format($yearSummary['best_month_hq']->hq_revenue, 2) }}
            </div>
        </div>
        @endif

        @if($yearSummary['worst_month_hq'])
        <div class="bg-orange-50 dark:bg-orange-900/20 border border-orange-200 dark:border-orange-800 rounded-lg p-4">
            <div class="text-sm font-medium text-orange-800 dark:text-orange-400 mb-2">Needs Optimization (HQ Only)</div>
            <div class="text-lg font-bold text-orange-900 dark:text-orange-300">
                {{ $yearSummary['worst_month_hq']->month_name }}: {{ number_format($yearSummary['worst_month_hq']->hq_roi_percentage, 1) }}% ROI
            </div>
            @php
                $worstHqRatio = $yearSummary['worst_month_hq']->total_marketing_spend > 0
                    ? $yearSummary['worst_month_hq']->hq_revenue / $yearSummary['worst_month_hq']->total_marketing_spend
                    : 0;
            @endphp
            <div class="text-sm font-semibold text-orange-800 dark:text-orange-300">
                Ratio: {{ number_format($worstHqRatio, 2) }}x
            </div>
            <div class="text-sm text-orange-700 dark:text-orange-400 mt-1">
                Spend: RM {{ number_format($yearSummary['worst_month_hq']->total_marketing_spend, 2) }} →
                HQ Revenue: RM {{ number_format($yearSummary['worst_month_hq']->hq_revenue, 2) }}
            </div>
        </div>
        @endif
    </div>

    <!-- Channel Performance -->
    @if($yearSummary['channel_performance'])
    <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-6 mb-6">
        <h2 class="text-xl font-bold text-gray-800 dark:text-white mb-4">Channel Performance (Average Spend)</h2>
        <div class="grid grid-cols-1 md:grid-cols-5 gap-4">
            @foreach($yearSummary['channel_performance'] as $channelKey => $channel)
                @php
                    $channelNames = [
                        'brand_ambassadors' => 'Brand Ambassadors',
                        'ads' => 'Ads',
                        'social_media' => 'Social Media',
                        'events' => 'Events',
                        'other' => 'Other'
                    ];
                @endphp
                <div class="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                    <div class="text-sm text-gray-600 dark:text-gray-400 mb-1">{{ $channelNames[$channelKey] }}</div>
                    <div class="text-lg font-bold text-gray-800 dark:text-white">
                        RM {{ number_format($channel['average'], 2) }}
                    </div>
                    <div class="text-xs text-gray-500 dark:text-gray-400 mt-1">
                        {{ $channel['count'] }} month(s)
                    </div>
                </div>
            @endforeach
        </div>
    </div>
    @endif

    <!-- ROI Chart -->
    <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-6 mb-6">
        <h2 class="text-xl font-bold text-gray-800 dark:text-white mb-4">Monthly ROI Trend - {{ $year }}</h2>
        <canvas id="roiChart" height="80"></canvas>
    </div>

    <!-- Monthly Data Table - All Sales View -->
    <div id="monthlyTableAllSales" class="bg-white dark:bg-gray-800 rounded-lg shadow-sm overflow-hidden">
        <div class="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
            <h2 class="text-xl font-bold text-gray-800 dark:text-white">Monthly Breakdown (All Sales)</h2>
        </div>
        <div class="overflow-x-auto">
            <table class="w-full">
                <thead class="bg-gray-50 dark:bg-gray-700">
                    <tr>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase">Month</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase">Marketing Spend</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase">Total Revenue</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase">ROI</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase">Best Channel</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase">Actions</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-gray-200 dark:divide-gray-700">
                    @forelse($marketingDataWithHq as $data)
                        <tr class="hover:bg-gray-50 dark:hover:bg-gray-700">
                            <td class="px-4 py-3 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">
                                {{ $data->month_name }}
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-600 dark:text-gray-400">
                                <div class="flex items-center gap-1">
                                    <span>RM {{ number_format($data->total_marketing_spend, 2) }}</span>
                                    @if($data->campaign_details)
                                        <span class="relative group">
                                            <svg class="w-4 h-4 text-blue-500 cursor-help" fill="currentColor" viewBox="0 0 20 20">
                                                <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"/>
                                            </svg>
                                            <!-- Tooltip -->
                                            <div class="absolute left-0 bottom-full mb-2 hidden group-hover:block w-64 p-3 bg-gray-900 dark:bg-gray-700 text-white text-xs rounded-lg shadow-lg z-50">
                                                <div class="font-semibold mb-1">Campaign Details:</div>
                                                <div class="text-gray-200 dark:text-gray-300">{{ $data->campaign_details }}</div>
                                                <!-- Arrow -->
                                                <div class="absolute top-full left-4 w-0 h-0 border-l-4 border-r-4 border-t-4 border-transparent border-t-gray-900 dark:border-t-gray-700"></div>
                                            </div>
                                        </span>
                                    @endif
                                </div>
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap text-sm font-semibold text-gray-900 dark:text-white">
                                RM {{ number_format($data->actual_revenue, 2) }}
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap">
                                <span class="text-sm font-semibold {{ $data->roi_percentage >= 100 ? 'text-green-600' : 'text-orange-600' }}">
                                    {{ number_format($data->roi_percentage, 1) }}%
                                </span>
                                @php
                                    $dataRatio = $data->total_marketing_spend > 0 ? $data->actual_revenue / $data->total_marketing_spend : 0;
                                @endphp
                                <div class="text-xs text-gray-600 dark:text-gray-400">
                                    {{ number_format($dataRatio, 2) }}x
                                </div>
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-600 dark:text-gray-400">
                                {{ $data->best_channel }}
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap text-sm">
                                <a href="{{ route('forecasting.marketing-strategy.input', ['year' => $data->year, 'month' => $data->month]) }}"
                                   class="text-blue-600 hover:text-blue-800 mr-3">
                                    Edit
                                </a>
                                @if($data->ai_recommendations)
                                    <img src="{{ asset('images/chatbot-aicon.png') }}" alt="AI" class="w-5 h-5 inline-block" title="Has AI suggestions">
                                @endif
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="6" class="px-4 py-8 text-center text-gray-500 dark:text-gray-400">
                                No marketing data found for {{ $year }}. Click "Input Marketing Data" to add.
                            </td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
    </div>

    <!-- Monthly Data Table - HQ Only View -->
    <div id="monthlyTableHqOnly" class="hidden bg-white dark:bg-gray-800 rounded-lg shadow-sm overflow-hidden">
        <div class="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
            <h2 class="text-xl font-bold text-gray-800 dark:text-white">Monthly Breakdown (HQ Sales Only)</h2>
        </div>
        <div class="overflow-x-auto">
            <table class="w-full">
                <thead class="bg-gray-50 dark:bg-gray-700">
                    <tr>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase">Month</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase">Marketing Spend</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase">HQ Revenue</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase">HQ ROI</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase">Best Channel</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase">Actions</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-gray-200 dark:divide-gray-700">
                    @forelse($marketingDataWithHq as $data)
                        <tr class="hover:bg-gray-50 dark:hover:bg-gray-700">
                            <td class="px-4 py-3 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">
                                {{ $data->month_name }}
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-600 dark:text-gray-400">
                                <div class="flex items-center gap-1">
                                    <span>RM {{ number_format($data->total_marketing_spend, 2) }}</span>
                                    @if($data->campaign_details)
                                        <span class="relative group">
                                            <svg class="w-4 h-4 text-purple-500 cursor-help" fill="currentColor" viewBox="0 0 20 20">
                                                <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"/>
                                            </svg>
                                            <!-- Tooltip -->
                                            <div class="absolute left-0 bottom-full mb-2 hidden group-hover:block w-64 p-3 bg-gray-900 dark:bg-gray-700 text-white text-xs rounded-lg shadow-lg z-50">
                                                <div class="font-semibold mb-1">Campaign Details:</div>
                                                <div class="text-gray-200 dark:text-gray-300">{{ $data->campaign_details }}</div>
                                                <!-- Arrow -->
                                                <div class="absolute top-full left-4 w-0 h-0 border-l-4 border-r-4 border-t-4 border-transparent border-t-gray-900 dark:border-t-gray-700"></div>
                                            </div>
                                        </span>
                                    @endif
                                </div>
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap text-sm font-semibold text-purple-900 dark:text-purple-300">
                                RM {{ number_format($data->hq_revenue, 2) }}
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap">
                                <span class="text-sm font-semibold {{ $data->hq_roi_percentage >= 100 ? 'text-green-600' : 'text-orange-600' }}">
                                    {{ number_format($data->hq_roi_percentage, 1) }}%
                                </span>
                                @php
                                    $hqDataRatio = $data->total_marketing_spend > 0 ? $data->hq_revenue / $data->total_marketing_spend : 0;
                                @endphp
                                <div class="text-xs text-purple-600 dark:text-purple-400">
                                    {{ number_format($hqDataRatio, 2) }}x
                                </div>
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-600 dark:text-gray-400">
                                {{ $data->best_channel }}
                            </td>
                            <td class="px-4 py-3 whitespace-nowrap text-sm">
                                <a href="{{ route('forecasting.marketing-strategy.input', ['year' => $data->year, 'month' => $data->month]) }}"
                                   class="text-blue-600 hover:text-blue-800 mr-3">
                                    Edit
                                </a>
                                @if($data->ai_recommendations)
                                    <img src="{{ asset('images/chatbot-aicon.png') }}" alt="AI" class="w-5 h-5 inline-block" title="Has AI suggestions">
                                @endif
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="6" class="px-4 py-8 text-center text-gray-500 dark:text-gray-400">
                                No marketing data found for {{ $year }}. Click "Input Marketing Data" to add.
                            </td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Chart.js -->
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.js"></script>

<script>
// Dashboard Revenue Toggle
let isDashboardHqMode = false;
let roiChart = null; // Store chart instance

function toggleDashboardRevenue() {
    isDashboardHqMode = !isDashboardHqMode;

    // Get elements
    const toggleLabel = document.getElementById('dashboardToggleLabel');
    const toggleButton = document.getElementById('dashboardRevenueToggle');
    const toggleThumb = document.getElementById('dashboardToggleThumb');

    // Summary cards
    const allRevenue = document.getElementById('dashboardAllRevenue');
    const hqRevenue = document.getElementById('dashboardHqRevenue');
    const allRoi = document.getElementById('dashboardAllRoi');
    const hqRoi = document.getElementById('dashboardHqRoi');
    const allProfit = document.getElementById('dashboardAllProfit');
    const hqProfit = document.getElementById('dashboardHqProfit');

    // Best/Worst months
    const bestWorstAll = document.getElementById('bestWorstAllSales');
    const bestWorstHq = document.getElementById('bestWorstHqOnly');

    // Monthly tables
    const tableAll = document.getElementById('monthlyTableAllSales');
    const tableHq = document.getElementById('monthlyTableHqOnly');

    if (isDashboardHqMode) {
        // Switch to HQ Only mode
        toggleLabel.textContent = 'HQ Sales Only';
        toggleButton.classList.remove('bg-gray-300', 'dark:bg-gray-600');
        toggleButton.classList.add('bg-purple-600');
        toggleThumb.classList.remove('translate-x-1');
        toggleThumb.classList.add('translate-x-6');

        // Toggle summary cards
        allRevenue.classList.add('hidden');
        hqRevenue.classList.remove('hidden');
        allRoi.classList.add('hidden');
        hqRoi.classList.remove('hidden');
        allProfit.classList.add('hidden');
        hqProfit.classList.remove('hidden');

        // Toggle best/worst months
        bestWorstAll.classList.add('hidden');
        bestWorstAll.classList.remove('grid');
        bestWorstHq.classList.remove('hidden');
        bestWorstHq.classList.add('grid');

        // Toggle monthly tables
        tableAll.classList.add('hidden');
        tableHq.classList.remove('hidden');

        // Toggle chart datasets
        if (roiChart) {
            roiChart.data.datasets[0].hidden = true;  // Hide All Sales ROI
            roiChart.data.datasets[2].hidden = false; // Show HQ ROI
            roiChart.update();
        }
    } else {
        // Switch to All Sales mode
        toggleLabel.textContent = 'All Sales (HQ + Stokis + Agent)';
        toggleButton.classList.add('bg-gray-300', 'dark:bg-gray-600');
        toggleButton.classList.remove('bg-purple-600');
        toggleThumb.classList.add('translate-x-1');
        toggleThumb.classList.remove('translate-x-6');

        // Toggle summary cards
        allRevenue.classList.remove('hidden');
        hqRevenue.classList.add('hidden');
        allRoi.classList.remove('hidden');
        hqRoi.classList.add('hidden');
        allProfit.classList.remove('hidden');
        hqProfit.classList.add('hidden');

        // Toggle best/worst months
        bestWorstAll.classList.remove('hidden');
        bestWorstAll.classList.add('grid');
        bestWorstHq.classList.add('hidden');
        bestWorstHq.classList.remove('grid');

        // Toggle monthly tables
        tableAll.classList.remove('hidden');
        tableHq.classList.add('hidden');

        // Toggle chart datasets
        if (roiChart) {
            roiChart.data.datasets[0].hidden = false; // Show All Sales ROI
            roiChart.data.datasets[2].hidden = true;  // Hide HQ ROI
            roiChart.update();
        }
    }
}

// ROI Chart
const ctx = document.getElementById('roiChart').getContext('2d');

const chartData = {
    labels: {!! json_encode($marketingDataWithHq->pluck('month_name')) !!},
    datasets: [
        {
            label: 'All Sales ROI %',
            data: {!! json_encode($marketingDataWithHq->pluck('roi_percentage')) !!},
            borderColor: 'rgb(34, 197, 94)',
            backgroundColor: 'rgba(34, 197, 94, 0.1)',
            tension: 0.4,
            fill: true,
            hidden: false
        },
        {
            label: 'Marketing Spend (RM)',
            data: {!! json_encode($marketingDataWithHq->pluck('total_marketing_spend')) !!},
            borderColor: 'rgb(239, 68, 68)',
            backgroundColor: 'rgba(239, 68, 68, 0.1)',
            tension: 0.4,
            yAxisID: 'y1'
        },
        {
            label: 'HQ Only ROI %',
            data: {!! json_encode($marketingDataWithHq->pluck('hq_roi_percentage')) !!},
            borderColor: 'rgb(147, 51, 234)',
            backgroundColor: 'rgba(147, 51, 234, 0.1)',
            tension: 0.4,
            fill: true,
            hidden: true
        }
    ]
};

roiChart = new Chart(ctx, {
    type: 'line',
    data: chartData,
    options: {
        responsive: true,
        maintainAspectRatio: true,
        interaction: {
            mode: 'index',
            intersect: false,
        },
        plugins: {
            legend: {
                position: 'top',
            },
            tooltip: {
                callbacks: {
                    label: function(context) {
                        let label = context.dataset.label || '';
                        if (label) {
                            label += ': ';
                        }
                        if (context.parsed.y !== null) {
                            if (context.datasetIndex === 0) {
                                label += context.parsed.y.toFixed(1) + '%';
                            } else {
                                label += 'RM ' + context.parsed.y.toLocaleString('en-MY', {minimumFractionDigits: 2, maximumFractionDigits: 2});
                            }
                        }
                        return label;
                    }
                }
            }
        },
        scales: {
            y: {
                type: 'linear',
                display: true,
                position: 'left',
                title: {
                    display: true,
                    text: 'ROI %'
                },
                ticks: {
                    callback: function(value) {
                        return value + '%';
                    }
                }
            },
            y1: {
                type: 'linear',
                display: true,
                position: 'right',
                title: {
                    display: true,
                    text: 'Spend (RM)'
                },
                grid: {
                    drawOnChartArea: false,
                },
                ticks: {
                    callback: function(value) {
                        return 'RM ' + value.toLocaleString();
                    }
                }
            }
        }
    }
});
</script>
@endsection
