@extends('layouts.app')

@section('title', 'Update Monthly Expenses')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <!-- Header -->
    <div class="mb-6 flex items-center justify-between">
        <div>
            <h1 class="text-2xl font-bold text-gray-900 dark:text-white">💰 Update Monthly Expenses</h1>
            <p class="text-gray-600 dark:text-gray-400">Track expenses for accurate profit calculations and forecasting</p>
        </div>
        <a href="{{ route('forecasting.index') }}" class="px-4 py-2 bg-gray-600 text-white rounded-md hover:bg-gray-700">
            ← Back to Dashboard
        </a>
    </div>

    <!-- Success Message -->
    @if(session('success'))
    <div class="mb-6 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-4">
        <p class="text-green-800 dark:text-green-200">{{ session('success') }}</p>
    </div>
    @endif

    <!-- Error Messages -->
    @if($errors->any())
    <div class="mb-6 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg p-4">
        <ul class="list-disc list-inside text-red-800 dark:text-red-200">
            @foreach($errors->all() as $error)
                <li>{{ $error }}</li>
            @endforeach
        </ul>
    </div>
    @endif

    <!-- Year Selector -->
    <div class="mb-6">
        <form method="GET" action="{{ route('forecasting.expenses') }}" class="flex items-center gap-4">
            <label class="text-sm font-medium text-gray-700 dark:text-gray-300">Year:</label>
            <select name="year" onchange="this.form.submit()" class="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-white">
                @for($y = 2022; $y <= date('Y'); $y++)
                    <option value="{{ $y }}" {{ $year == $y ? 'selected' : '' }}>{{ $y }}</option>
                @endfor
            </select>
        </form>
    </div>

    <!-- Monthly Expenses Table -->
    <div class="bg-white dark:bg-gray-800 rounded-lg shadow overflow-hidden">
        <div class="p-6 border-b border-gray-200 dark:border-gray-700">
            <h2 class="text-lg font-semibold text-gray-900 dark:text-white">{{ $year }} Monthly Expenses</h2>
            <p class="text-sm text-gray-600 dark:text-gray-400 mt-1">Update expenses at the end of each month for accurate tracking</p>
        </div>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                <thead class="bg-gray-50 dark:bg-gray-900">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Month</th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Revenue</th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Orders</th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Current Expenses</th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Net Profit</th>
                        <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Last Updated</th>
                        <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Action</th>
                    </tr>
                </thead>
                <tbody class="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                    @foreach($currentYearData as $month => $data)
                    <tr>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">
                            {{ date('F', mktime(0, 0, 0, $month, 1)) }}
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-right text-gray-700 dark:text-gray-300">
                            RM {{ number_format($data->revenue ?? 0, 2) }}
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-right text-gray-700 dark:text-gray-300">
                            {{ number_format($data->orders ?? 0) }}
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-right text-red-600 dark:text-red-400">
                            RM {{ number_format($data->expenses ?? 0, 2) }}
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-right font-medium {{ (($data->revenue ?? 0) - ($data->expenses ?? 0)) >= 0 ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400' }}">
                            RM {{ number_format(($data->revenue ?? 0) - ($data->expenses ?? 0), 2) }}
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-center text-gray-700 dark:text-gray-300">
                            @if($data->expense_updated_at)
                                <span class="text-xs">{{ \Carbon\Carbon::parse($data->expense_updated_at)->format('M d, Y') }}</span>
                            @else
                                <span class="text-xs text-gray-400">Not updated</span>
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-center">
                            <button
                                onclick="openExpenseModal({{ $year }}, {{ $month }}, {{ $data->expenses ?? 0 }}, '{{ date('F', mktime(0, 0, 0, $month, 1)) }}')"
                                class="px-3 py-1 bg-blue-600 text-white text-xs rounded hover:bg-blue-700">
                                Update
                            </button>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
                <tfoot class="bg-gray-50 dark:bg-gray-900">
                    <tr>
                        <td class="px-6 py-4 text-sm font-bold text-gray-900 dark:text-white">TOTAL</td>
                        <td class="px-6 py-4 text-sm text-right font-bold text-gray-900 dark:text-white">
                            RM {{ number_format(collect($currentYearData)->sum('revenue'), 2) }}
                        </td>
                        <td class="px-6 py-4 text-sm text-right font-bold text-gray-900 dark:text-white">
                            {{ number_format(collect($currentYearData)->sum('orders')) }}
                        </td>
                        <td class="px-6 py-4 text-sm text-right font-bold text-red-600 dark:text-red-400">
                            RM {{ number_format(collect($currentYearData)->sum('expenses'), 2) }}
                        </td>
                        <td class="px-6 py-4 text-sm text-right font-bold text-green-600 dark:text-green-400">
                            RM {{ number_format(collect($currentYearData)->sum('revenue') - collect($currentYearData)->sum('expenses'), 2) }}
                        </td>
                        <td colspan="2"></td>
                    </tr>
                </tfoot>
            </table>
        </div>
    </div>

    <!-- Info Box -->
    <div class="mt-6 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-6">
        <h3 class="text-sm font-semibold text-blue-900 dark:text-blue-200 mb-2">💡 Tips for Expense Tracking</h3>
        <ul class="text-sm text-blue-800 dark:text-blue-300 space-y-1 list-disc list-inside">
            <li>Update expenses at the end of each month for accurate profit calculations</li>
            <li>Include all business expenses: marketing, operations, salaries, utilities, etc.</li>
            <li>Revenue from November onwards is automatically pulled from your system's order data</li>
            <li>Accurate expense data improves AI forecasting predictions</li>
            <li>Historical data shows last update timestamp for audit purposes</li>
        </ul>
    </div>
</div>

<!-- Modal for Expense Update -->
<div id="expenseModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white dark:bg-gray-800">
        <div class="mt-3">
            <h3 class="text-lg font-medium text-gray-900 dark:text-white mb-4" id="modalTitle">Update Expense</h3>
            <form method="POST" action="{{ route('forecasting.expenses.update') }}">
                @csrf
                <input type="hidden" name="year" id="modalYear">
                <input type="hidden" name="month" id="modalMonth">

                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Expense Amount (RM)
                    </label>
                    <input
                        type="number"
                        name="expenses"
                        id="modalExpense"
                        step="0.01"
                        min="0"
                        class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-white focus:outline-none focus:ring-2 focus:ring-blue-500"
                        required
                        autofocus
                    >
                </div>

                <div class="flex gap-3">
                    <button
                        type="submit"
                        class="flex-1 px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                        Save
                    </button>
                    <button
                        type="button"
                        onclick="closeExpenseModal()"
                        class="flex-1 px-4 py-2 bg-gray-300 dark:bg-gray-600 text-gray-700 dark:text-white rounded-md hover:bg-gray-400 dark:hover:bg-gray-500">
                        Cancel
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function openExpenseModal(year, month, currentExpense, monthName) {
    document.getElementById('modalTitle').textContent = `Update Expense for ${monthName} ${year}`;
    document.getElementById('modalYear').value = year;
    document.getElementById('modalMonth').value = month;
    document.getElementById('modalExpense').value = currentExpense;
    document.getElementById('expenseModal').classList.remove('hidden');
    document.getElementById('modalExpense').focus();
}

function closeExpenseModal() {
    document.getElementById('expenseModal').classList.add('hidden');
}

// Close modal on escape key
document.addEventListener('keydown', function(event) {
    if (event.key === 'Escape') {
        closeExpenseModal();
    }
});

// Close modal on outside click
document.getElementById('expenseModal').addEventListener('click', function(event) {
    if (event.target === this) {
        closeExpenseModal();
    }
});
</script>
@endsection
