@extends('layouts.app')

@section('title', 'Dashboard')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
    <!-- PWA Notification Button -->
    <div class="flex justify-end mb-4">
        <x-pwa-notification-button />
    </div>

    <!-- Compact Period Analytics Section -->
    <div class="mb-4">
        <!-- Compact Period Stats Grid -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-4">
            <!-- Server DateTime Card -->
            <div class="bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 p-5 rounded-xl shadow-md hover:shadow-lg transition-all duration-300">
                <div class="flex items-center justify-between gap-3">
                    <div class="flex-1 min-w-0">
                        <div class="text-sm text-gray-600 dark:text-gray-400 mb-1">Server DateTime</div>
                        <div class="text-lg font-bold text-gray-900 dark:text-white font-mono" id="serverDateTime">
                            {{ now()->format('Y-m-d H:i:s') }}
                        </div>
                        <div class="text-xs text-gray-500 dark:text-gray-400 mt-1">{{ config('app.timezone') }}</div>
                    </div>
                    <div class="flex-shrink-0">
                        <div class="w-12 h-12 bg-blue-100 dark:bg-blue-900/30 rounded-full flex items-center justify-center">
                            <svg class="w-6 h-6 text-blue-600 dark:text-blue-400" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z" clip-rule="evenodd"/>
                            </svg>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Total Orders Card -->
            <div class="bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 p-5 rounded-xl shadow-md hover:shadow-lg transition-all duration-300">
                <div class="flex items-center justify-between gap-3">
                    <div class="flex-1 min-w-0">
                        <div class="text-sm text-gray-600 dark:text-gray-400 mb-1">Total Orders</div>
                        <div class="text-2xl font-bold text-gray-900 dark:text-white truncate">{{ number_format($periodStats['combined_orders']) }}</div>
                        @if(isset($periodStats['historical_orders']) && $periodStats['historical_orders'] > 0)
                            <div class="text-xs text-blue-600 dark:text-blue-400 mt-1">
                                <span class="inline-flex items-center">
                                    <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                    </svg>
                                    Includes {{ number_format($periodStats['historical_orders']) }} historical orders
                                </span>
                            </div>
                        @endif
                        <div class="text-xs text-gray-500 dark:text-gray-400 mt-1 truncate">{{ $periodStats['completed_orders'] + $periodStats['completed_cod_orders'] }} completed • {{ $periodStats['processing_orders'] }} processing</div>
                    </div>
                    <div class="flex-shrink-0">
                        <div class="w-12 h-12 bg-purple-100 dark:bg-purple-900/30 rounded-full flex items-center justify-center">
                            <svg class="w-6 h-6 text-purple-600 dark:text-purple-400" fill="currentColor" viewBox="0 0 20 20">
                                <path d="M3 1a1 1 0 000 2h1.22l.305 1.222a.997.997 0 00.01.042l1.358 5.43-.893.892C3.74 11.846 4.632 14 6.414 14H15a1 1 0 000-2H6.414l1-1H14a1 1 0 00.894-.553l3-6A1 1 0 0017 3H6.28l-.31-1.243A1 1 0 005 1H3zM16 16.5a1.5 1.5 0 11-3 0 1.5 1.5 0 013 0zM6.5 18a1.5 1.5 0 100-3 1.5 1.5 0 000 3z"/>
                            </svg>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Total Revenue Card -->
            <div class="bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 p-5 rounded-xl shadow-md hover:shadow-lg transition-all duration-300">
                <div class="flex items-center justify-between gap-3">
                    <div class="flex-1 min-w-0">
                        <div class="text-sm text-gray-600 dark:text-gray-400 mb-1">Total Revenue</div>
                        @php
                            $primaryCurrency = $periodStats['primary_currency'] ?? 'MYR';
                            $currencySymbol = match(strtoupper($primaryCurrency)) {
                                'MYR' => 'RM',
                                'USD' => '$',
                                'EUR' => '€',
                                'GBP' => '£',
                                'JPY' => '¥',
                                'CNY' => '¥',
                                'SGD' => 'S$',
                                'THB' => '฿',
                                'IDR' => 'Rp',
                                'PHP' => '₱',
                                'VND' => '₫',
                                'KRW' => '₩',
                                'INR' => '₹',
                                'AUD' => 'A$',
                                'CAD' => 'C$',
                                'HKD' => 'HK$',
                                'NZD' => 'NZ$',
                                default => $primaryCurrency . ' ',
                            };
                            // Use combined revenue (completed + processing)
                            $displayRevenue = $periodStats['combined_revenue'] ?? 0;
                            // Determine font size based on number length
                            $revenueLength = strlen(number_format($displayRevenue, 2, '.', ','));
                            $fontClass = match(true) {
                                $revenueLength <= 10 => 'text-2xl',    // RM999.99
                                $revenueLength <= 14 => 'text-xl',     // RM99,999.99
                                $revenueLength <= 17 => 'text-lg',     // RM9,999,999.99
                                default => 'text-base'                  // RM10,000,000.00+
                            };
                        @endphp
                        <div class="{{ $fontClass }} font-bold text-gray-900 dark:text-white break-all leading-tight">
                            {{ $currencySymbol }}{{ number_format($displayRevenue, 2) }}
                        </div>
                        @if(isset($periodStats['historical_revenue']) && $periodStats['historical_revenue'] > 0)
                            <div class="text-xs text-blue-600 dark:text-blue-400 mt-1">
                                <span class="inline-flex items-center">
                                    <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                    </svg>
                                    Includes {{ $currencySymbol }}{{ number_format($periodStats['historical_revenue'], 2) }} historical data
                                </span>
                            </div>
                        @endif
                        <div class="text-xs text-gray-500 dark:text-gray-400 mt-1">
                            All active orders (excl. cancelled)
                        </div>
                    </div>
                    <div class="flex-shrink-0">
                        <div class="w-12 h-12 bg-green-100 dark:bg-green-900/30 rounded-full flex items-center justify-center">
                            <svg class="w-6 h-6 text-green-600 dark:text-green-400" fill="currentColor" viewBox="0 0 20 20">
                                <path d="M8.433 7.418c.155-.103.346-.196.567-.267v1.698a2.305 2.305 0 01-.567-.267C8.07 8.34 8 8.114 8 8c0-.114.07-.34.433-.582zM11 12.849v-1.698c.22.071.412.164.567.267.364.243.433.468.433.582 0 .114-.07.34-.433.582a2.305 2.305 0 01-.567.267z"/>
                                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-13a1 1 0 10-2 0v.092a4.535 4.535 0 00-1.676.662C6.602 6.234 6 7.009 6 8c0 .99.602 1.765 1.324 2.246.48.32 1.054.545 1.676.662v1.941c-.391-.127-.68-.317-.843-.504a1 1 0 10-1.51 1.31c.562.649 1.413 1.076 2.353 1.253V15a1 1 0 102 0v-.092a4.535 4.535 0 001.676-.662C13.398 13.766 14 12.991 14 12c0-.99-.602-1.765-1.324-2.246A4.535 4.535 0 0011 9.092V7.151c.391.127.68.317.843.504a1 1 0 101.511-1.31c-.563-.649-1.413-1.076-2.354-1.253V5z" clip-rule="evenodd"/>
                            </svg>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Avg Order Value Card -->
            <div class="bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 p-5 rounded-xl shadow-md hover:shadow-lg transition-all duration-300">
                <div class="flex items-center justify-between gap-3">
                    <div class="flex-1 min-w-0">
                        <div class="text-sm text-gray-600 dark:text-gray-400 mb-1">Avg Order Value</div>
                        <div class="text-2xl font-bold text-gray-900 dark:text-white truncate">
                    @php
                        $primaryCurrency = $periodStats['primary_currency'] ?? 'MYR';
                        $currencySymbol = match(strtoupper($primaryCurrency)) {
                            'MYR' => 'RM',
                            'USD' => '$',
                            'EUR' => '€',
                            'GBP' => '£',
                            'JPY' => '¥',
                            'CNY' => '¥',
                            'SGD' => 'S$',
                            'THB' => '฿',
                            'IDR' => 'Rp',
                            'PHP' => '₱',
                            'VND' => '₫',
                            'KRW' => '₩',
                            'INR' => '₹',
                            'AUD' => 'A$',
                            'CAD' => 'C$',
                            'HKD' => 'HK$',
                            'NZD' => 'NZ$',
                            default => $primaryCurrency . ' ',
                        };
                        // Use combined average order value (completed + processing)
                        $avgOrderValue = $periodStats['average_order_value'] ?? 0;
                    @endphp
                    {{ $currencySymbol }}{{ number_format($avgOrderValue, 2) }}
                        </div>
                        <div class="text-xs text-gray-500 dark:text-gray-400 mt-1">Per order average</div>
                    </div>
                    <div class="flex-shrink-0">
                        <div class="w-12 h-12 bg-orange-100 dark:bg-orange-900/30 rounded-full flex items-center justify-center">
                            <svg class="w-6 h-6 text-orange-600 dark:text-orange-400" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M12 7a1 1 0 110-2h5a1 1 0 011 1v5a1 1 0 11-2 0V8.414l-4.293 4.293a1 1 0 01-1.414 0L8 10.414l-4.293 4.293a1 1 0 01-1.414-1.414l5-5a1 1 0 011.414 0L11 10.586 14.586 7H12z" clip-rule="evenodd"/>
                            </svg>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Compact Analytics & Filter Grid -->
        <div class="grid grid-cols-1 lg:grid-cols-4 gap-4 mb-4">
            <!-- Confirmed Revenue (Non-COD) -->
            <div class="bg-white dark:bg-gray-800 p-4 rounded-lg shadow">
                <h3 class="text-sm font-semibold mb-3">Confirmed Revenue</h3>
                <div class="space-y-2">
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-2">
                            <div class="w-2 h-2 bg-green-500 rounded-full"></div>
                            <span class="text-xs">Completed</span>
                        </div>
                        <span class="text-xs font-medium">
                            @php
                                $primaryCurrency = $periodStats['primary_currency'] ?? 'MYR';
                                $currencySymbol = match(strtoupper($primaryCurrency)) {
                                    'MYR' => 'RM',
                                    'USD' => '$',
                                    'EUR' => '€',
                                    'GBP' => '£',
                                    'JPY' => '¥',
                                    'CNY' => '¥',
                                    'SGD' => 'S$',
                                    'THB' => '฿',
                                    'IDR' => 'Rp',
                                    'PHP' => '₱',
                                    'VND' => '₫',
                                    'KRW' => '₩',
                                    'INR' => '₹',
                                    'AUD' => 'A$',
                                    'CAD' => 'C$',
                                    'HKD' => 'HK$',
                                    'NZD' => 'NZ$',
                                    default => $primaryCurrency . ' ',
                                };
                            @endphp
                            {{ $currencySymbol }}{{ number_format($periodStats['revenue_breakdown_legacy']['completed'] ?? 0, 0) }}
                        </span>
                    </div>
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-2">
                            <div class="w-2 h-2 bg-yellow-500 rounded-full"></div>
                            <span class="text-xs">Processing</span>
                        </div>
                        <span class="text-xs font-medium">
                            {{ $currencySymbol }}{{ number_format($periodStats['revenue_breakdown_legacy']['processing'] ?? 0, 0) }}
                        </span>
                    </div>
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-2">
                            <div class="w-2 h-2 bg-gray-500 rounded-full"></div>
                            <span class="text-xs">Pending</span>
                        </div>
                        <span class="text-xs font-medium">
                            {{ $currencySymbol }}{{ number_format($periodStats['revenue_breakdown_legacy']['pending'] ?? 0, 0) }}
                        </span>
                    </div>
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-2">
                            <div class="w-2 h-2 bg-orange-500 rounded-full"></div>
                            <span class="text-xs">On Hold</span>
                        </div>
                        <span class="text-xs font-medium">
                            {{ $currencySymbol }}{{ number_format($periodStats['revenue_breakdown_legacy']['on_hold'] ?? 0, 0) }}
                        </span>
                    </div>
                </div>
            </div>

            <!-- COD Revenue (Unconfirmed until completed) -->
            <div class="bg-white dark:bg-gray-800 p-4 rounded-lg shadow">
                <h3 class="text-sm font-semibold mb-3">COD Revenue</h3>
                <div class="space-y-2">
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-2">
                            <div class="w-2 h-2 bg-green-500 rounded-full"></div>
                            <span class="text-xs">Completed COD</span>
                        </div>
                        <span class="text-xs font-medium">
                            {{ $currencySymbol }}{{ number_format($periodStats['revenue_breakdown_legacy']['completed_cod'] ?? 0, 0) }}
                        </span>
                    </div>
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-2">
                            <div class="w-2 h-2 bg-yellow-500 rounded-full"></div>
                            <span class="text-xs">Processing COD</span>
                        </div>
                        <span class="text-xs font-medium">
                            {{ $currencySymbol }}{{ number_format($periodStats['revenue_breakdown_legacy']['processing_cod'] ?? 0, 0) }}
                        </span>
                    </div>
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-2">
                            <div class="w-2 h-2 bg-gray-500 rounded-full"></div>
                            <span class="text-xs">Pending COD</span>
                        </div>
                        <span class="text-xs font-medium">
                            {{ $currencySymbol }}{{ number_format($periodStats['revenue_breakdown_legacy']['pending_cod'] ?? 0, 0) }}
                        </span>
                    </div>
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-2">
                            <div class="w-2 h-2 bg-orange-500 rounded-full"></div>
                            <span class="text-xs">On Hold COD</span>
                        </div>
                        <span class="text-xs font-medium">
                            {{ $currencySymbol }}{{ number_format($periodStats['revenue_breakdown_legacy']['on_hold_cod'] ?? 0, 0) }}
                        </span>
                    </div>
                </div>
            </div>

            <!-- Orders by Status -->
            <div class="bg-white dark:bg-gray-800 p-4 rounded-lg shadow">
                <h3 class="text-sm font-semibold mb-3">Orders by Status</h3>
                <div class="space-y-2">
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-2">
                            <div class="w-2 h-2 bg-green-500 rounded-full"></div>
                            <span class="text-xs">Completed</span>
                        </div>
                        <span class="text-xs font-medium">{{ number_format($periodStats['completed_orders']) }}</span>
                    </div>
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-2">
                            <div class="w-2 h-2 bg-yellow-500 rounded-full"></div>
                            <span class="text-xs">Processing</span>
                        </div>
                        <span class="text-xs font-medium">{{ number_format($periodStats['processing_orders']) }}</span>
                    </div>
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-2">
                            <div class="w-2 h-2 bg-gray-500 rounded-full"></div>
                            <span class="text-xs">Pending</span>
                        </div>
                        <span class="text-xs font-medium">{{ number_format($periodStats['pending_orders']) }}</span>
                    </div>
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-2">
                            <div class="w-2 h-2 bg-red-500 rounded-full"></div>
                            <span class="text-xs">Cancelled</span>
                        </div>
                        <span class="text-xs font-medium">{{ number_format($periodStats['cancelled_orders']) }}</span>
                    </div>
                </div>
            </div>

            <!-- Time Period Filter -->
            <div class="bg-white dark:bg-gray-800 p-4 rounded-lg shadow">
                <form method="GET" action="{{ route('dashboard') }}" class="space-y-3" id="periodForm">
                    <h3 class="text-sm font-semibold">Time Period</h3>
                    <select name="period" id="periodSelect" class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 text-xs">
                        <option value="today" {{ $period == 'today' ? 'selected' : '' }}>Today</option>
                        <option value="yesterday" {{ $period == 'yesterday' ? 'selected' : '' }}>Yesterday</option>
                        <option value="this_week" {{ $period == 'this_week' ? 'selected' : '' }}>This Week</option>
                        <option value="last_week" {{ $period == 'last_week' ? 'selected' : '' }}>Last Week</option>
                        <option value="this_month" {{ $period == 'this_month' ? 'selected' : '' }}>This Month</option>
                        <option value="last_month" {{ $period == 'last_month' ? 'selected' : '' }}>Last Month</option>
                        <option value="this_year" {{ $period == 'this_year' ? 'selected' : '' }}>This Year</option>
                        <option value="last_year" {{ $period == 'last_year' ? 'selected' : '' }}>Last Year</option>
                        <option value="custom" {{ $period == 'custom' ? 'selected' : '' }}>Custom Range</option>
                    </select>

                    <!-- Custom Date Range (hidden by default) -->
                    <div id="customDates" class="space-y-2 {{ $period != 'custom' ? 'hidden' : '' }}">
                        <input type="date" name="start_date" value="{{ $startDate }}" placeholder="Start Date"
                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 text-xs">
                        <input type="date" name="end_date" value="{{ $endDate }}" placeholder="End Date"
                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 text-xs">
                        <button type="submit" class="w-full px-3 py-1 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-xs font-medium">
                            Apply Filter
                        </button>
                    </div>

                    <!-- Period Info -->
                    <div class="text-xs text-gray-600 dark:text-gray-400">
                        <strong>{{ $periodStats['period_label'] }}:</strong>
                        {{ $periodStats['start_date'] }}@if($periodStats['start_date'] != $periodStats['end_date']) - {{ $periodStats['end_date'] }}@endif
                    </div>
                </form>
            </div>
        </div>

        <!-- Compact Revenue Chart -->
        @if(count($periodStats['daily_breakdown']) > 1)
        <div class="bg-white dark:bg-gray-800 p-4 rounded-lg shadow mb-4">
            <h3 class="text-sm font-semibold mb-2">
                Revenue Trend @if(isset($periodStats['chart_title']))<span class="text-xs font-normal text-gray-500 dark:text-gray-400">({{ $periodStats['chart_title'] }})</span>@endif
            </h3>
            <div class="h-32">
                <div class="flex items-end justify-between h-24 w-full px-1 gap-1">
                    @php
                        $maxRevenue = max(array_column($periodStats['daily_breakdown'], 'revenue'));
                        $barCount = count($periodStats['daily_breakdown']);
                        $barWidth = $barCount > 20 ? 'w-2' : ($barCount > 10 ? 'w-3' : 'w-4');
                        $primaryCurrency = $periodStats['primary_currency'] ?? 'MYR';
                        $currencySymbol = match(strtoupper($primaryCurrency)) {
                            'MYR' => 'RM',
                            'USD' => '$',
                            'EUR' => '€',
                            'GBP' => '£',
                            'JPY' => '¥',
                            'CNY' => '¥',
                            'SGD' => 'S$',
                            'THB' => '฿',
                            'IDR' => 'Rp',
                            'PHP' => '₱',
                            'VND' => '₫',
                            'KRW' => '₩',
                            'INR' => '₹',
                            'AUD' => 'A$',
                            'CAD' => 'C$',
                            'HKD' => 'HK$',
                            'NZD' => 'NZ$',
                            default => $primaryCurrency . ' ',
                        };
                    @endphp
                    @foreach($periodStats['daily_breakdown'] as $day)
                    @php
                        $tooltipText = $day['full_date'] . ': ' . $currencySymbol . number_format($day['revenue'], 2) . ' (' . $day['orders'] . ' orders)';
                    @endphp
                    <div class="flex flex-col items-center space-y-1 flex-1 min-w-0"
                         title="{{ $tooltipText }}">
                        <div class="bg-blue-500 rounded-t transition-all duration-300 hover:bg-blue-600 cursor-pointer {{ $barWidth }} mx-auto"
                             style="height: {{ $maxRevenue > 0 ? max(($day['revenue'] / $maxRevenue) * 80, 1) : 1 }}px;">
                        </div>
                        <!-- Hide date numbers on mobile (sm and below) to prevent overlap -->
                        <div class="text-xs text-gray-500 dark:text-gray-400 text-center hidden md:block">{{ $day['day'] }}</div>
                    </div>
                    @endforeach
                </div>
                <div class="flex justify-between items-center mt-2 text-xs text-gray-500 dark:text-gray-400">
                    <div class="hidden md:inline">Hover for details</div>
                    <div class="md:hidden text-center">Tap bars for date details</div>
                    <div class="font-medium">Total:
                        @php
                            $totalRevenue = array_sum(array_column($periodStats['daily_breakdown'], 'revenue'));
                        @endphp
                        {{ $currencySymbol }}{{ number_format($totalRevenue, 2) }}
                    </div>
                </div>
            </div>
        </div>
        @endif
    </div>

    <!-- Advanced Analytics Section -->
    <div class="bg-white dark:bg-gray-800 shadow rounded-lg overflow-hidden p-6 mb-4">
        <div class="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-4 gap-3">
            <div>
                <h2 class="text-lg font-semibold text-gray-900 dark:text-gray-100">Advanced Analytics</h2>
                @if($analyticsSource && $analyticsSource !== 'all')
                    <p class="text-xs text-blue-600 dark:text-blue-400 mt-1">
                        @if(strpos($analyticsSource, 'store_') === 0)
                            @php $store = \App\Models\Store::find(str_replace('store_', '', $analyticsSource)); @endphp
                            📍 Showing data for: <strong>{{ $store->name ?? 'Unknown Store' }}</strong>
                        @elseif(strpos($analyticsSource, 'webhook_') === 0)
                            @php $webhook = \App\Models\WebhookSource::find(str_replace('webhook_', '', $analyticsSource)); @endphp
                            📍 Showing data for: <strong>{{ $webhook->name ?? 'Unknown Webhook' }}</strong>
                        @endif
                    </p>
                @endif
            </div>

            <!-- Analytics Source Filter & Export Button - Right Side -->
            <div class="flex items-center gap-2 ml-auto">
                @if(auth()->user()->isAdmin())
                    <form method="GET" action="{{ route('dashboard') }}" class="flex items-center gap-2">
                        <!-- Preserve existing filters -->
                        <input type="hidden" name="period" value="{{ $period }}">
                        @if($period === 'custom')
                            <input type="hidden" name="start_date" value="{{ $startDate }}">
                            <input type="hidden" name="end_date" value="{{ $endDate }}">
                        @endif

                        <label class="text-xs text-gray-600 dark:text-gray-400 whitespace-nowrap">Filter by:</label>
                        <select name="analytics_source" onchange="this.form.submit()" size="1"
                                class="rounded border-gray-300 dark:border-gray-600 dark:bg-gray-700 text-xs px-2 py-1 scrollable-select">
                            <option value="all" {{ !$analyticsSource || $analyticsSource == 'all' ? 'selected' : '' }}>All Sources</option>
                            <optgroup label="WooCommerce Stores">
                                @foreach(\App\Models\Store::where('is_active', true)->get() as $store)
                                    <option value="store_{{ $store->id }}" {{ $analyticsSource == 'store_'.$store->id ? 'selected' : '' }}>
                                        {{ $store->name }}
                                    </option>
                                @endforeach
                            </optgroup>
                            <optgroup label="Webhook Sources">
                                @foreach(\App\Models\WebhookSource::where('is_active', true)->get() as $webhook)
                                    <option value="webhook_{{ $webhook->id }}" {{ $analyticsSource == 'webhook_'.$webhook->id ? 'selected' : '' }}>
                                        {{ $webhook->name }}
                                    </option>
                                @endforeach
                            </optgroup>
                        </select>
                    </form>
                @endif

                <!-- Export Button -->
                <a href="{{ route('dashboard.export-analytics', array_filter([
                    'period' => $period,
                    'start_date' => $period === 'custom' ? $startDate : null,
                    'end_date' => $period === 'custom' ? $endDate : null,
                    'analytics_source' => $analyticsSource
                ])) }}"
                   class="inline-flex items-center gap-1 px-3 py-1.5 bg-green-600 hover:bg-green-700 text-white rounded text-xs font-medium whitespace-nowrap">
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                    </svg>
                    Export XLSX
                </a>
            </div>
        </div>

        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            <!-- TikTok Shop Revenue Detail (Admin: all data, Manager: only if assigned access) -->
            @php
                $showTikTok = false;
                if (auth()->user()->isAdmin()) {
                    // Admin sees all TikTok data
                    $showTikTok = isset($periodStats['tiktok_orders']);
                } elseif (auth()->user()->isManager() && auth()->user()->can_manage_tiktok_shop) {
                    // Manager only sees TikTok if they have permission
                    $showTikTok = isset($periodStats['tiktok_orders']);
                }
            @endphp
            @if($showTikTok)
            <div class="bg-gradient-to-br from-pink-50 to-purple-50 dark:from-pink-900/20 dark:to-purple-900/20 p-4 rounded-lg border border-pink-200 dark:border-pink-800">
                <div class="flex items-center justify-between mb-3">
                    <h3 class="text-sm font-semibold flex items-center text-pink-900 dark:text-pink-200">
                        <span class="text-lg mr-2">📱</span> TikTok Shop
                    </h3>
                    <a href="{{ route('tiktok-shop.index') }}" class="text-xs text-indigo-600 dark:text-indigo-400 hover:underline">View →</a>
                </div>

                @if($periodStats['tiktok_orders'] > 0)
                    <!-- Total Orders -->
                    <div class="mb-3">
                        <div class="text-2xl font-bold text-pink-900 dark:text-pink-100">
                            {{ number_format($periodStats['tiktok_orders']) }}
                        </div>
                        <div class="text-xs text-pink-600 dark:text-pink-400">orders settled</div>
                    </div>

                    <!-- Financial Breakdown -->
                    <div class="space-y-2 mb-3">
                        <div class="flex justify-between text-xs">
                            <span class="text-pink-700 dark:text-pink-300">Bank Transfer:</span>
                            <span class="font-semibold text-pink-900 dark:text-pink-100">{{ $currencySymbol }}{{ number_format($periodStats['tiktok_revenue'], 2) }}</span>
                        </div>
                    </div>

                    <!-- Period Info -->
                    <div class="pt-3 border-t border-pink-200 dark:border-pink-700">
                        <div class="text-xs text-pink-600 dark:text-pink-400">
                            <div>After refunds & TikTok fees</div>
                            <div class="mt-1 text-pink-500 dark:text-pink-500">Settlement period: {{ $periodStats['start_date'] }} - {{ $periodStats['end_date'] }}</div>
                        </div>
                    </div>
                @else
                    <p class="text-xs text-pink-600 dark:text-pink-400 text-center py-4">
                        No TikTok settlements in this period
                    </p>
                @endif
            </div>
            @endif

            <!-- Shipping Cost Estimation -->
            <div class="bg-gradient-to-br from-indigo-50 to-blue-50 dark:from-indigo-900/20 dark:to-blue-900/20 p-4 rounded-lg border border-indigo-200 dark:border-indigo-800">
                <h3 class="text-sm font-semibold mb-3 flex items-center text-indigo-900 dark:text-indigo-200">
                    <span class="text-lg mr-2">📦</span> Estimated Shipping Costs
                </h3>

                @if($shippingCosts['total_orders'] > 0)
                    <!-- Total Cost -->
                    <div class="mb-4">
                        <div class="text-2xl font-bold text-indigo-900 dark:text-indigo-100">
                            RM {{ number_format($shippingCosts['total_cost'], 2) }}
                        </div>
                        <div class="text-xs text-indigo-600 dark:text-indigo-400">
                            {{ $shippingCosts['total_orders'] }} orders shipped
                        </div>
                    </div>

                    <!-- Stats -->
                    <div class="space-y-2 mb-3">
                        <div class="flex justify-between text-xs">
                            <span class="text-indigo-700 dark:text-indigo-300">Avg per Order:</span>
                            <span class="font-semibold text-indigo-900 dark:text-indigo-100">RM {{ number_format($shippingCosts['avg_per_order'], 2) }}</span>
                        </div>
                        @if(isset($periodStats['combined_revenue']) && $periodStats['combined_revenue'] > 0)
                        <div class="flex justify-between text-xs">
                            <span class="text-indigo-700 dark:text-indigo-300">% of Revenue:</span>
                            <span class="font-semibold text-indigo-900 dark:text-indigo-100">
                                {{ number_format(($shippingCosts['total_cost'] / $periodStats['combined_revenue']) * 100, 1) }}%
                            </span>
                        </div>
                        @endif
                    </div>

                    <!-- Breakdown by Status -->
                    <div class="border-t border-indigo-200 dark:border-indigo-700 pt-3 space-y-1.5">
                        <div class="text-xs font-semibold text-indigo-800 dark:text-indigo-300 mb-2">Breakdown:</div>
                        @foreach(['completed' => '✅', 'printed' => '🖨️', 'approval' => '📋', 'processing' => '⚙️'] as $status => $icon)
                            @if($shippingCosts['breakdown'][$status]['count'] > 0)
                            <div class="flex justify-between text-xs">
                                <span class="text-indigo-600 dark:text-indigo-400">
                                    {{ $icon }} {{ ucfirst($status) }}:
                                </span>
                                <span class="font-medium text-indigo-900 dark:text-indigo-100">
                                    RM {{ number_format($shippingCosts['breakdown'][$status]['cost'], 2) }}
                                    <span class="text-indigo-500">({{ $shippingCosts['breakdown'][$status]['count'] }})</span>
                                </span>
                            </div>
                            @endif
                        @endforeach
                    </div>

                    <!-- Rates Info -->
                    <div class="mt-3 pt-3 border-t border-indigo-200 dark:border-indigo-700">
                        <div class="text-xs text-indigo-600 dark:text-indigo-400">
                            <div class="font-semibold mb-1">Current Rates:</div>
                            <div>• Online: RM {{ number_format($shippingCosts['rates']['online_payment'], 2) }}</div>
                            <div>• COD East: RM {{ number_format($shippingCosts['rates']['cod_east_malaysia'], 2) }}</div>
                            <div>• COD Peninsular: RM {{ number_format($shippingCosts['rates']['cod_peninsular'], 2) }}</div>
                        </div>
                    </div>
                @else
                    <p class="text-xs text-indigo-600 dark:text-indigo-400 text-center py-4">
                        No orders with tracking numbers in this period
                    </p>
                @endif
            </div>

            <!-- ROI Calculator -->
            <div class="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                <h3 class="text-sm font-semibold mb-3 flex items-center">
                    <span class="text-lg mr-2">💰</span> ROI Calculator
                </h3>

                <!-- Mode Toggle -->
                <div class="mb-3 flex gap-2">
                    <button type="button" onclick="toggleROIMode('basic')" id="btnBasicMode" class="flex-1 px-3 py-2 text-xs font-semibold rounded bg-indigo-500 text-white">
                        Basic
                    </button>
                    <button type="button" onclick="toggleROIMode('detailed')" id="btnDetailedMode" class="flex-1 px-3 py-2 text-xs font-semibold rounded bg-gray-300 dark:bg-gray-600 text-gray-700 dark:text-gray-300">
                        Detailed
                    </button>
                </div>

                <!-- Total Sales Display -->
                <div class="mb-3 p-2 bg-blue-50 dark:bg-blue-900/20 rounded border border-blue-200 dark:border-blue-800">
                    <div class="text-xs text-gray-600 dark:text-gray-400">Total Sales ({{ $periodStats['period_label'] }})</div>
                    <div class="text-lg font-bold text-blue-900 dark:text-blue-200">
                        {{ $periodStats['primary_currency'] }} {{ number_format($periodStats['combined_revenue'], 2) }}
                    </div>
                </div>

                <!-- Basic Mode -->
                <div id="roiBasicMode" class="space-y-2">
                    <div>
                        <label class="text-xs text-gray-600 dark:text-gray-400">Total Spending ({{ $periodStats['primary_currency'] }})</label>
                        <input type="number" id="basicSpending" step="0.01" min="0" class="w-full mt-1 px-2 py-1.5 text-xs border border-gray-300 dark:border-gray-600 rounded bg-white dark:bg-gray-800 text-gray-900 dark:text-white" placeholder="e.g., 30000" oninput="calculateBasicROI()">
                    </div>
                    <div id="basicResult" class="hidden p-3 bg-white dark:bg-gray-600 rounded">
                        <div class="text-xs text-gray-600 dark:text-gray-400 mb-1">ROI</div>
                        <div class="text-2xl font-bold" id="basicROIValue">0%</div>
                        <div class="text-xs text-gray-500 dark:text-gray-400 mt-2" id="basicProfit">Profit: {{ $periodStats['primary_currency'] }} 0.00</div>
                    </div>
                </div>

                <!-- Detailed Mode -->
                <div id="roiDetailedMode" class="space-y-2 hidden">
                    <div>
                        <label class="text-xs text-gray-600 dark:text-gray-400">Product Cost</label>
                        <div class="flex gap-1 mt-1 items-center">
                            <input type="number" id="productCostPercent" step="0.01" min="0" max="100" class="flex-1 min-w-0 px-2 py-1.5 text-xs border border-gray-300 dark:border-gray-600 rounded bg-white dark:bg-gray-800 text-gray-900 dark:text-white" placeholder="%" oninput="calculateDetailedROI()">
                            <span class="text-xs text-gray-400 dark:text-gray-500">or</span>
                            <input type="number" id="productCostValue" step="0.01" min="0" class="flex-1 min-w-0 px-2 py-1.5 text-xs border border-gray-300 dark:border-gray-600 rounded bg-white dark:bg-gray-800 text-gray-900 dark:text-white" placeholder="RM" oninput="calculateDetailedROI()">
                        </div>
                    </div>
                    <div>
                        <label class="text-xs text-gray-600 dark:text-gray-400">Shipping Cost</label>
                        <div class="flex gap-1 mt-1 items-center">
                            <input type="number" id="shippingCostPercent" step="0.01" min="0" max="100" class="flex-1 min-w-0 px-2 py-1.5 text-xs border border-gray-300 dark:border-gray-600 rounded bg-white dark:bg-gray-800 text-gray-900 dark:text-white" placeholder="%" oninput="calculateDetailedROI()">
                            <span class="text-xs text-gray-400 dark:text-gray-500">or</span>
                            <input type="number" id="shippingCostValue" step="0.01" min="0" class="flex-1 min-w-0 px-2 py-1.5 text-xs border border-gray-300 dark:border-gray-600 rounded bg-white dark:bg-gray-800 text-gray-900 dark:text-white" placeholder="RM" oninput="calculateDetailedROI()">
                        </div>
                    </div>
                    <div>
                        <label class="text-xs text-gray-600 dark:text-gray-400">Ads Spending ({{ $periodStats['primary_currency'] }})</label>
                        <input type="number" id="adsSpending" step="0.01" min="0" class="w-full mt-1 px-2 py-1.5 text-xs border border-gray-300 dark:border-gray-600 rounded bg-white dark:bg-gray-800 text-gray-900 dark:text-white" placeholder="e.g., 5000" oninput="calculateDetailedROI()">
                    </div>
                    <div id="detailedResult" class="hidden p-3 bg-white dark:bg-gray-600 rounded">
                        <div class="text-xs text-gray-600 dark:text-gray-400 mb-1">Detailed ROI</div>
                        <div class="text-2xl font-bold" id="detailedROIValue">0%</div>
                        <div class="text-xs text-gray-500 dark:text-gray-400 mt-2 space-y-1">
                            <div id="detailedBreakdown"></div>
                            <div class="font-semibold" id="detailedProfit">Profit: {{ $periodStats['primary_currency'] }} 0.00</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Delivery by State -->
            <div class="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                <h3 class="text-sm font-semibold mb-3 flex items-center">
                    <span class="text-lg mr-2">📊</span> Delivery by State
                </h3>
                @if(count($advancedAnalytics['delivery_by_state']['data']) > 0)
                    <div class="space-y-2 max-h-64 overflow-y-auto">
                        @foreach(array_slice($advancedAnalytics['delivery_by_state']['data'], 0, 15) as $state)
                        <div class="flex items-center justify-between">
                            <div class="flex-1 min-w-0">
                                <div class="text-xs font-medium truncate">{{ $state['state'] }}</div>
                                <div class="w-full bg-gray-200 dark:bg-gray-600 rounded-full h-1.5 mt-1">
                                    <div class="bg-blue-600 h-1.5 rounded-full" style="width: {{ $state['percentage'] }}%"></div>
                                </div>
                            </div>
                            <div class="ml-3 text-xs font-semibold whitespace-nowrap">
                                {{ $state['percentage'] }}% <span class="text-gray-500">({{ $state['count'] }})</span>
                            </div>
                        </div>
                        @endforeach
                    </div>
                    <div class="mt-3 text-xs text-gray-500 dark:text-gray-400 border-t pt-2">
                        Total Deliveries: {{ number_format($advancedAnalytics['delivery_by_state']['total']) }}
                    </div>
                @else
                    <p class="text-xs text-gray-500 dark:text-gray-400 text-center py-4">No delivery data available</p>
                @endif
            </div>

            <!-- Top Products -->
            <div class="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                <h3 class="text-sm font-semibold mb-3 flex items-center">
                    <span class="text-lg mr-2">🏆</span> Top Products
                </h3>
                @if(count($advancedAnalytics['top_products']) > 0)
                    <div class="space-y-2 max-h-64 overflow-y-auto">
                        @foreach(array_slice($advancedAnalytics['top_products'], 0, 20) as $index => $product)
                        <div class="flex items-start justify-between p-2 bg-white dark:bg-gray-600 rounded">
                            <div class="flex-1 min-w-0">
                                <div class="flex items-center gap-1">
                                    <span class="text-xs font-bold text-gray-500 dark:text-gray-400">#{{ $index + 1 }}</span>
                                    <span class="text-xs font-semibold text-blue-600 dark:text-blue-400">{{ $product['sku'] }}</span>
                                </div>
                                <div class="text-xs text-gray-600 dark:text-gray-300 truncate">{{ Str::limit($product['name'], 30) }}</div>
                                <div class="text-xs text-gray-500 dark:text-gray-400">{{ $product['orders'] }} orders</div>
                            </div>
                            <div class="ml-2 text-xs font-bold text-green-600 dark:text-green-400 whitespace-nowrap">
                                {{ number_format($product['units']) }} units
                            </div>
                        </div>
                        @endforeach
                    </div>
                @else
                    <p class="text-xs text-gray-500 dark:text-gray-400 text-center py-4">No product data available</p>
                @endif
            </div>

            <!-- Units Out (by SKU) -->
            <div class="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                <h3 class="text-sm font-semibold mb-3 flex items-center">
                    <span class="text-lg mr-2">📦</span> Units Out (by SKU)
                </h3>
                @if(count($advancedAnalytics['units_out']['data']) > 0)
                    <div class="space-y-1.5 max-h-64 overflow-y-auto">
                        @foreach(array_slice($advancedAnalytics['units_out']['data'], 0, 25) as $sku)
                        <div class="flex items-center justify-between p-1.5 bg-white dark:bg-gray-600 rounded">
                            <span class="text-xs font-mono font-semibold text-blue-600 dark:text-blue-400">{{ $sku['sku'] }}</span>
                            <span class="text-xs font-bold text-gray-900 dark:text-white">{{ number_format($sku['units']) }}</span>
                        </div>
                        @endforeach
                    </div>
                    <div class="mt-3 text-xs text-gray-500 dark:text-gray-400 border-t pt-2">
                        Total Units: {{ number_format($advancedAnalytics['units_out']['total_units']) }}
                    </div>
                @else
                    <p class="text-xs text-gray-500 dark:text-gray-400 text-center py-4">No units data available</p>
                @endif
            </div>

            <!-- Repeat Customers -->
            <div class="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                <h3 class="text-sm font-semibold mb-3 flex items-center">
                    <span class="text-lg mr-2">👥</span> Repeat Customers
                </h3>
                @if(count($advancedAnalytics['repeat_customers']['data']) > 0)
                    <div class="mb-3 p-2 bg-blue-100 dark:bg-blue-900 rounded">
                        <div class="text-xs text-gray-700 dark:text-gray-300">
                            <strong>{{ $advancedAnalytics['repeat_customers']['total_repeat_customers'] }}</strong> repeat customers
                            (<strong>{{ $advancedAnalytics['repeat_customers']['repeat_rate'] }}%</strong> repeat rate)
                        </div>
                    </div>
                    <div class="space-y-2 max-h-48 overflow-y-auto">
                        @foreach(array_slice($advancedAnalytics['repeat_customers']['data'], 0, 30) as $customer)
                        <div class="p-2 bg-white dark:bg-gray-600 rounded">
                            <div class="flex justify-between items-start">
                                <div class="flex-1 min-w-0">
                                    <div class="text-xs font-semibold truncate">{{ $customer['name'] ?: 'Unknown' }}</div>
                                    <div class="text-xs text-gray-500 dark:text-gray-400 truncate">📱 {{ $customer['phone'] ?: $customer['email'] }}</div>
                                </div>
                                <div class="ml-2 text-right">
                                    <div class="text-xs font-bold text-blue-600 dark:text-blue-400">{{ $customer['order_count'] }} orders</div>
                                    <div class="text-xs text-gray-500 dark:text-gray-400">
                                        @php
                                            $primaryCurrency = $periodStats['primary_currency'] ?? 'MYR';
                                            $currencySymbol = match(strtoupper($primaryCurrency)) {
                                                'MYR' => 'RM',
                                                'USD' => '$',
                                                default => $primaryCurrency . ' ',
                                            };
                                        @endphp
                                        {{ $currencySymbol }}{{ number_format($customer['total_spent'], 0) }}
                                    </div>
                                </div>
                            </div>
                        </div>
                        @endforeach
                    </div>
                @else
                    <p class="text-xs text-gray-500 dark:text-gray-400 text-center py-4">No repeat customers in this period</p>
                @endif
            </div>

            <!-- Sales Insights -->
            <div class="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                <h3 class="text-sm font-semibold mb-3 flex items-center">
                    <span class="text-lg mr-2">📈</span> Sales Insights
                </h3>
                <div class="space-y-3">
                    <!-- Top State -->
                    @if(count($advancedAnalytics['delivery_by_state']['data']) > 0)
                    <div class="p-2 bg-white dark:bg-gray-600 rounded">
                        <div class="text-xs text-gray-500 dark:text-gray-400 mb-1">Top Delivery State</div>
                        <div class="font-semibold text-sm">{{ $advancedAnalytics['delivery_by_state']['data'][0]['state'] }}</div>
                        <div class="text-xs text-blue-600 dark:text-blue-400">
                            {{ $advancedAnalytics['delivery_by_state']['data'][0]['percentage'] }}% of all deliveries
                        </div>
                    </div>
                    @endif

                    <!-- Best Product -->
                    @if(count($advancedAnalytics['top_products']) > 0)
                    <div class="p-2 bg-white dark:bg-gray-600 rounded">
                        <div class="text-xs text-gray-500 dark:text-gray-400 mb-1">Best Seller</div>
                        <div class="font-semibold text-sm">{{ $advancedAnalytics['top_products'][0]['sku'] }}</div>
                        <div class="text-xs text-green-600 dark:text-green-400">
                            {{ number_format($advancedAnalytics['top_products'][0]['units']) }} units sold
                        </div>
                    </div>
                    @endif

                    <!-- Customer Loyalty -->
                    <div class="p-2 bg-white dark:bg-gray-600 rounded">
                        <div class="text-xs text-gray-500 dark:text-gray-400 mb-1">Customer Loyalty</div>
                        <div class="font-semibold text-sm">{{ $advancedAnalytics['repeat_customers']['repeat_rate'] }}% Repeat Rate</div>
                        <div class="text-xs text-purple-600 dark:text-purple-400">
                            {{ $advancedAnalytics['repeat_customers']['total_repeat_customers'] }} of {{ $advancedAnalytics['repeat_customers']['total_customers'] }} customers
                        </div>
                    </div>
                </div>
            </div>

            <!-- Focus Areas -->
            <div class="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                <h3 class="text-sm font-semibold mb-3 flex items-center">
                    <span class="text-lg mr-2">🎯</span> Focus Areas
                </h3>
                <div class="space-y-2">
                    @php
                        $focusAreas = [];

                        // Low repeat rate
                        if($advancedAnalytics['repeat_customers']['repeat_rate'] < 20) {
                            $focusAreas[] = ['icon' => '⚠️', 'title' => 'Low Repeat Rate', 'desc' => 'Focus on customer retention strategies'];
                        }

                        // High concentration in one state
                        if(count($advancedAnalytics['delivery_by_state']['data']) > 0 && $advancedAnalytics['delivery_by_state']['data'][0]['percentage'] > 40) {
                            $focusAreas[] = ['icon' => '🗺️', 'title' => 'Geographic Concentration', 'desc' => 'Consider expanding to other states'];
                        }

                        // Top products
                        if(count($advancedAnalytics['top_products']) > 0) {
                            $focusAreas[] = ['icon' => '✅', 'title' => 'Stock Top Sellers', 'desc' => 'Ensure inventory for SKU: ' . $advancedAnalytics['top_products'][0]['sku']];
                        }

                        // Good repeat rate
                        if($advancedAnalytics['repeat_customers']['repeat_rate'] >= 30) {
                            $focusAreas[] = ['icon' => '🎉', 'title' => 'Good Customer Loyalty', 'desc' => 'Maintain quality and service levels'];
                        }
                    @endphp

                    @if(count($focusAreas) > 0)
                        @foreach($focusAreas as $area)
                        <div class="p-2 bg-white dark:bg-gray-600 rounded">
                            <div class="flex items-start gap-2">
                                <span class="text-sm">{{ $area['icon'] }}</span>
                                <div>
                                    <div class="text-xs font-semibold">{{ $area['title'] }}</div>
                                    <div class="text-xs text-gray-600 dark:text-gray-300">{{ $area['desc'] }}</div>
                                </div>
                            </div>
                        </div>
                        @endforeach
                    @else
                        <p class="text-xs text-gray-500 dark:text-gray-400 text-center py-4">Keep monitoring your metrics</p>
                    @endif
                </div>
            </div>
        </div>
    </div>

    <!-- Top Sales Performance Section (Admin & Manager Only) -->
    @if($topSales !== null)
    <div class="mb-4">
        <div class="bg-white dark:bg-gray-800 shadow rounded-lg overflow-hidden">
            <div class="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                <div class="flex items-center justify-between">
                    <div class="flex items-center gap-2">
                        <svg class="w-6 h-6 text-yellow-500" fill="currentColor" viewBox="0 0 20 20">
                            <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z"/>
                        </svg>
                        <h2 class="text-xl font-bold text-gray-900 dark:text-white">Top Sales Performance</h2>
                    </div>
                    <div class="text-sm text-gray-500 dark:text-gray-400">
                        {{ $dateRange['label'] }}
                    </div>
                </div>
            </div>

            @if($topSales->count() > 0)
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                    <thead class="bg-gray-50 dark:bg-gray-700">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Rank</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Seller Name</th>
                            <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Total Orders</th>
                            <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Total Revenue</th>
                            <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Avg Order Value</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                        @foreach($topSales as $index => $seller)
                        <tr class="hover:bg-gray-50 dark:hover:bg-gray-700">
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="flex items-center">
                                    @if($index === 0)
                                        <span class="inline-flex items-center justify-center w-8 h-8 rounded-full bg-yellow-100 dark:bg-yellow-900/30 text-yellow-600 dark:text-yellow-400 font-bold">🥇</span>
                                    @elseif($index === 1)
                                        <span class="inline-flex items-center justify-center w-8 h-8 rounded-full bg-gray-100 dark:bg-gray-700 text-gray-600 dark:text-gray-400 font-bold">🥈</span>
                                    @elseif($index === 2)
                                        <span class="inline-flex items-center justify-center w-8 h-8 rounded-full bg-orange-100 dark:bg-orange-900/30 text-orange-600 dark:text-orange-400 font-bold">🥉</span>
                                    @else
                                        <span class="inline-flex items-center justify-center w-8 h-8 rounded-full bg-gray-100 dark:bg-gray-700 text-gray-600 dark:text-gray-300 font-semibold text-sm">{{ $index + 1 }}</span>
                                    @endif
                                </div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="flex items-center">
                                    <div>
                                        <div class="text-sm font-medium text-gray-900 dark:text-white">
                                            {{ $seller->name }}
                                            @if(isset($seller->has_historical_data) && $seller->has_historical_data)
                                                <span class="ml-1 inline-flex items-center text-xs text-blue-600 dark:text-blue-400" title="Includes historical data">
                                                    <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                                    </svg>
                                                </span>
                                            @endif
                                        </div>
                                        <div class="text-xs text-gray-500 dark:text-gray-400">{{ $seller->email }}</div>
                                    </div>
                                </div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-right">
                                <div class="text-sm font-semibold text-gray-900 dark:text-white">{{ number_format($seller->total_orders) }}</div>
                            </td>
                            <td class="px-6 py-4 text-right">
                                <div class="text-sm font-bold text-green-600 dark:text-green-400 break-all">RM {{ number_format($seller->total_revenue, 2) }}</div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-right">
                                <div class="text-sm text-gray-900 dark:text-white">RM {{ number_format($seller->avg_order_value, 2) }}</div>
                            </td>
                        </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>
            @else
            <div class="px-6 py-12 text-center">
                <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"/>
                </svg>
                <p class="mt-2 text-sm text-gray-500 dark:text-gray-400">No sales data available for the selected period.</p>
            </div>
            @endif
        </div>
    </div>
    @endif

    <!-- Recent Orders Section - Hidden for now -->
    <div class="bg-white dark:bg-gray-800 shadow rounded-lg overflow-hidden hidden" style="min-height: 50vh;">
        <!-- Orders Header -->
        <div class="px-4 py-3 border-b border-gray-200 dark:border-gray-700 flex justify-between items-center">
            <h2 class="text-lg font-semibold text-gray-900 dark:text-gray-100">Recent Orders</h2>
            <form method="GET" action="{{ route('dashboard') }}" class="flex items-center space-x-2">
                <label class="text-xs text-gray-600 dark:text-gray-400">Show:</label>
                <select name="per_page" onchange="this.form.submit()"
                        class="rounded border-gray-300 dark:border-gray-600 dark:bg-gray-700 text-xs px-2 py-1">
                    <option value="25" {{ $perPage == 25 ? 'selected' : '' }}>25</option>
                    <option value="50" {{ $perPage == 50 ? 'selected' : '' }}>50</option>
                    <option value="100" {{ $perPage == 100 ? 'selected' : '' }}>100</option>
                    <option value="200" {{ $perPage == 200 ? 'selected' : '' }}>200</option>
                </select>
            </form>
        </div>

        <!-- Desktop Table View (hidden on mobile) -->
        <div class="hidden lg:block overflow-x-auto" style="max-height: 60vh;">
            <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                <thead class="bg-gray-50 dark:bg-gray-700 sticky top-0">
                    <tr>
                        <th class="px-3 py-2 text-left text-xs font-medium uppercase tracking-wider text-gray-500 dark:text-gray-400">ID</th>
                        <th class="px-3 py-2 text-left text-xs font-medium uppercase tracking-wider text-gray-500 dark:text-gray-400">Order / Store</th>
                        <th class="px-3 py-2 text-left text-xs font-medium uppercase tracking-wider text-gray-500 dark:text-gray-400">Customer</th>
                        <th class="px-3 py-2 text-left text-xs font-medium uppercase tracking-wider text-gray-500 dark:text-gray-400">Status</th>
                        <th class="px-3 py-2 text-left text-xs font-medium uppercase tracking-wider text-gray-500 dark:text-gray-400">Payment</th>
                        <th class="px-3 py-2 text-left text-xs font-medium uppercase tracking-wider text-gray-500 dark:text-gray-400">Total</th>
                        <th class="px-3 py-2 text-left text-xs font-medium uppercase tracking-wider text-gray-500 dark:text-gray-400">Date</th>
                        <th class="px-3 py-2 text-left text-xs font-medium uppercase tracking-wider text-gray-500 dark:text-gray-400">Actions</th>
                    </tr>
                </thead>
                <tbody class="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                    @forelse($orders as $order)
                    <tr class="hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors">
                        <td class="px-3 py-3 text-sm">
                            <div class="font-medium text-blue-600 dark:text-blue-400 truncate max-w-24" title="{{ $order->global_order_id }}">
                                {{ $order->global_order_id }}
                            </div>
                        </td>
                        <td class="px-3 py-3 text-sm">
                            <div class="font-medium text-gray-900 dark:text-gray-100">#{{ $order->order_number }}</div>
                            @if($order->woo_order_id)
                                <div class="text-gray-500 dark:text-gray-400 text-xs">WOO: {{ $order->woo_order_id }}</div>
                            @elseif($order->webhookSource)
                                <div class="text-blue-500 dark:text-blue-400 text-xs">Webhook</div>
                            @endif
                            @if($order->store)
                                <div class="text-gray-600 dark:text-gray-300 text-xs font-medium mt-1" title="{{ $order->store->name }}">{{ Str::limit($order->store->name, 20) }}</div>
                            @elseif($order->webhookSource)
                                <div class="text-blue-600 dark:text-blue-300 text-xs font-medium mt-1" title="{{ $order->webhookSource->name }}">{{ Str::limit($order->webhookSource->name, 20) }}</div>
                            @endif
                        </td>
                        <td class="px-3 py-3 text-sm">
                            <div class="truncate max-w-32 text-gray-900 dark:text-gray-100" title="{{ ($order->billing['first_name'] ?? '') . ' ' . ($order->billing['last_name'] ?? '') }}">
                                {{ $order->billing['first_name'] ?? '' }} {{ $order->billing['last_name'] ?? '' }}
                            </div>
                            <div class="text-gray-500 dark:text-gray-400 text-xs truncate max-w-32" title="{{ $order->billing['email'] ?? '' }}">
                                {{ $order->billing['email'] ?? '' }}
                            </div>
                        </td>
                        <td class="px-3 py-3">
                            <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full
                                bg-{{ $order->status_color }}-100 text-{{ $order->status_color }}-800
                                dark:bg-{{ $order->status_color }}-900 dark:text-{{ $order->status_color }}-200">
                                {{ ucfirst($order->status) }}
                            </span>
                        </td>
                        <td class="px-3 py-3">
                            <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full
                                bg-{{ $order->payment_color }}-100 text-{{ $order->payment_color }}-800
                                dark:bg-{{ $order->payment_color }}-900 dark:text-{{ $order->payment_color }}-200">
                                {{ Str::limit($order->payment_method_title, 8) }}
                            </span>
                            @if($order->isLocalPickup())
                                <span class="ml-1 px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200">
                                    Pickup
                                </span>
                            @endif
                        </td>
                        <td class="px-3 py-3 text-sm font-medium text-gray-900 dark:text-gray-100 whitespace-nowrap">
                            {{ $order->formatCurrency() }}
                        </td>
                        <td class="px-3 py-3 text-sm text-gray-500 dark:text-gray-400 whitespace-nowrap">
                            {{ $order->date_created->format('M d, H:i') }}
                        </td>
                        <td class="px-3 py-3 text-sm whitespace-nowrap">
                            <a href="{{ route('orders.edit', $order) }}"
                               class="text-indigo-600 dark:text-indigo-400 hover:text-indigo-900 font-medium">
                                Edit
                            </a>
                        </td>
                    </tr>
                    @empty
                    <tr>
                        <td colspan="8" class="px-6 py-8 text-center text-gray-500 dark:text-gray-400">
                            No orders found.
                        </td>
                    </tr>
                    @endforelse
                </tbody>
            </table>
        </div>

        <!-- Mobile Card View (visible on mobile and tablet) -->
        <div class="lg:hidden" style="max-height: 60vh; overflow-y-auto;">
            @forelse($orders as $order)
            <div class="border-b border-gray-200 dark:border-gray-700 p-4 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors">
                <div class="flex justify-between items-start mb-3">
                    <div class="flex-1 min-w-0">
                        <div class="flex items-center space-x-2 mb-1">
                            <div class="font-medium text-blue-600 dark:text-blue-400 text-sm">
                                {{ $order->global_order_id }}
                            </div>
                            <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full
                                bg-{{ $order->status_color }}-100 text-{{ $order->status_color }}-800
                                dark:bg-{{ $order->status_color }}-900 dark:text-{{ $order->status_color }}-200">
                                {{ ucfirst($order->status) }}
                            </span>
                        </div>
                        <div class="text-sm text-gray-900 dark:text-gray-100 font-medium">
                            Order #{{ $order->order_number }}
                        </div>
                        <div class="text-xs text-gray-500 dark:text-gray-400">
                            @if($order->store)
                                {{ $order->store->name }} • WOO ID: {{ $order->woo_order_id }}
                            @elseif($order->webhookSource)
                                {{ $order->webhookSource->name }} • Webhook Order
                            @else
                                Unknown Source
                            @endif
                        </div>
                    </div>
                    <div class="text-right">
                        <div class="text-sm font-medium text-gray-900 dark:text-gray-100">
                            {{ $order->formatCurrency() }}
                        </div>
                        <div class="text-xs text-gray-500 dark:text-gray-400">
                            {{ $order->date_created->format('M d, H:i') }}
                        </div>
                    </div>
                </div>

                <div class="grid grid-cols-1 gap-2 mb-3">
                    <div>
                        <div class="text-sm text-gray-900 dark:text-gray-100">
                            {{ $order->billing['first_name'] ?? '' }} {{ $order->billing['last_name'] ?? '' }}
                        </div>
                        <div class="text-xs text-gray-500 dark:text-gray-400">
                            {{ $order->billing['email'] ?? '' }}
                        </div>
                        @if(isset($order->billing['phone']))
                            <div class="text-xs text-gray-500 dark:text-gray-400">
                                {{ $order->billing['phone'] }}
                            </div>
                        @endif
                    </div>
                </div>

                <div class="flex justify-between items-center">
                    <div class="flex items-center space-x-2">
                        <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full
                            bg-{{ $order->payment_color }}-100 text-{{ $order->payment_color }}-800
                            dark:bg-{{ $order->payment_color }}-900 dark:text-{{ $order->payment_color }}-200">
                            {{ $order->payment_method_title }}
                        </span>
                        @if($order->isLocalPickup())
                            <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200">
                                Pickup
                            </span>
                        @endif
                    </div>
                    <div class="flex space-x-3">
                        <a href="{{ route('orders.edit', $order) }}"
                           class="text-indigo-600 dark:text-indigo-400 hover:text-indigo-900 text-sm font-medium">
                            Edit
                        </a>
                    </div>
                </div>
            </div>
            @empty
            <div class="p-8 text-center text-gray-500 dark:text-gray-400">
                No orders found.
            </div>
            @endforelse
        </div>

        <!-- Pagination -->
        @if($orders->hasPages())
        <div class="bg-white dark:bg-gray-800 px-4 py-3 border-t border-gray-200 dark:border-gray-700">
            {{ $orders->links() }}
        </div>
        @endif
    </div>
</div>

<style>
/* Make select dropdown scrollable after 10 items */
.scrollable-select {
    appearance: auto;
    -webkit-appearance: menulist;
    -moz-appearance: menulist;
}

/* For browsers that support it, limit dropdown height */
select.scrollable-select:focus,
select.scrollable-select:active {
    max-height: 300px; /* Approximately 10-12 items */
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const periodSelect = document.getElementById('periodSelect');
    const customDates = document.getElementById('customDates');

    function toggleCustomDates() {
        if (periodSelect.value === 'custom') {
            customDates.classList.remove('hidden');
        } else {
            customDates.classList.add('hidden');
        }
    }

    // Initial check
    toggleCustomDates();

    // Listen for changes
    periodSelect.addEventListener('change', function() {
        toggleCustomDates();

        // Auto-submit for non-custom periods
        if (this.value !== 'custom') {
            document.getElementById('periodForm').submit();
        }
    });
});

// Server DateTime Live Update
function updateServerDateTime() {
    const now = new Date();
    const year = now.getFullYear();
    const month = String(now.getMonth() + 1).padStart(2, '0');
    const day = String(now.getDate()).padStart(2, '0');
    const hours = String(now.getHours()).padStart(2, '0');
    const minutes = String(now.getMinutes()).padStart(2, '0');
    const seconds = String(now.getSeconds()).padStart(2, '0');

    const formatted = `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
    const element = document.getElementById('serverDateTime');
    if (element) {
        element.textContent = formatted;
    }
}

// Update datetime every second
setInterval(updateServerDateTime, 1000);

// ROI Calculator Functions
const totalSales = {{ $periodStats['combined_revenue'] }};
const currency = '{{ $periodStats['primary_currency'] }}';

function toggleROIMode(mode) {
    const basicMode = document.getElementById('roiBasicMode');
    const detailedMode = document.getElementById('roiDetailedMode');
    const btnBasic = document.getElementById('btnBasicMode');
    const btnDetailed = document.getElementById('btnDetailedMode');

    if (mode === 'basic') {
        basicMode.classList.remove('hidden');
        detailedMode.classList.add('hidden');
        btnBasic.classList.remove('bg-gray-300', 'dark:bg-gray-600', 'text-gray-700', 'dark:text-gray-300');
        btnBasic.classList.add('bg-indigo-500', 'text-white');
        btnDetailed.classList.remove('bg-indigo-500', 'text-white');
        btnDetailed.classList.add('bg-gray-300', 'dark:bg-gray-600', 'text-gray-700', 'dark:text-gray-300');
    } else {
        basicMode.classList.add('hidden');
        detailedMode.classList.remove('hidden');
        btnDetailed.classList.remove('bg-gray-300', 'dark:bg-gray-600', 'text-gray-700', 'dark:text-gray-300');
        btnDetailed.classList.add('bg-indigo-500', 'text-white');
        btnBasic.classList.remove('bg-indigo-500', 'text-white');
        btnBasic.classList.add('bg-gray-300', 'dark:bg-gray-600', 'text-gray-700', 'dark:text-gray-300');
    }
}

function calculateBasicROI() {
    const spending = parseFloat(document.getElementById('basicSpending').value) || 0;
    const resultDiv = document.getElementById('basicResult');
    const roiValue = document.getElementById('basicROIValue');
    const profitDiv = document.getElementById('basicProfit');

    if (spending > 0) {
        const profit = totalSales - spending;
        const roi = ((profit / spending) * 100).toFixed(2);

        roiValue.textContent = roi + '%';
        roiValue.className = 'text-2xl font-bold ' + (roi >= 0 ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400');
        profitDiv.textContent = `Profit: ${currency} ${profit.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}`;

        resultDiv.classList.remove('hidden');
    } else {
        resultDiv.classList.add('hidden');
    }
}

function calculateDetailedROI() {
    // Get product cost
    let productCost = 0;
    const productCostPercent = parseFloat(document.getElementById('productCostPercent').value) || 0;
    const productCostValue = parseFloat(document.getElementById('productCostValue').value) || 0;

    if (productCostValue > 0) {
        productCost = productCostValue;
        document.getElementById('productCostPercent').value = ''; // Clear percent if value is used
    } else if (productCostPercent > 0) {
        productCost = (totalSales * productCostPercent) / 100;
    }

    // Get shipping cost
    let shippingCost = 0;
    const shippingCostPercent = parseFloat(document.getElementById('shippingCostPercent').value) || 0;
    const shippingCostValue = parseFloat(document.getElementById('shippingCostValue').value) || 0;

    if (shippingCostValue > 0) {
        shippingCost = shippingCostValue;
        document.getElementById('shippingCostPercent').value = ''; // Clear percent if value is used
    } else if (shippingCostPercent > 0) {
        shippingCost = (totalSales * shippingCostPercent) / 100;
    }

    // Get ads spending
    const adsSpending = parseFloat(document.getElementById('adsSpending').value) || 0;

    const resultDiv = document.getElementById('detailedResult');
    const roiValue = document.getElementById('detailedROIValue');
    const breakdownDiv = document.getElementById('detailedBreakdown');
    const profitDiv = document.getElementById('detailedProfit');

    const totalCosts = productCost + shippingCost + adsSpending;

    if (totalCosts > 0) {
        const profit = totalSales - totalCosts;
        const roi = ((profit / totalCosts) * 100).toFixed(2);

        roiValue.textContent = roi + '%';
        roiValue.className = 'text-2xl font-bold ' + (roi >= 0 ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400');

        // Breakdown
        let breakdown = `<div>Total Sales: ${currency} ${totalSales.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</div>`;
        breakdown += `<div>- Product Cost: ${currency} ${productCost.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</div>`;
        breakdown += `<div>- Shipping Cost: ${currency} ${shippingCost.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</div>`;
        breakdown += `<div>- Ads Spending: ${currency} ${adsSpending.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</div>`;
        breakdown += `<div class="border-t border-gray-300 dark:border-gray-500 mt-1 pt-1">Total Costs: ${currency} ${totalCosts.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</div>`;

        breakdownDiv.innerHTML = breakdown;
        profitDiv.textContent = `Profit: ${currency} ${profit.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}`;

        resultDiv.classList.remove('hidden');
    } else {
        resultDiv.classList.add('hidden');
    }
}

// Reset ROI inputs when page loads (happens when filter changes)
document.addEventListener('DOMContentLoaded', function() {
    document.getElementById('basicSpending').value = '';
    document.getElementById('productCostPercent').value = '';
    document.getElementById('productCostValue').value = '';
    document.getElementById('shippingCostPercent').value = '';
    document.getElementById('shippingCostValue').value = '';
    document.getElementById('adsSpending').value = '';
    document.getElementById('basicResult').classList.add('hidden');
    document.getElementById('detailedResult').classList.add('hidden');
});

// Update Server DateTime every second
setInterval(function() {
    const now = new Date();
    const year = now.getFullYear();
    const month = String(now.getMonth() + 1).padStart(2, '0');
    const day = String(now.getDate()).padStart(2, '0');
    const hours = String(now.getHours()).padStart(2, '0');
    const minutes = String(now.getMinutes()).padStart(2, '0');
    const seconds = String(now.getSeconds()).padStart(2, '0');

    const formatted = `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;

    const element = document.getElementById('serverDateTime');
    if (element) {
        element.textContent = formatted;
    }
}, 1000);
</script>
@endsection