<?php
/**
 * Omnia PWA Installer
 *
 * A simple installation wizard for Omnia PWA
 * DELETE THIS FILE after successful installation!
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

// Prevent access if already installed and locked
$lockFile = __DIR__ . '/../storage/installed.lock';
if (file_exists($lockFile) && !isset($_GET['force'])) {
    die('Installation is locked. Delete storage/installed.lock or add ?force=1 to reinstall.');
}

// Define base path
define('BASE_PATH', realpath(__DIR__ . '/..'));

// Current step
$step = isset($_GET['step']) ? (int)$_GET['step'] : 1;
$action = isset($_POST['action']) ? $_POST['action'] : '';

// Handle AJAX requests
if ($action) {
    handleAction($action);
    exit;
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Omnia Installer</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: 10px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            overflow: hidden;
        }
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        .header h1 { font-size: 32px; margin-bottom: 10px; }
        .header p { opacity: 0.9; }
        .content { padding: 40px; }
        .progress-bar {
            height: 8px;
            background: #e0e0e0;
            border-radius: 4px;
            overflow: hidden;
            margin-bottom: 30px;
        }
        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #667eea 0%, #764ba2 100%);
            transition: width 0.3s;
        }
        .step { display: none; }
        .step.active { display: block; }
        .requirement {
            display: flex;
            justify-content: space-between;
            padding: 12px;
            border-bottom: 1px solid #e0e0e0;
        }
        .requirement:last-child { border-bottom: none; }
        .status {
            padding: 4px 12px;
            border-radius: 4px;
            font-size: 14px;
            font-weight: 600;
        }
        .status.pass { background: #d4edda; color: #155724; }
        .status.fail { background: #f8d7da; color: #721c24; }
        .status.warn { background: #fff3cd; color: #856404; }
        .btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 14px 30px;
            border-radius: 6px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s;
        }
        .btn:hover { transform: translateY(-2px); }
        .btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
            transform: none;
        }
        .form-group {
            margin-bottom: 20px;
        }
        .form-group label {
            display: block;
            margin-bottom: 6px;
            font-weight: 600;
            color: #333;
        }
        .form-group input, .form-group select {
            width: 100%;
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 6px;
            font-size: 14px;
        }
        .form-group input:focus, .form-group select:focus {
            outline: none;
            border-color: #667eea;
        }
        .alert {
            padding: 16px;
            border-radius: 6px;
            margin-bottom: 20px;
        }
        .alert-success { background: #d4edda; color: #155724; border-left: 4px solid #28a745; }
        .alert-danger { background: #f8d7da; color: #721c24; border-left: 4px solid #dc3545; }
        .alert-warning { background: #fff3cd; color: #856404; border-left: 4px solid #ffc107; }
        .alert-info { background: #d1ecf1; color: #0c5460; border-left: 4px solid #17a2b8; }
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255,255,255,0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 1s linear infinite;
        }
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        .step-number {
            display: inline-block;
            width: 30px;
            height: 30px;
            background: #667eea;
            color: white;
            border-radius: 50%;
            text-align: center;
            line-height: 30px;
            margin-right: 10px;
            font-weight: 600;
        }
        code {
            background: #f4f4f4;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: monospace;
        }
        .command-box {
            background: #2d2d2d;
            color: #f8f8f2;
            padding: 16px;
            border-radius: 6px;
            font-family: monospace;
            font-size: 14px;
            overflow-x: auto;
            margin: 16px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🚀 Omnia PWA Installer</h1>
            <p>Welcome! Let's get your PWA installed in a few simple steps.</p>
            <?php if (stripos(PHP_OS, 'WIN') === 0): ?>
            <p style="background: rgba(255,255,255,0.2); padding: 10px; border-radius: 6px; margin-top: 10px;">
                ℹ️ Running on Windows (<?php echo PHP_OS; ?>)
            </p>
            <?php endif; ?>
        </div>

        <div class="content">
            <div class="progress-bar">
                <div class="progress-fill" id="progressBar" style="width: <?php echo ($step / 6 * 100); ?>%"></div>
            </div>

            <!-- Step 1: Requirements Check -->
            <div class="step <?php echo $step == 1 ? 'active' : ''; ?>" id="step1">
                <h2><span class="step-number">1</span>System Requirements</h2>
                <p style="margin: 20px 0; color: #666;">Checking if your server meets the requirements...</p>

                <div id="requirements">
                    <?php echo checkRequirements(); ?>
                </div>

                <div style="margin-top: 30px;">
                    <button class="btn" onclick="nextStep(2)" id="continueBtn1">Continue to Database Setup</button>
                </div>
            </div>

            <!-- Step 2: Database Configuration -->
            <div class="step <?php echo $step == 2 ? 'active' : ''; ?>" id="step2">
                <h2><span class="step-number">2</span>Database Configuration</h2>
                <p style="margin: 20px 0; color: #666;">Enter your database credentials:</p>

                <form id="dbForm" onsubmit="return false;">
                    <div class="form-group">
                        <label>Database Host</label>
                        <input type="text" name="db_host" value="127.0.0.1" required>
                    </div>
                    <div class="form-group">
                        <label>Database Port</label>
                        <input type="text" name="db_port" value="3306" required>
                    </div>
                    <div class="form-group">
                        <label>Database Name</label>
                        <input type="text" name="db_database" required placeholder="omnia_db">
                    </div>
                    <div class="form-group">
                        <label>Database Username</label>
                        <input type="text" name="db_username" required>
                    </div>
                    <div class="form-group">
                        <label>Database Password</label>
                        <input type="password" name="db_password">
                    </div>

                    <div id="dbTestResult"></div>

                    <button type="button" class="btn" onclick="testDatabase()">Test Connection</button>
                    <button type="button" class="btn" onclick="nextStep(3)" id="continueBtn2" disabled>Continue</button>
                </form>
            </div>

            <!-- Step 3: Application Settings -->
            <div class="step <?php echo $step == 3 ? 'active' : ''; ?>" id="step3">
                <h2><span class="step-number">3</span>Application Settings</h2>
                <p style="margin: 20px 0; color: #666;">Configure your application:</p>

                <form id="appForm">
                    <div class="form-group">
                        <label>Application Name</label>
                        <input type="text" name="app_name" value="Omnia" required>
                    </div>
                    <div class="form-group">
                        <label>Application URL</label>
                        <input type="url" name="app_url" value="<?php echo getCurrentUrl(); ?>" required>
                    </div>
                    <div class="form-group">
                        <label>Environment</label>
                        <select name="app_env">
                            <option value="production">Production</option>
                            <option value="local">Local/Development</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label>Admin Email (for PWA notifications)</label>
                        <input type="email" name="admin_email" required placeholder="admin@yourdomain.com">
                    </div>

                    <button type="button" class="btn" onclick="saveSettings()">Save & Continue</button>
                </form>
            </div>

            <!-- Step 4: Installation -->
            <div class="step <?php echo $step == 4 ? 'active' : ''; ?>" id="step4">
                <h2><span class="step-number">4</span>Installing...</h2>
                <p style="margin: 20px 0; color: #666;">Please wait while we set up your application.</p>

                <div id="installProgress"></div>

                <div style="margin-top: 30px;">
                    <button class="btn" onclick="startInstallation()" id="installBtn">Start Installation</button>
                </div>
            </div>

            <!-- Step 5: Completion -->
            <div class="step <?php echo $step == 5 ? 'active' : ''; ?>" id="step5">
                <h2><span class="step-number">✓</span>Installation Complete!</h2>

                <div class="alert alert-success" style="margin: 20px 0;">
                    <strong>Congratulations!</strong> Omnia PWA has been successfully installed.
                </div>

                <div id="completionInfo"></div>

                <div class="alert alert-warning" style="margin: 20px 0;">
                    <strong>⚠️ Important Security Step:</strong><br>
                    Please delete this <code>install.php</code> file immediately for security reasons.
                </div>

                <div class="command-box">
                    rm public/install.php
                </div>

                <div style="margin-top: 30px;">
                    <button class="btn" onclick="window.location.href='/'">Go to Application</button>
                </div>
            </div>
        </div>
    </div>

    <script>
        function nextStep(step) {
            document.querySelectorAll('.step').forEach(s => s.classList.remove('active'));
            document.getElementById('step' + step).classList.add('active');
            updateProgress(step);
            window.history.pushState({}, '', '?step=' + step);
        }

        function updateProgress(step) {
            const progress = (step / 6) * 100;
            document.getElementById('progressBar').style.width = progress + '%';
        }

        async function testDatabase() {
            const form = document.getElementById('dbForm');
            const formData = new FormData(form);
            formData.append('action', 'test_database');

            const resultDiv = document.getElementById('dbTestResult');
            resultDiv.innerHTML = '<div class="alert alert-info">Testing connection...</div>';

            try {
                const response = await fetch('install.php', {
                    method: 'POST',
                    body: formData
                });
                const result = await response.json();

                if (result.success) {
                    resultDiv.innerHTML = '<div class="alert alert-success">' + result.message + '</div>';
                    document.getElementById('continueBtn2').disabled = false;
                } else {
                    resultDiv.innerHTML = '<div class="alert alert-danger">' + result.message + '</div>';
                }
            } catch (error) {
                resultDiv.innerHTML = '<div class="alert alert-danger">Error: ' + error.message + '</div>';
            }
        }

        async function saveSettings() {
            const form = document.getElementById('appForm');
            const formData = new FormData(form);

            // Add database settings from previous step
            const dbForm = document.getElementById('dbForm');
            new FormData(dbForm).forEach((value, key) => {
                formData.append(key, value);
            });

            formData.append('action', 'save_env');

            try {
                const response = await fetch('install.php', {
                    method: 'POST',
                    body: formData
                });
                const result = await response.json();

                if (result.success) {
                    nextStep(4);
                } else {
                    alert('Error: ' + result.message);
                }
            } catch (error) {
                alert('Error: ' + error.message);
            }
        }

        async function startInstallation() {
            const btn = document.getElementById('installBtn');
            btn.disabled = true;
            btn.innerHTML = '<span class="loading"></span> Installing...';

            const progressDiv = document.getElementById('installProgress');
            const steps = [
                { action: 'generate_key', label: 'Generating APP_KEY' },
                { action: 'run_migrations', label: 'Running database migrations' },
                { action: 'generate_vapid', label: 'Generating VAPID keys for PWA' },
                { action: 'set_permissions', label: 'Setting file permissions' },
                { action: 'create_lock', label: 'Creating installation lock' }
            ];

            for (const step of steps) {
                progressDiv.innerHTML += '<div class="alert alert-info">' + step.label + '...</div>';

                try {
                    const formData = new FormData();
                    formData.append('action', step.action);

                    const response = await fetch('install.php', {
                        method: 'POST',
                        body: formData
                    });
                    const result = await response.json();

                    if (result.success) {
                        progressDiv.lastChild.className = 'alert alert-success';
                        progressDiv.lastChild.innerHTML = '✓ ' + step.label + ' - Complete';

                        if (result.data) {
                            document.getElementById('completionInfo').innerHTML +=
                                '<div class="alert alert-info">' + result.data + '</div>';
                        }

                        if (result.debug) {
                            console.log('Debug:', result.debug);
                        }
                    } else {
                        progressDiv.lastChild.className = 'alert alert-danger';
                        let errorMsg = '✗ ' + step.label + ' - Failed: ' + result.message;
                        if (result.debug) {
                            errorMsg += '<br><small style="opacity:0.8">' + result.debug + '</small>';
                        }
                        progressDiv.lastChild.innerHTML = errorMsg;
                        return;
                    }
                } catch (error) {
                    progressDiv.lastChild.className = 'alert alert-danger';
                    progressDiv.lastChild.innerHTML = '✗ ' + step.label + ' - Error: ' + error.message;
                    return;
                }

                await new Promise(resolve => setTimeout(resolve, 500));
            }

            nextStep(5);
        }
    </script>
</body>
</html>

<?php

function checkRequirements() {
    $requirements = [
        'PHP Version >= 8.2' => version_compare(PHP_VERSION, '8.2.0', '>='),
        'PDO Extension' => extension_loaded('pdo'),
        'PDO MySQL Driver' => extension_loaded('pdo_mysql'),
        'OpenSSL Extension' => extension_loaded('openssl'),
        'Mbstring Extension' => extension_loaded('mbstring'),
        'Tokenizer Extension' => extension_loaded('tokenizer'),
        'XML Extension' => extension_loaded('xml'),
        'Ctype Extension' => extension_loaded('ctype'),
        'JSON Extension' => extension_loaded('json'),
        'BCMath Extension' => extension_loaded('bcmath'),
        'GMP Extension (for VAPID)' => extension_loaded('gmp'),
        '.env.example exists' => file_exists(BASE_PATH . '/.env.example'),
        'vendor/autoload.php exists' => file_exists(BASE_PATH . '/vendor/autoload.php'),
        'storage/ writable' => is_writable(BASE_PATH . '/storage'),
        'bootstrap/cache writable' => is_writable(BASE_PATH . '/bootstrap/cache'),
    ];

    $html = '';
    $allPassed = true;

    foreach ($requirements as $requirement => $passed) {
        $status = $passed ? 'pass' : 'fail';
        $text = $passed ? 'PASS' : 'FAIL';

        if (!$passed) $allPassed = false;

        $html .= sprintf(
            '<div class="requirement"><span>%s</span><span class="status %s">%s</span></div>',
            $requirement,
            $status,
            $text
        );
    }

    if (!$allPassed) {
        $html .= '<div class="alert alert-danger" style="margin-top: 20px;">
            <strong>Some requirements failed!</strong> Please fix the issues above before continuing.
        </div>';

        // Special message if vendor is missing
        if (!file_exists(BASE_PATH . '/vendor/autoload.php')) {
            $html .= '<div class="alert alert-warning" style="margin-top: 10px;">
                <strong>Missing Composer Dependencies:</strong><br>
                Please run: <code>composer install --no-dev</code> from the project root directory first.
            </div>';
        }
    }

    return $html;
}

function getCurrentUrl() {
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' || $_SERVER['SERVER_PORT'] == 443) ? "https://" : "http://";
    $domainName = $_SERVER['HTTP_HOST'];
    return $protocol . $domainName;
}

function getPhpBinary() {
    // Try to find PHP binary
    $phpBinary = PHP_BINARY;

    // If running on Windows/Laragon, try to find PHP
    if (stripos(PHP_OS, 'WIN') === 0) {
        // Check if php is in PATH
        exec('where php 2>&1', $output, $return);
        if ($return === 0 && !empty($output[0])) {
            return 'php';
        }

        // Try Laragon paths
        $laragonPaths = [
            'C:/laragon/bin/php/php-8.3.26-Win32-vs16-x64/php.exe',
            'C:/laragon/bin/php/php-8.2.*/php.exe',
            'C:/laragon/bin/php/php-8.1.*/php.exe',
        ];

        foreach ($laragonPaths as $path) {
            $matches = glob($path);
            if (!empty($matches) && file_exists($matches[0])) {
                return '"' . $matches[0] . '"';
            }
        }

        // Use PHP_BINARY as fallback
        return '"' . $phpBinary . '"';
    }

    // Linux/Unix - just use 'php'
    return 'php';
}

function handleAction($action) {
    header('Content-Type: application/json');

    $php = getPhpBinary();

    try {
        switch ($action) {
            case 'test_database':
                $host = $_POST['db_host'] ?? '';
                $port = $_POST['db_port'] ?? '3306';
                $database = $_POST['db_database'] ?? '';
                $username = $_POST['db_username'] ?? '';
                $password = $_POST['db_password'] ?? '';

                try {
                    $dsn = "mysql:host=$host;port=$port;dbname=$database";
                    $pdo = new PDO($dsn, $username, $password);
                    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

                    echo json_encode([
                        'success' => true,
                        'message' => 'Database connection successful!'
                    ]);
                } catch (PDOException $e) {
                    echo json_encode([
                        'success' => false,
                        'message' => 'Connection failed: ' . $e->getMessage()
                    ]);
                }
                break;

            case 'save_env':
                $envExample = file_get_contents(BASE_PATH . '/.env.example');

                // Backup existing .env if exists
                if (file_exists(BASE_PATH . '/.env')) {
                    $backup = BASE_PATH . '/.env.backup.' . date('Y-m-d-H-i-s');
                    copy(BASE_PATH . '/.env', $backup);
                }

                // Replace values
                $env = str_replace([
                    'APP_NAME=Laravel',
                    'APP_ENV=local',
                    'APP_URL=http://localhost',
                    'DB_HOST=127.0.0.1',
                    'DB_PORT=3306',
                    'DB_DATABASE=multistore',
                    'DB_USERNAME=root',
                    'DB_PASSWORD=',
                ], [
                    'APP_NAME=' . $_POST['app_name'],
                    'APP_ENV=' . $_POST['app_env'],
                    'APP_URL=' . rtrim($_POST['app_url'], '/'),
                    'DB_HOST=' . $_POST['db_host'],
                    'DB_PORT=' . $_POST['db_port'],
                    'DB_DATABASE=' . $_POST['db_database'],
                    'DB_USERNAME=' . $_POST['db_username'],
                    'DB_PASSWORD=' . $_POST['db_password'],
                ], $envExample);

                // Add VAPID subject
                $env = str_replace(
                    'VAPID_SUBJECT=mailto:admin@example.com',
                    'VAPID_SUBJECT=mailto:' . $_POST['admin_email'],
                    $env
                );

                file_put_contents(BASE_PATH . '/.env', $env);

                echo json_encode([
                    'success' => true,
                    'message' => 'Environment file created successfully!'
                ]);
                break;

            case 'generate_key':
                // Use correct cd command based on OS
                $cdCmd = (stripos(PHP_OS, 'WIN') === 0) ? 'cd /d' : 'cd';
                $cmd = $cdCmd . ' "' . BASE_PATH . '" && ' . $php . ' artisan key:generate --force 2>&1';
                exec($cmd, $output, $return);
                echo json_encode([
                    'success' => $return === 0,
                    'message' => $return === 0 ? 'APP_KEY generated' : 'Failed: ' . implode("\n", $output),
                    'debug' => 'PHP: ' . $php . ' | OS: ' . PHP_OS . ' | CMD: ' . $cmd
                ]);
                break;

            case 'run_migrations':
                // Use correct cd command based on OS
                $cdCmd = (stripos(PHP_OS, 'WIN') === 0) ? 'cd /d' : 'cd';
                $cmd = $cdCmd . ' "' . BASE_PATH . '" && ' . $php . ' artisan migrate --force 2>&1';
                exec($cmd, $output, $return);
                echo json_encode([
                    'success' => $return === 0,
                    'message' => $return === 0 ? 'Migrations completed' : 'Failed: ' . implode("\n", $output),
                    'data' => $return === 0 ? 'Database tables created successfully (88 migrations)' : null
                ]);
                break;

            case 'generate_vapid':
                // Use correct cd command based on OS
                $cdCmd = (stripos(PHP_OS, 'WIN') === 0) ? 'cd /d' : 'cd';
                $cmd = $cdCmd . ' "' . BASE_PATH . '" && ' . $php . ' artisan webpush:vapid 2>&1';
                exec($cmd, $output, $return);

                // Parse VAPID keys from output
                $publicKey = '';
                $privateKey = '';
                foreach ($output as $line) {
                    if (strpos($line, 'Public key:') !== false) {
                        $publicKey = trim(str_replace('Public key:', '', $line));
                    }
                    if (strpos($line, 'Private key:') !== false) {
                        $privateKey = trim(str_replace('Private key:', '', $line));
                    }
                }

                // Update .env with VAPID keys
                if ($publicKey && $privateKey) {
                    $env = file_get_contents(BASE_PATH . '/.env');
                    $env = preg_replace('/VAPID_PUBLIC_KEY=.*/', 'VAPID_PUBLIC_KEY=' . $publicKey, $env);
                    $env = preg_replace('/VAPID_PRIVATE_KEY=.*/', 'VAPID_PRIVATE_KEY=' . $privateKey, $env);
                    file_put_contents(BASE_PATH . '/.env', $env);
                }

                echo json_encode([
                    'success' => $return === 0 && $publicKey && $privateKey,
                    'message' => $return === 0 ? 'VAPID keys generated' : 'Failed: ' . implode("\n", $output),
                    'data' => ($publicKey && $privateKey) ? '<strong>VAPID Keys Generated:</strong><br>Public: ' . substr($publicKey, 0, 20) . '...<br>Private: ' . substr($privateKey, 0, 20) . '...<br><em>(Saved to .env file)</em>' : null
                ]);
                break;

            case 'set_permissions':
                // On Windows, permissions are handled differently
                if (stripos(PHP_OS, 'WIN') === 0) {
                    // Windows - just check if directories are writable
                    $storageWritable = is_writable(BASE_PATH . '/storage');
                    $cacheWritable = is_writable(BASE_PATH . '/bootstrap/cache');

                    echo json_encode([
                        'success' => true,
                        'message' => 'File permissions checked (Windows)',
                        'data' => ($storageWritable && $cacheWritable) ?
                            'All directories are writable ✓' :
                            'Warning: Some directories may not be writable. Check storage/ and bootstrap/cache/'
                    ]);
                } else {
                    // Linux/Unix - set permissions
                    exec('chmod -R 775 ' . escapeshellarg(BASE_PATH . '/storage') . ' 2>&1', $output1, $return1);
                    exec('chmod -R 775 ' . escapeshellarg(BASE_PATH . '/bootstrap/cache') . ' 2>&1', $output2, $return2);

                    echo json_encode([
                        'success' => true,
                        'message' => 'File permissions set (775)'
                    ]);
                }
                break;

            case 'create_lock':
                $lockContent = "Installation completed on " . date('Y-m-d H:i:s') . "\nDo not delete this file unless you want to reinstall.";
                file_put_contents(BASE_PATH . '/storage/installed.lock', $lockContent);

                echo json_encode([
                    'success' => true,
                    'message' => 'Installation locked',
                    'data' => '<strong>Next Steps:</strong><ul style="margin-left: 20px; margin-top: 10px;">
                        <li>Delete <code>public/install.php</code> for security</li>
                        <li>Setup queue workers (see docs)</li>
                        <li>Setup cron jobs (see docs)</li>
                        <li>Configure SSL certificate (REQUIRED for PWA!)</li>
                        <li>Test PWA notifications</li>
                    </ul>'
                ]);
                break;

            default:
                echo json_encode([
                    'success' => false,
                    'message' => 'Unknown action'
                ]);
        }
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => $e->getMessage()
        ]);
    }
}
?>
