#!/usr/bin/env python3
"""
Prophet Forecasting Model for Sales Prediction
Free and open-source forecasting tool by Meta/Facebook

Installation:
    pip install prophet pandas

Usage:
    python forecast_prophet.py
"""

import sys
import json
import pandas as pd
from datetime import datetime, timedelta
import warnings

# Suppress warnings for cleaner output
warnings.filterwarnings('ignore')

# Redirect stderr to devnull to suppress Prophet's logging
import os
stderr = sys.stderr
sys.stderr = open(os.devnull, 'w')

try:
    from prophet import Prophet
except ImportError:
    try:
        # Try legacy fbprophet import
        from fbprophet import Prophet
    except ImportError:
        sys.stderr = stderr
        print(json.dumps({
            'error': 'Prophet not installed. Run: pip install prophet',
            'install_command': 'pip install prophet pandas cmdstanpy'
        }))
        sys.exit(1)


def load_data_from_json(json_data):
    """Load historical sales data from JSON input"""
    try:
        data = json.loads(json_data)
        df = pd.DataFrame(data)

        # Create proper date column
        df['ds'] = pd.to_datetime(df['year'].astype(str) + '-' +
                                  df['month'].astype(str).str.zfill(2) + '-01')
        df['y'] = df['revenue'].astype(float)

        return df[['ds', 'y']].sort_values('ds')
    except Exception as e:
        print(json.dumps({'error': f'Data loading error: {str(e)}'}))
        sys.exit(1)


def forecast_sales(historical_data, periods=6, frequency='M'):
    """
    Generate sales forecast using Prophet

    Args:
        historical_data: JSON string with columns: year, month, revenue
        periods: Number of periods to forecast (default: 6 months)
        frequency: 'M' for monthly, 'D' for daily

    Returns:
        JSON with forecast data
    """
    # Load and prepare data
    df = load_data_from_json(historical_data)

    if len(df) < 12:
        return json.dumps({
            'error': 'Insufficient data. Need at least 12 months of historical data.',
            'data_points': len(df)
        })

    # Initialize Prophet model
    model = Prophet(
        yearly_seasonality=True,
        weekly_seasonality=False,
        daily_seasonality=False,
        seasonality_mode='multiplicative',  # Better for business data with varying magnitude
        changepoint_prior_scale=0.05,  # Flexibility of trend changes
        seasonality_prior_scale=10.0,  # Strength of seasonality
        interval_width=0.95  # 95% confidence interval
    )

    # Fit the model
    model.fit(df)

    # Create future dataframe
    future = model.make_future_dataframe(periods=periods, freq='MS')  # MS = Month Start

    # Generate forecast
    forecast = model.predict(future)

    # Prepare output
    forecast_output = []
    for idx, row in forecast.tail(periods).iterrows():
        forecast_output.append({
            'date': row['ds'].strftime('%Y-%m-%d'),
            'year': row['ds'].year,
            'month': row['ds'].month,
            'month_name': row['ds'].strftime('%B'),
            'forecast_revenue': max(0, round(row['yhat'], 2)),  # Don't allow negative
            'lower_bound': max(0, round(row['yhat_lower'], 2)),
            'upper_bound': max(0, round(row['yhat_upper'], 2)),
            'confidence': 'high' if len(df) >= 36 else ('medium' if len(df) >= 24 else 'low')
        })

    # Calculate trend
    recent_trend = forecast['trend'].tail(3).mean()
    older_trend = forecast['trend'].head(3).mean()
    trend_direction = 'growing' if recent_trend > older_trend * 1.05 else (
        'declining' if recent_trend < older_trend * 0.95 else 'stable'
    )

    # Get seasonality components
    seasonality_strength = forecast['yearly'].abs().mean() if 'yearly' in forecast.columns else 0

    total_forecast = sum(f['forecast_revenue'] for f in forecast_output)
    avg_monthly = total_forecast / len(forecast_output) if forecast_output else 0

    return json.dumps({
        'success': True,
        'model': 'Prophet by Meta',
        'data_points': len(df),
        'forecast_periods': periods,
        'trend': trend_direction,
        'seasonality_strength': round(seasonality_strength, 2),
        'forecasts': forecast_output,
        'summary': {
            'next_month_revenue': forecast_output[0]['forecast_revenue'] if forecast_output else 0,
            'next_6_months_total': total_forecast,  # This key name is generic but used for all periods
            'average_monthly': avg_monthly
        }
    }, indent=2)


def main():
    """Main entry point"""
    try:
        # Get periods from command line argument (default: 6)
        periods = 6
        if len(sys.argv) > 1 and sys.argv[1].isdigit():
            periods = int(sys.argv[1])
            # Read data from stdin
            historical_data = sys.stdin.read()
        elif len(sys.argv) > 2:
            # First arg is data, second is periods
            historical_data = sys.argv[1]
            periods = int(sys.argv[2]) if sys.argv[2].isdigit() else 6
        else:
            # Read from stdin
            historical_data = sys.stdin.read()

        # Generate forecast
        result = forecast_sales(historical_data, periods=periods)

        # Restore stderr before printing result
        sys.stderr = stderr

        # Print only the JSON result
        print(result)
    except Exception as e:
        sys.stderr = stderr
        print(json.dumps({
            'error': f'Forecasting failed: {str(e)}',
            'type': type(e).__name__
        }))


if __name__ == '__main__':
    main()
