# General Debugging Guide

This guide covers common issues and debugging techniques for the Multistore system.

## Laravel Logs

The first place to check when something goes wrong:

```bash
tail -100 storage/logs/laravel.log
```

Watch logs in real-time:
```bash
tail -f storage/logs/laravel.log
```

Filter for specific issues:
```bash
grep -i "error" storage/logs/laravel.log | tail -50
grep -i "google" storage/logs/laravel.log | tail -50
grep -i "queue" storage/logs/laravel.log | tail -50
```

## Enable Debug Mode (Development Only)

**Never enable debug mode in production!**

Edit `.env`:
```env
APP_DEBUG=true
APP_ENV=local
```

Clear config cache:
```bash
php artisan config:clear
```

This will show detailed error messages in the browser.

## Clear All Caches

When things aren't updating or behaving strangely:

```bash
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

Or all at once:
```bash
php artisan optimize:clear
```

## Database Issues

### Connection Errors

**Check database credentials** in `.env`:
```env
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=multistore
DB_USERNAME=your_username
DB_PASSWORD=your_password
```

**Test database connection:**
```bash
php artisan tinker
```

Then try:
```php
DB::connection()->getPdo();
```

If successful, you'll see PDO object details. If it fails, check your credentials.

### Migration Errors

**Reset and re-run migrations** (development only - destroys all data):
```bash
php artisan migrate:fresh --seed
```

**Check migration status:**
```bash
php artisan migrate:status
```

**Rollback last migration:**
```bash
php artisan migrate:rollback
```

## Permission Issues

### File Permission Errors

Laravel needs write access to certain directories:

```bash
chmod -R 775 storage bootstrap/cache
chown -R www-data:www-data storage bootstrap/cache
```

Replace `www-data` with your web server user.

### Storage Link Missing

If uploaded files don't show:

```bash
php artisan storage:link
```

This creates a symbolic link from `public/storage` to `storage/app/public`.

## Blank White Page

**Possible causes:**

1. **PHP errors with debug mode off**
   - Check `storage/logs/laravel.log`
   - Temporarily enable APP_DEBUG=true to see the error

2. **Permission issues**
   - Check storage and bootstrap/cache permissions

3. **Missing .htaccess**
   - Make sure `public/.htaccess` exists
   - Enable mod_rewrite in Apache

4. **Memory limit**
   - Increase PHP memory_limit in php.ini

## 500 Internal Server Error

Check these in order:

1. **Laravel logs:** `storage/logs/laravel.log`
2. **Web server logs:**
   - Apache: `/var/log/apache2/error.log`
   - Nginx: `/var/log/nginx/error.log`
3. **PHP-FPM logs:** `/var/log/php8.1-fpm.log` (version may vary)

Common causes:
- Syntax errors in code
- Missing PHP extensions
- Incorrect file permissions
- Memory limit exceeded

## 404 Not Found

**Routes not working:**

1. Check if route exists:
   ```bash
   php artisan route:list | grep "your-route-name"
   ```

2. Clear route cache:
   ```bash
   php artisan route:clear
   ```

3. Check `.htaccess` or Nginx configuration
4. Ensure mod_rewrite is enabled (Apache)

## Session/Authentication Issues

### Users Getting Logged Out

1. **Check session driver** in `.env`:
   ```env
   SESSION_DRIVER=file
   ```

2. **Clear sessions:**
   ```bash
   php artisan session:flush
   ```

3. **Check session permissions:**
   ```bash
   chmod -R 775 storage/framework/sessions
   ```

### CSRF Token Mismatch (419 Error)

See dedicated [CSRF Errors Guide](csrf-errors.md).

Quick fixes:
```bash
php artisan cache:clear
php artisan config:clear
```

## Queue Jobs Not Processing

**Check if worker is running:**
```bash
ps aux | grep "queue:work"
```

**Start queue worker:**
```bash
php artisan queue:work database --verbose
```

**Check failed jobs:**
```bash
php artisan queue:failed
```

**Retry failed jobs:**
```bash
php artisan queue:retry all
```

For more details, see [Queue Workers Guide](../deployment/queue-workers.md).

## Slow Performance

### Database Query Optimization

Enable query logging to find slow queries:

```php
// In AppServiceProvider boot() method
DB::listen(function($query) {
    if ($query->time > 1000) {  // Queries taking > 1 second
        Log::warning('Slow query', [
            'sql' => $query->sql,
            'time' => $query->time
        ]);
    }
});
```

### Cache Everything

Use caching for frequently accessed data:

```bash
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

**Note:** After code changes, clear these caches.

### Optimize Composer Autoloader

```bash
composer dump-autoload --optimize
```

## Browser Console Errors

Press **F12** in your browser to open Developer Tools.

Check the **Console** tab for JavaScript errors.
Check the **Network** tab for failed requests.

Common issues:
- Missing JavaScript files (404)
- CORS errors (cross-origin requests)
- JavaScript syntax errors

## Debugging Techniques

### Laravel Tinker

Interactive PHP shell:

```bash
php artisan tinker
```

Test code live:
```php
$user = App\Models\User::find(1);
$user->name;

$orders = App\Models\Order::count();
```

### dd() and dump()

Add to your code for debugging:

```php
// Die and dump
dd($variable);

// Dump and continue
dump($variable);
```

### Log Messages

Add custom log messages:

```php
use Illuminate\Support\Facades\Log;

Log::info('User logged in', ['user_id' => $user->id]);
Log::warning('Something suspicious', ['data' => $data]);
Log::error('Critical error occurred', ['exception' => $e->getMessage()]);
```

View logs:
```bash
tail -f storage/logs/laravel.log
```

## Common Error Messages

### "Class not found"

Run:
```bash
composer dump-autoload
```

### "SQLSTATE[HY000] [2002] Connection refused"

Database not running:
```bash
sudo systemctl start mysql
# or
sudo systemctl start mariadb
```

### "No application encryption key has been set"

Generate key:
```bash
php artisan key:generate
```

### "The stream or file could not be opened"

Permission issue:
```bash
chmod -R 775 storage bootstrap/cache
```

## Production Debugging

**Never enable APP_DEBUG=true in production.**

Instead:

1. **Check logs:**
   ```bash
   tail -100 storage/logs/laravel.log
   ```

2. **Use error tracking service** (Sentry, Bugsnag, etc.)

3. **Enable maintenance mode** while fixing:
   ```bash
   php artisan down
   # Fix the issue
   php artisan up
   ```

## Getting Help

If you're still stuck:

1. Check the specific troubleshooting guide for your issue:
   - [CSRF Errors](csrf-errors.md)
   - [Cache Issues](cache.md)
   - [Google Sheets Issues](../features/google-sheets/troubleshooting.md)
   - [PWA Issues](../features/pwa-push-notifications/troubleshooting.md)

2. Review recent changes in [CHANGELOG](../../CHANGELOG.md)

3. Check Laravel logs thoroughly

4. Search the codebase for similar issues

## Useful Commands Reference

```bash
# View routes
php artisan route:list

# View config
php artisan config:show

# Clear everything
php artisan optimize:clear

# Run tests
php artisan test

# Check queue status
php artisan queue:monitor

# View failed jobs
php artisan queue:failed

# Database status
php artisan migrate:status

# List all artisan commands
php artisan list
```

---

**Last updated:** 2025-11-21
