# Installation Guide

This guide walks you through installing the Multistore order management system from scratch.

## System Requirements

Before you begin, make sure your server meets these requirements:

- **PHP:** 8.1 or higher
- **Database:** MySQL 5.7+ or MariaDB 10.3+
- **Web Server:** Apache, Nginx, or OpenLiteSpeed
- **Composer:** Latest version
- **Node.js:** 16+ (for asset compilation)
- **SSL Certificate:** Required for production (PWA features need HTTPS)

For detailed requirements, see [System Requirements](requirements.md).

## Installation Steps

### 1. Clone the Repository

```bash
cd /var/www
git clone https://github.com/your-repo/multistore.git
cd multistore
```

### 2. Install PHP Dependencies

```bash
composer install --optimize-autoloader --no-dev
```

For development:
```bash
composer install
```

### 3. Install Node Dependencies

```bash
npm install
```

### 4. Configure Environment

Copy the example environment file:

```bash
cp .env.example .env
```

Generate application key:

```bash
php artisan key:generate
```

Edit `.env` and configure your database:

```env
APP_NAME="Multistore System"
APP_ENV=production
APP_DEBUG=false
APP_URL=https://yourdomain.com

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=multistore
DB_USERNAME=your_db_user
DB_PASSWORD=your_db_password
```

### 5. Create Database

Create a new MySQL database:

```sql
CREATE DATABASE multistore CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
```

Grant permissions:

```sql
GRANT ALL PRIVILEGES ON multistore.* TO 'your_db_user'@'localhost';
FLUSH PRIVILEGES;
```

### 6. Run Migrations

```bash
php artisan migrate --seed
```

This creates all database tables and seeds initial data.

### 7. Set Permissions

Set correct file permissions:

```bash
chmod -R 775 storage bootstrap/cache
chown -R www-data:www-data storage bootstrap/cache
```

Replace `www-data` with your web server user (might be `apache`, `nginx`, etc.).

### 8. Build Assets

```bash
npm run build
```

For development:
```bash
npm run dev
```

### 9. Configure Web Server

**Apache (.htaccess included):**

Point document root to `public/` directory:

```apache
<VirtualHost *:80>
    ServerName yourdomain.com
    DocumentRoot /var/www/multistore/public

    <Directory /var/www/multistore/public>
        AllowOverride All
        Require all granted
    </Directory>
</VirtualHost>
```

**Nginx:**

```nginx
server {
    listen 80;
    server_name yourdomain.com;
    root /var/www/multistore/public;

    index index.php;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.1-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }
}
```

### 10. Set Up SSL (Production)

Use Let's Encrypt for free SSL:

```bash
sudo apt install certbot python3-certbot-apache
sudo certbot --apache -d yourdomain.com
```

Or for Nginx:
```bash
sudo certbot --nginx -d yourdomain.com
```

### 11. Configure Queue Worker

For background job processing (required for Google Sheets sync):

```bash
php artisan queue:work database --sleep=3 --tries=3
```

For production, use Supervisor. See [Queue Workers Guide](../deployment/queue-workers.md).

### 12. Create Admin User

```bash
php artisan tinker
```

```php
$user = new App\Models\User();
$user->name = 'Admin';
$user->email = 'admin@yourdomain.com';
$user->password = bcrypt('your-secure-password');
$user->role = 'admin';
$user->save();
```

### 13. Test Installation

Visit your domain: `https://yourdomain.com`

You should see the login page. Log in with your admin credentials.

## Post-Installation

After successful installation:

1. **Configure features** you need (Google Sheets, PWA, etc.)
2. **Set up backups** - See [Backup Guide](../admin/backup-recovery.md)
3. **Review security** settings
4. **Configure queue workers** for production
5. **Test all functionality**

## Optional Features

These features require additional setup:

- **Google Sheets Sync:** [Setup Guide](../features/google-sheets/setup.md)
- **PWA Push Notifications:** [Setup Guide](../features/pwa-push-notifications/setup.md)
- **Sales Forecasting:** [Installation Guide](../features/forecasting/installation.md)
- **Custom Domains:** [Setup Guide](../features/custom-domains/setup.md)

## Troubleshooting

**Database connection errors:**
- Check your `.env` database credentials
- Make sure MySQL/MariaDB is running
- Verify database user has proper permissions

**Permission errors:**
- Run: `chmod -R 775 storage bootstrap/cache`
- Check web server user ownership

**Blank page:**
- Check Laravel logs: `storage/logs/laravel.log`
- Enable debug mode: `APP_DEBUG=true` in `.env` (development only)

For more help, see [Troubleshooting Guide](../troubleshooting/debugging.md).

## Next Steps

- [Deploy to Production](../deployment/production.md)
- [Admin Guide](../admin/setup-guide.md)
- [Features Overview](../admin/features.md)

---

**Need help?** Check the full documentation at `docs/index.md`

**Last updated:** 2025-11-21
