# 🔄 Reinstall with Existing Database

**How to reinstall/update Omnia PWA while keeping your old database data safe**

**Last Updated:** 2025-11-19

---

## 🎯 Scenarios Covered

1. **Update/Reinstall with same database** (keep all data)
2. **New code, old database** (preserve orders, stores, users)
3. **Fresh install but import old data**

---

## ⚠️ IMPORTANT - Backup First!

**ALWAYS backup before any operation!**

```bash
# Backup database
mysqldump -u username -p database_name > backup-$(date +%Y%m%d-%H%M%S).sql

# Backup .env file
cp .env .env.backup-$(date +%Y%m%d-%H%M%S)

# Backup uploaded files (if any)
tar -czf uploads-backup-$(date +%Y%m%d-%H%M%S).tar.gz storage/app/public/
```

**Store backups in safe location!**

---

## 📋 Scenario 1: Update Code, Keep Database

**Use Case:** You have existing site with data, want to update code files only.

### **Step 1: Backup Everything**

```bash
# Backup database
mysqldump -u root -p your_database > backup-before-update.sql

# Backup .env
cp .env .env.backup
```

### **Step 2: Update Code Files**

**Option A: Via Git**
```bash
cd /var/www/your-domain

# Stash local changes
git stash

# Pull latest code
git pull origin main

# Apply stash if needed
git stash pop
```

**Option B: Upload New Files**
- Upload new files via FTP
- **DO NOT overwrite:**
  - `.env` (keep your existing one)
  - `storage/` (keep existing data)
  - Database (don't touch it)

### **Step 3: Update Dependencies**

```bash
# Update Composer packages
composer install --no-dev

# Update Node packages
npm install

# Rebuild assets
npm run build
```

### **Step 4: Run New Migrations (Safe)**

```bash
# This is SAFE - Laravel only runs NEW migrations
php artisan migrate --force

# Laravel checks 'migrations' table and only runs missing ones
# Your existing data is NOT affected!
```

**What happens:**
- ✅ Laravel checks which migrations already ran
- ✅ Only runs NEW migrations (if any)
- ✅ Existing tables and data are SAFE
- ✅ No data loss

### **Step 5: Clear Caches**

```bash
php artisan optimize:clear
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

### **Step 6: Update Service Worker Version**

```bash
# Edit public/service-worker.js
# Change version:
const CACHE_NAME = 'omnia-v2-2025-11-20'; // New date

# Tell users to hard refresh
# Ctrl+Shift+R (Windows) or Cmd+Shift+R (Mac)
```

### **✅ Done!**
- Your code is updated
- Database data is safe
- New features available

---

## 📋 Scenario 2: Fresh Install, Import Old Data

**Use Case:** New server, want to import old database.

### **Step 1: Export Old Database**

**On old server:**
```bash
mysqldump -u username -p old_database > database-export.sql

# Download this file to your computer
```

### **Step 2: Fresh Install on New Server**

```bash
# Upload files
# Run: composer install --no-dev
# Run: npm install && npm run build

# DON'T run installer yet!
```

### **Step 3: Import Old Database**

```bash
# Create database
mysql -u root -p -e "CREATE DATABASE new_database CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"

# Import old data
mysql -u root -p new_database < database-export.sql
```

### **Step 4: Configure .env**

```bash
cp .env.example .env
nano .env
```

Update these:
```env
APP_NAME=Omnia
APP_ENV=production
APP_DEBUG=false
APP_URL=https://your-new-domain.com

DB_DATABASE=new_database
DB_USERNAME=your_username
DB_PASSWORD=your_password

# Copy these from OLD .env:
APP_KEY=base64:... (copy from old .env)
VAPID_PUBLIC_KEY=BM... (copy from old .env)
VAPID_PRIVATE_KEY=Rf... (copy from old .env)
```

**Important:** Use the same APP_KEY and VAPID keys from old site!

### **Step 5: Run New Migrations (If Any)**

```bash
# This is safe - only runs missing migrations
php artisan migrate --force
```

### **Step 6: Clear Caches**

```bash
php artisan optimize:clear
php artisan config:cache
```

### **✅ Done!**
- Old data imported
- New server running
- All data preserved

---

## 📋 Scenario 3: Reinstall on Same Server

**Use Case:** Something broke, want to reinstall but keep database.

### **Step 1: Backup Critical Files**

```bash
# Backup .env
cp .env .env.backup

# Backup database
mysqldump -u root -p database_name > backup.sql

# Backup uploads (if any)
tar -czf uploads-backup.tar.gz storage/app/public/
```

### **Step 2: Delete Old Files (Except .env and storage)**

```bash
# Save .env and storage first
mv .env ~/env-backup
mv storage ~/storage-backup
mv database ~/database-backup # Keep migrations record

# Delete everything
rm -rf /var/www/your-domain/*

# Restore
mv ~/env-backup /var/www/your-domain/.env
mv ~/storage-backup /var/www/your-domain/storage
mv ~/database-backup /var/www/your-domain/database
```

### **Step 3: Upload Fresh Files**

Upload all new files to `/var/www/your-domain/`

### **Step 4: Install Dependencies**

```bash
composer install --no-dev
npm install && npm run build
```

### **Step 5: Run Migrations (Safe)**

```bash
# Laravel will only run NEW migrations
php artisan migrate --force
```

### **Step 6: Clear Caches**

```bash
php artisan optimize:clear
php artisan config:cache
```

### **✅ Done!**
- Fresh code installed
- Database data safe
- Everything working

---

## 🛡️ Safety: How Laravel Migrations Work

### **Why it's SAFE to run `php artisan migrate`:**

Laravel has a **`migrations`** table that tracks which migrations already ran:

```sql
SELECT * FROM migrations;
-- Shows list of all completed migrations
```

**When you run `php artisan migrate`:**

1. ✅ Laravel checks `migrations` table
2. ✅ Compares with files in `database/migrations/`
3. ✅ Only runs migrations NOT in the table
4. ✅ Skips migrations already completed
5. ✅ Your existing data is SAFE

**Example:**
```
Existing migrations: 1-88 (already ran)
New migrations: 89-92 (new code)
Result: Only runs 89-92
Data: SAFE ✅
```

---

## ⚠️ What NOT to Do

### **DON'T:**

❌ **Run `php artisan migrate:fresh`**
   - This DROPS all tables!
   - You will lose ALL data!

❌ **Run `php artisan migrate:refresh`**
   - This rollbacks and re-runs all migrations
   - You will lose ALL data!

❌ **Run `php artisan db:wipe`**
   - This deletes everything!

❌ **Delete the `migrations` table**
   - Laravel won't know what already ran
   - Might try to recreate existing tables

### **DO:**

✅ **Run `php artisan migrate`**
   - Safe - only runs new migrations

✅ **Run `php artisan migrate --force`**
   - Same as above, but in production mode

✅ **Run `php artisan migrate:status`**
   - Shows which migrations ran
   - Doesn't change anything

---

## 🔍 Check Migration Status

### **Before Running Migrations:**

```bash
# See what will run
php artisan migrate:status

# Output shows:
# Ran? Migration
# [x] 2024_01_01_000001_create_stores_table (already ran)
# [x] 2024_01_01_000002_create_orders_table (already ran)
# [ ] 2024_11_19_000001_add_new_feature (will run)
```

**Green [x] = Already ran (safe)**
**Red [ ] = Will run (new)**

---

## 🎯 Using the Installer with Existing Database

### **Can you use install.php with existing database?**

**⚠️ Partially Yes, but NOT recommended!**

The installer will:
- ✅ Create .env file (safe)
- ✅ Generate APP_KEY (overwrites existing!)
- ⚠️ Run migrations (safe if database exists)
- ⚠️ Generate NEW VAPID keys (breaks existing PWA subscriptions!)

**Problems:**
1. New APP_KEY = can't decrypt old data
2. New VAPID keys = all users lose notifications

**Better Approach:**
1. Copy old .env values
2. Run installer
3. Restore APP_KEY and VAPID keys from old .env

---

## 📖 Step-by-Step: Update with Installer

### **If you want to use installer but keep database:**

### **Step 1: Backup Old .env**

```bash
cp .env .env.old
cat .env.old | grep APP_KEY
cat .env.old | grep VAPID
# Save these values!
```

### **Step 2: Note Critical Values**

```env
# From old .env - SAVE THESE:
APP_KEY=base64:abc123... ← SAVE THIS
VAPID_PUBLIC_KEY=BM... ← SAVE THIS
VAPID_PRIVATE_KEY=Rf... ← SAVE THIS
```

### **Step 3: Upload New Files**

```bash
# Delete old files except:
# - .env (backup already made)
# - storage/
# - database/

# Upload all new files
```

### **Step 4: Install Dependencies**

```bash
composer install --no-dev
npm install && npm run build
```

### **Step 5: Run Installer**

Visit: `https://your-domain.com/install.php`

- Use **SAME database** credentials
- Installer will detect existing database
- Will only run NEW migrations

### **Step 6: Restore Critical Keys**

```bash
nano .env

# Replace these with OLD values:
APP_KEY=base64:... (use old value)
VAPID_PUBLIC_KEY=BM... (use old value)
VAPID_PRIVATE_KEY=Rf... (use old value)
```

### **Step 7: Clear Caches**

```bash
php artisan config:clear
php artisan cache:clear
```

### **✅ Done!**
- New code installed
- Old database safe
- Old encryption keys preserved
- PWA notifications still work

---

## 🔄 Migration Conflicts

### **What if new migration conflicts with existing table?**

**Example:** New migration tries to create table that already exists.

**Laravel will show error:**
```
SQLSTATE[42S01]: Base table or view already exists: 1050 Table 'orders' already exists
```

**Solutions:**

### **Solution 1: Skip Specific Migration**

```bash
# Mark migration as ran without actually running it
php artisan migrate:fake database/migrations/2024_01_01_000002_create_orders_table.php
```

### **Solution 2: Modify Migration**

Edit the migration file:
```php
// Before:
public function up() {
    Schema::create('orders', function (Blueprint $table) {
        // ...
    });
}

// After (safe):
public function up() {
    if (!Schema::hasTable('orders')) {
        Schema::create('orders', function (Blueprint $table) {
            // ...
        });
    }
}
```

### **Solution 3: Roll Back One Migration**

```bash
# Rollback last migration (careful!)
php artisan migrate:rollback --step=1

# Then run again
php artisan migrate
```

---

## 📊 Database Version Checking

### **Check if your database is up-to-date:**

```bash
# Show migration status
php artisan migrate:status

# Count total migrations
php artisan migrate:status | grep -c "Ran"

# Expected: 88 migrations (current version)
```

### **If you have LESS than 88 migrations:**

```bash
# Run to update database
php artisan migrate --force

# This adds new tables/columns
# Existing data is SAFE
```

---

## 🎯 Quick Decision Tree

**Do you have existing database with data?**

**YES → Use Method 1 (Update Code, Keep Database)**
```bash
1. Backup database
2. Upload new files (don't overwrite .env)
3. composer install --no-dev
4. npm install && npm run build
5. php artisan migrate --force (safe!)
6. php artisan optimize:clear
```

**Moving to new server?**
```bash
1. Export old database
2. Fresh install on new server
3. Import old database
4. Copy APP_KEY and VAPID keys from old .env
5. php artisan migrate --force (safe!)
```

**Database is corrupted/broken?**
```bash
1. Backup what you can
2. Fresh install
3. Import old database
4. php artisan migrate:status (check status)
5. Fix conflicts manually
```

---

## ✅ Safety Checklist

Before any operation:

- [ ] Backup database (mysqldump)
- [ ] Backup .env file
- [ ] Backup storage/app/public/
- [ ] Save APP_KEY from old .env
- [ ] Save VAPID keys from old .env
- [ ] Test backup can be restored
- [ ] Know how to rollback

**Golden Rule:** If in doubt, backup first!

---

## 🚨 Emergency Recovery

### **If something goes wrong:**

### **Restore Database:**
```bash
mysql -u root -p database_name < backup.sql
```

### **Restore .env:**
```bash
cp .env.backup .env
```

### **Restore Files:**
```bash
tar -xzf uploads-backup.tar.gz -C storage/app/public/
```

### **Clear Everything:**
```bash
php artisan optimize:clear
php artisan config:clear
php artisan cache:clear
```

---

## 📞 Quick Commands Reference

```bash
# SAFE - Check migration status (read-only)
php artisan migrate:status

# SAFE - Run new migrations only
php artisan migrate --force

# SAFE - Backup database
mysqldump -u root -p database_name > backup.sql

# SAFE - Restore database
mysql -u root -p database_name < backup.sql

# SAFE - Check which migrations will run
php artisan migrate --pretend

# DANGEROUS - Don't use these!
# php artisan migrate:fresh    ← DELETES ALL DATA!
# php artisan migrate:refresh  ← DELETES ALL DATA!
# php artisan db:wipe          ← DELETES ALL DATA!
```

---

## ✅ Summary

**Reinstalling with old database is SAFE if you:**

1. ✅ Backup first
2. ✅ Use `php artisan migrate` (not fresh/refresh)
3. ✅ Keep old .env keys (APP_KEY, VAPID)
4. ✅ Don't drop tables manually

**Laravel migrations are designed to be safe:**
- Only runs new migrations
- Skips existing tables
- Preserves data

**Your data will be SAFE!** ✅

---

**Document Version:** 1.0
**Last Updated:** 2025-11-19
**Status:** Production Ready
