# Phase 5: Cloudflare API Integration - Complete

This document details the Cloudflare API integration implementation for automatic DNS management of custom seller domains.

---

## Overview

Phase 5 adds full Cloudflare API integration to automate DNS record creation, verification, and deletion for seller custom domains. This eliminates manual DNS configuration and streamlines the domain setup process.

---

## What Was Implemented

### 1. System Settings Infrastructure ✅

**Files Created/Modified:**
- `database/migrations/2025_10_29_121925_create_system_settings_table.php` - Settings storage
- `app/Models/SystemSetting.php` - Settings model with helper methods

**Features:**
- Flexible key-value storage for system configuration
- Type casting (string, boolean, json, encrypted)
- Caching layer (1 hour TTL)
- Cloudflare-specific helper methods:
  - `getCloudflareApiToken()`
  - `getCloudflareZoneId()`
  - `getCloudflareEmail()`
  - `isCloudflareEnabled()`
  - `getServerIp()`
  - `isAutoDnsEnabled()`
  - `isAutoVerificationEnabled()`

---

### 2. Cloudflare Service ✅

**File:** `app/Services/CloudflareService.php`

**Capabilities:**

#### DNS Record Management
- ✅ `createDnsRecord()` - Create A and CNAME records
- ✅ `updateDnsRecord()` - Update existing records
- ✅ `deleteDnsRecord()` - Remove DNS records
- ✅ `listDnsRecords()` - List all zone DNS records
- ✅ `dnsRecordExists()` - Check if record exists

#### Domain Verification
- ✅ `verifyDnsPropagation()` - Check DNS resolution
- ✅ `testConnection()` - Verify API credentials
- ✅ `getZoneDetails()` - Fetch zone information

#### Features
- Automatic root (@) record creation (A record)
- Automatic www CNAME record creation
- Proxied through Cloudflare (orange cloud) by default
- Comprehensive error handling and logging
- Duplicate record detection

**Example Usage:**
```php
$cloudflare = new CloudflareService();

// Create DNS records
$result = $cloudflare->createDnsRecord('example.com', '1.2.3.4', true);
// Creates:
// - A record: example.com → 1.2.3.4 (proxied)
// - CNAME record: www.example.com → example.com (proxied)

// Verify DNS propagation
$verify = $cloudflare->verifyDnsPropagation('example.com', '1.2.3.4');
if ($verify['verified']) {
    // Domain is verified!
}
```

---

### 3. Admin Settings Interface ✅

**Files Created/Modified:**
- `app/Http/Controllers/CloudflareSettingsController.php` - Settings controller
- `resources/views/cloudflare-settings/index.blade.php` - Settings UI
- `routes/web.php` - Added Cloudflare settings routes
- `resources/views/layouts/app.blade.php` - Added navigation link

**Settings Page Sections:**

#### 1. Cloudflare API Credentials
- **API Token** (Required) - Cloudflare API bearer token
- **Zone ID** (Required) - Cloudflare zone identifier
- **Account Email** (Optional) - Legacy auth email
- **Test Connection** button - Verify credentials

#### 2. Server Configuration
- **Server IP Address** (Required) - Where DNS should point

#### 3. Automation Settings
- **Automatic DNS Creation** - Toggle auto-create DNS records
- **Automatic Verification** - Toggle auto-verify after DNS creation

#### Features
- Status banner showing configuration state
- Inline help text with Cloudflare dashboard links
- Real-time connection testing
- Comprehensive setup instructions
- Link to documentation

**Navigation:** Admin → Management → Cloudflare Settings

---

### 4. Automatic Domain Management ✅

**File Modified:** `app/Http/Controllers/SellerDomainController.php`

#### Enhanced `store()` Method

**When seller adds new domain:**

1. **Check if auto-DNS is enabled**
   - If YES: Create DNS records in Cloudflare automatically
   - Store Cloudflare record IDs for future management
   - Provide success/error feedback

2. **Check if auto-verification is enabled**
   - If YES: Verify DNS propagation immediately
   - Auto-verify if DNS resolves correctly
   - Mark domain as active automatically

3. **Provide comprehensive feedback**
   - Success: "Domain added + DNS created + Verified!"
   - Partial: "Domain added + DNS created, wait for propagation"
   - Manual: "Domain added, configure DNS manually"

**Example Flow:**
```
Seller adds: customdomain.com
        ↓
Auto DNS enabled?
    YES → Create A record (@ → server IP)
    YES → Create CNAME (www → customdomain.com)
    YES → Store record IDs in database
        ↓
Auto verification enabled?
    YES → Check DNS resolution
    YES → Domain resolves? Mark as verified!
        ↓
Redirect to domain page with status message
```

#### Enhanced `verify()` Method

**Manual verification with DNS check:**

1. **If Cloudflare enabled:**
   - Check DNS resolution via `gethostbyname()`
   - Compare resolved IP with configured server IP
   - Verify only if IPs match
   - Provide specific error messages

2. **Fallback:**
   - Manual verification (mark as verified)

**User Experience:**
- Clear success/error messages
- Specific failure reasons (wrong IP, not resolved, etc.)
- Guidance on next steps

#### Enhanced `destroy()` Method

**When seller deletes domain:**

1. **Check for Cloudflare records**
   - Parse stored record IDs
   - Delete root A record
   - Delete www CNAME record
   - Provide deletion feedback

2. **Remove database entry**

3. **Handle errors gracefully**
   - Log failures
   - Notify user if manual cleanup needed

---

## Configuration Guide

### Step 1: Get Cloudflare Credentials

**API Token:**
1. Login to https://dash.cloudflare.com
2. Go to **My Profile** → **API Tokens**
3. Click **Create Token**
4. Use **Edit zone DNS** template
5. Select your zone
6. Copy the generated token

**Zone ID:**
1. Go to your domain in Cloudflare
2. Scroll to right sidebar on Overview page
3. Copy **Zone ID**

### Step 2: Configure in Platform

1. Login as **Admin**
2. Navigate: **Management** → **Cloudflare Settings**
3. Enter credentials:
   - API Token: `your_cloudflare_api_token`
   - Zone ID: `your_zone_id`
   - Server IP: `your_server_ip_address`
4. Enable automation:
   - ☑ Automatic DNS Record Creation
   - ☑ Automatic Domain Verification
5. Click **Test Connection** to verify
6. Click **Save Settings**

### Step 3: Test with Seller Domain

1. Login as **Seller**
2. Go to: **My Stores** → **View My Domains**
3. Click **+ Add Custom Domain**
4. Enter a test domain
5. Click **Add Domain**

**Expected Result:**
```
✅ Custom domain added successfully!
DNS records created automatically in Cloudflare.
Domain verified automatically!
```

---

## How It Works

### Automatic DNS Creation Flow

```
Seller submits domain form
        ↓
SellerDomainController::store()
        ↓
Check: SystemSetting::isAutoDnsEnabled()
        ↓ YES
CloudflareService::createDnsRecord(domain, serverIp)
        ↓
Cloudflare API: POST /zones/{zone_id}/dns_records
        ↓
Create A record (@ → serverIp, proxied=true)
        ↓
Create CNAME (www → domain, proxied=true)
        ↓
Return record IDs
        ↓
Store IDs in seller_domains.cloudflare_record_id
        ↓
Check: SystemSetting::isAutoVerificationEnabled()
        ↓ YES
CloudflareService::verifyDnsPropagation(domain, serverIp)
        ↓
PHP: gethostbyname(domain)
        ↓
Compare resolved IP with serverIp
        ↓ MATCH
SellerDomain::markAsVerified()
        ↓
Redirect with success message
```

### Manual Verification Flow

```
Seller clicks "Verify Domain Now"
        ↓
SellerDomainController::verify()
        ↓
Check: SystemSetting::isCloudflareEnabled()
        ↓ YES
CloudflareService::verifyDnsPropagation(domain, serverIp)
        ↓
Check DNS resolution
        ↓ VERIFIED
SellerDomain::markAsVerified()
        ↓
Success: "DNS pointing to correct server"
        ↓ NOT VERIFIED
Error: "Wrong IP" or "Not resolved"
        ↓
Back with error message
```

### Domain Deletion Flow

```
Seller clicks "Delete Domain"
        ↓
Confirm deletion
        ↓
SellerDomainController::destroy()
        ↓
Check: cloudflare_record_id exists
        ↓ YES
Parse JSON record IDs
        ↓
CloudflareService::deleteDnsRecord(root_id)
        ↓
CloudflareService::deleteDnsRecord(www_id)
        ↓
Delete from database
        ↓
Success: "Domain + DNS records removed"
```

---

## API Endpoints

### Admin Routes (Admin only)

```
GET  /cloudflare-settings           - Settings page
PUT  /cloudflare-settings           - Update settings
POST /cloudflare-settings/test      - Test connection
GET  /cloudflare-settings/zone      - Get zone details (JSON)
GET  /cloudflare-settings/dns-records - List DNS records (JSON)
```

### Seller Routes (Authenticated)

```
POST /seller-domains                - Create domain (auto DNS)
POST /seller-domains/{id}/verify    - Verify domain (DNS check)
DELETE /seller-domains/{id}         - Delete domain (remove DNS)
```

---

## Database Schema

### system_settings Table

```sql
CREATE TABLE `system_settings` (
  `id` bigint unsigned NOT NULL AUTO_INCREMENT,
  `key` varchar(255) NOT NULL,
  `value` text,
  `type` varchar(255) DEFAULT 'string',
  `description` text,
  `created_at` timestamp NULL,
  `updated_at` timestamp NULL,
  PRIMARY KEY (`id`),
  UNIQUE KEY `system_settings_key_unique` (`key`),
  KEY `system_settings_key_index` (`key`)
);
```

**Cloudflare Settings Stored:**
- `cloudflare_api_token` - API bearer token
- `cloudflare_zone_id` - Zone identifier
- `cloudflare_email` - Account email (optional)
- `server_ip` - Server IP address for A records
- `auto_dns_enabled` - Toggle auto DNS creation
- `auto_verification_enabled` - Toggle auto verification

### seller_domains.cloudflare_record_id

**Format:** JSON string
```json
{
  "root": "cloudflare_record_id_for_root",
  "www": "cloudflare_record_id_for_www"
}
```

This allows tracking and deletion of created records.

---

## Error Handling

### Cloudflare API Errors

**Handled Errors:**
- Invalid API token
- Zone not found
- Rate limiting
- Duplicate records (code 81057)
- Network timeouts
- DNS record not found

**Error Logging:**
```php
Log::warning('Automatic DNS creation failed', [
    'domain' => $domain,
    'error' => $result['message']
]);
```

**User Feedback:**
- ✅ Success: Clear success message
- ⚠️ Warning: Partial success with guidance
- ❌ Error: Specific error with troubleshooting steps

### DNS Verification Errors

**Scenarios:**
1. **Domain doesn't resolve**
   - Message: "Domain does not resolve to any IP address"
   - Action: Wait for DNS propagation

2. **Wrong IP**
   - Message: "Domain resolves to X.X.X.X but expected Y.Y.Y.Y"
   - Action: Check DNS configuration

3. **API error**
   - Message: Specific Cloudflare error
   - Action: Check API credentials

---

## Security Considerations

### API Token Security

✅ **Implemented:**
- Tokens stored in database (not .env)
- No token exposure in frontend
- Server-side API calls only
- Logged errors don't include tokens

⚠️ **Recommended:**
- Encrypt `cloudflare_api_token` in database
- Add `type='encrypted'` to SystemSetting
- Use Laravel's encryption helpers

### Rate Limiting

**Cloudflare Limits:**
- API: 1200 requests/5 minutes
- DNS writes: More restrictive

**Our Protection:**
- Single-domain operations (not bulk)
- Error handling for rate limits
- Caching of zone details
- Consider queue for bulk operations

### Input Validation

✅ **Implemented:**
- Domain name sanitization
- IP address validation
- API token format check
- Zone ID format check

---

## Performance Considerations

### Caching Strategy

**SystemSettings:**
- Cache TTL: 1 hour (3600 seconds)
- Cache key: `setting.{key}`
- Cleared on update
```php
Cache::remember("setting.cloudflare_api_token", 3600, function() {
    // Query database
});
```

**Cloudflare API Calls:**
- No caching (real-time accuracy needed)
- Consider caching `getZoneDetails()` for 24 hours

### DNS Propagation

**Timing:**
- Immediate: Cloudflare proxied records (1-2 minutes)
- Standard: Non-proxied (5-30 minutes)
- Maximum: 48 hours globally

**Auto-verification:**
- Waits 2 seconds after DNS creation
- Only verifies if Cloudflare proxied (fast)
- Falls back to manual verification if fails

### Database Queries

**Optimizations:**
- SystemSetting uses caching
- Single query per domain operation
- JSON encoding for record IDs (minimal overhead)

---

## Testing

### Test Checklist

**Setup:**
- [ ] Create Cloudflare account
- [ ] Get API token with DNS edit permission
- [ ] Copy Zone ID
- [ ] Get server IP address
- [ ] Configure in admin panel
- [ ] Test connection (should succeed)

**Auto DNS Creation:**
- [ ] Enable auto DNS in settings
- [ ] Add test domain as seller
- [ ] Check Cloudflare dashboard for A record
- [ ] Check for CNAME (www) record
- [ ] Verify records are proxied (orange cloud)
- [ ] Check `cloudflare_record_id` in database

**Auto Verification:**
- [ ] Enable auto verification
- [ ] Add domain (should verify instantly if Cloudflare)
- [ ] Check status changes to "Active"
- [ ] Verify `verified_at` timestamp set

**Manual Verification:**
- [ ] Disable auto verification
- [ ] Add domain
- [ ] Click "Verify Domain Now"
- [ ] Should verify if DNS resolves correctly
- [ ] Should fail with clear message if not

**Domain Deletion:**
- [ ] Delete domain via UI
- [ ] Check Cloudflare dashboard (records should be gone)
- [ ] Check database (record deleted)
- [ ] Verify no orphaned DNS records

**Error Scenarios:**
- [ ] Invalid API token → Clear error message
- [ ] Wrong Zone ID → Error on test connection
- [ ] Duplicate domain → Proper handling
- [ ] Network timeout → Graceful fallback

---

## Troubleshooting

### Issue: "Cloudflare API not configured"

**Cause:** Missing API token or Zone ID

**Solution:**
1. Go to Admin → Cloudflare Settings
2. Enter valid API token and Zone ID
3. Click "Test Connection"
4. Save settings

---

### Issue: DNS records not created

**Cause:** Auto DNS disabled or missing server IP

**Solution:**
1. Check Admin → Cloudflare Settings
2. Verify "Automatic DNS Creation" is checked
3. Verify Server IP is configured
4. Check API credentials are valid

---

### Issue: Verification fails with "wrong IP"

**Cause:** DNS points to different server

**Solution:**
1. Check DNS at registrar
2. Ensure A record points to configured server IP
3. Wait for DNS propagation (use dnschecker.org)
4. Try verification again

---

### Issue: "Rate limit exceeded"

**Cause:** Too many API requests

**Solution:**
1. Wait 5 minutes
2. Retry operation
3. Consider queueing bulk operations

---

## Future Enhancements

### 1. Bulk Domain Management
- Import multiple domains via CSV
- Queue-based DNS creation
- Batch verification

### 2. DNS Record Customization
- Custom TTL values
- A/AAAA/CNAME options
- Subdomain support

### 3. Enhanced Verification
- TXT record verification
- File-based verification (.well-known)
- Email confirmation

### 4. Multi-Zone Support
- Multiple Cloudflare zones
- Per-seller zone assignment
- Zone selection dropdown

### 5. Analytics & Monitoring
- DNS creation success rate
- Verification time tracking
- API usage monitoring
- Alert on failures

### 6. Automated SSL Management
- Cloudflare Origin Certificate auto-install
- SSL mode auto-configuration
- Certificate expiry monitoring

---

## API Reference

### CloudflareService Methods

#### `createDnsRecord(string $domain, string $ipAddress, bool $proxied = true): array`

Creates A and CNAME records for a domain.

**Parameters:**
- `$domain` - Domain name (e.g., "example.com")
- `$ipAddress` - Server IP address
- `$proxied` - Enable Cloudflare proxy (default: true)

**Returns:**
```php
[
    'success' => true,
    'message' => 'DNS records created successfully',
    'root_record_id' => 'abc123',
    'www_record_id' => 'def456'
]
```

#### `verifyDnsPropagation(string $domain, string $expectedIp): array`

Verifies DNS resolves to expected IP.

**Returns:**
```php
[
    'verified' => true,
    'message' => 'DNS propagation successful',
    'resolved_ip' => '1.2.3.4'
]
```

#### `deleteDnsRecord(string $recordId): array`

Deletes a DNS record by ID.

**Returns:**
```php
[
    'success' => true,
    'message' => 'DNS record deleted successfully'
]
```

#### `testConnection(): array`

Tests Cloudflare API credentials.

**Returns:**
```php
[
    'success' => true,
    'message' => 'Successfully connected to Cloudflare',
    'zone_name' => 'example.com'
]
```

---

## Support & Documentation

**Related Documentation:**
- [CUSTOM_DOMAIN_SETUP.md](./CUSTOM_DOMAIN_SETUP.md) - Complete setup guide
- [SELLER_DOMAIN_GUIDE.md](./SELLER_DOMAIN_GUIDE.md) - Seller instructions
- [ADMIN_SETUP_GUIDE.md](./ADMIN_SETUP_GUIDE.md) - Admin reference

**External Resources:**
- [Cloudflare API Docs](https://developers.cloudflare.com/api/)
- [DNS Management API](https://developers.cloudflare.com/api/operations/dns-records-for-a-zone-create-dns-record)
- [API Authentication](https://developers.cloudflare.com/fundamentals/api/get-started/create-token/)

---

## Changelog

**v1.0.0 - Phase 5 Complete**
- ✅ System settings infrastructure
- ✅ Cloudflare service with full API integration
- ✅ Admin settings interface
- ✅ Automatic DNS record creation
- ✅ Automatic domain verification
- ✅ DNS record deletion on domain removal
- ✅ Comprehensive error handling
- ✅ Admin navigation integration

---

**Implementation Date:** 2025-10-29
**Status:** Complete and ready for testing
**Next Phase:** Testing and production deployment
