# Queue Worker Setup Guide

**Date:** 2025-11-21
**Purpose:** Keep Laravel queue workers running permanently
**Server:** Production & Development

---

## 🎯 Why Queue Workers Are Needed

**Without Queue Worker:**
- Jobs get queued ✓
- But never processed ❌
- Orders don't sync to Google Sheets ❌

**With Queue Worker:**
- Jobs get queued ✓
- Worker processes them immediately ✓
- Orders sync automatically ✓

---

## ⚡ Server Load Impact

### Idle State (No Jobs):
- **CPU Usage:** ~0.1% (nearly zero)
- **Memory:** ~50-80 MB per worker
- **Impact:** Negligible

### Processing Jobs:
- **CPU Usage:** 5-20% (only while processing)
- **Memory:** 80-150 MB per worker
- **Duration:** 1-2 seconds per job

### Summary:
✅ **Very minimal impact!** The worker sleeps when there's no work.

---

## 📊 Performance Numbers

**Typical Usage:**
- **10 orders/hour:** Worker idles 99.9% of the time
- **100 orders/hour:** Worker busy ~3 minutes/hour (95% idle)
- **1000 orders/hour:** Worker busy ~30 minutes/hour (50% idle)

**Conclusion:** For normal business operations, queue workers use almost no resources!

---

## 🔧 Method 1: Supervisor (RECOMMENDED for Production)

### What is Supervisor?

A process monitor that:
- ✅ Keeps workers running 24/7
- ✅ Auto-restarts if crash
- ✅ Auto-starts on server reboot
- ✅ Manages multiple workers
- ✅ Provides logging

### Installation (Linux/Ubuntu)

```bash
sudo apt-get update
sudo apt-get install supervisor -y
```

### Configuration

Create file: `/etc/supervisor/conf.d/laravel-worker.conf`

```ini
[program:laravel-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /var/www/mamadil.team/artisan queue:work redis --queue=default --sleep=3 --tries=3 --max-time=3600 --memory=256
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user=www-data
numprocs=2
redirect_stderr=true
stdout_logfile=/var/www/mamadil.team/storage/logs/worker.log
stopwaitsecs=3600
```

### Explanation of Settings:

| Setting | Value | Purpose |
|---------|-------|---------|
| `command` | php artisan queue:work | The command to run |
| `autostart` | true | Start on server boot |
| `autorestart` | true | Restart if crashes |
| `user` | www-data | Run as web server user |
| `numprocs` | 2 | Run 2 workers (parallel processing) |
| `sleep` | 3 | Sleep 3 seconds when idle |
| `tries` | 3 | Retry failed jobs 3 times |
| `max-time` | 3600 | Restart worker every hour (prevents memory leaks) |
| `memory` | 256 | Max 256MB RAM per worker |

### Start Supervisor

```bash
# Reload configuration
sudo supervisorctl reread
sudo supervisorctl update

# Start workers
sudo supervisorctl start laravel-worker:*

# Check status
sudo supervisorctl status
```

### Managing Workers

```bash
# View status
sudo supervisorctl status laravel-worker:*

# Start all workers
sudo supervisorctl start laravel-worker:*

# Stop all workers
sudo supervisorctl stop laravel-worker:*

# Restart all workers (after code updates)
sudo supervisorctl restart laravel-worker:*

# View logs
tail -f /var/www/mamadil.team/storage/logs/worker.log
```

---

## 🔧 Method 2: Background Process (Simple, Development)

### Windows (Laragon/XAMPP)

```cmd
start /B php artisan queue:work database --sleep=3 --tries=3
```

**To stop:**
```cmd
tasklist | findstr php
taskkill /PID [process_id] /F
```

### Linux/Mac

```bash
nohup php artisan queue:work redis --sleep=3 --tries=3 >> storage/logs/queue.log 2>&1 &
```

**To stop:**
```bash
ps aux | grep "queue:work"
kill [process_id]
```

### Pros & Cons

✅ **Pros:**
- Easy to start
- No extra software needed

❌ **Cons:**
- Stops when terminal closes
- No auto-restart on crash
- No auto-start on reboot
- Manual management

---

## 🔧 Method 3: Cron Job (Not Recommended)

**DON'T use cron to run `queue:work`** - it's for long-running processes.

Instead, use `queue:listen` with cron (but Supervisor is better):

```cron
* * * * * cd /var/www/mamadil.team && php artisan schedule:run >> /dev/null 2>&1
```

---

## 🎯 Recommended Setup by Server Type

### Production Server (Live):
✅ **Use Supervisor**
- Most reliable
- Auto-restarts
- Professional solution

### Development Server (Local):
✅ **Use Background Process** or just run manually when testing
- Simple
- Good enough for dev

### Staging Server:
✅ **Use Supervisor**
- Same as production

---

## 📊 How Many Workers?

### Low Traffic (< 100 orders/day):
```ini
numprocs=1
```
**RAM Usage:** ~80 MB

### Medium Traffic (100-1000 orders/day):
```ini
numprocs=2
```
**RAM Usage:** ~160 MB

### High Traffic (> 1000 orders/day):
```ini
numprocs=4
```
**RAM Usage:** ~320 MB

### Very High Traffic (> 10,000 orders/day):
```ini
numprocs=8-16
```
**RAM Usage:** ~640 MB - 1.2 GB

**Recommendation:** Start with 2 workers, increase if queue gets backed up.

---

## 🔍 Monitoring Queue Health

### Check Queue Status

```bash
# See pending jobs
php artisan queue:monitor redis:default

# See failed jobs
php artisan queue:failed

# Retry failed jobs
php artisan queue:retry all

# Clear failed jobs
php artisan queue:flush
```

### Check Worker Status

```bash
# With Supervisor
sudo supervisorctl status laravel-worker:*

# Without Supervisor (manual check)
ps aux | grep "queue:work"
```

### Check Logs

```bash
# Laravel logs
tail -f storage/logs/laravel.log | grep -i "queue"

# Worker logs (if using Supervisor)
tail -f storage/logs/worker.log
```

---

## ⚠️ Important: After Code Updates

**Always restart workers after updating code:**

```bash
# With Supervisor
sudo supervisorctl restart laravel-worker:*

# Without Supervisor (kill and restart)
killall php
php artisan queue:work redis --sleep=3 --tries=3 &
```

**Why?** Workers load code once on start. If you update code but don't restart workers, they'll still use old code!

---

## 🐛 Troubleshooting

### Workers Not Processing Jobs

**Check if running:**
```bash
ps aux | grep "queue:work"
```

**Start manually:**
```bash
php artisan queue:work database --verbose
```

**Check for errors:**
```bash
tail -50 storage/logs/laravel.log
```

### High Memory Usage

**Add memory limit:**
```bash
php artisan queue:work redis --memory=256
```

**Restart workers hourly:**
```bash
php artisan queue:work redis --max-time=3600
```

### Workers Keep Dying

**Check logs:**
```bash
tail -100 storage/logs/worker.log
```

**Common causes:**
- Memory limit exceeded (increase `--memory`)
- Fatal errors in job code
- Database connection lost
- Google API errors

**Solution:** Fix the code errors, then restart workers.

---

## 📝 Best Practices

### 1. Use Database Queue for Development
```php
// .env
QUEUE_CONNECTION=database
```

### 2. Use Redis Queue for Production
```php
// .env
QUEUE_CONNECTION=redis
```

**Why?** Redis is faster and more reliable for high traffic.

### 3. Set Proper Timeouts
```bash
php artisan queue:work redis --timeout=60 --sleep=3
```

### 4. Log Everything
```bash
redirect_stderr=true
stdout_logfile=/var/www/mamadil.team/storage/logs/worker.log
```

### 5. Monitor Failed Jobs
```bash
# Daily check
php artisan queue:failed
```

### 6. Graceful Restarts
```bash
# Send SIGTERM, not SIGKILL
sudo supervisorctl restart laravel-worker:*
```

---

## 🎯 Your Specific Setup

### For Live Server (mamadil.team):

**Step 1: Install Supervisor**
```bash
sudo apt-get install supervisor -y
```

**Step 2: Create config file**
```bash
sudo nano /etc/supervisor/conf.d/laravel-worker.conf
```

Paste this:
```ini
[program:laravel-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /var/www/mamadil.team/artisan queue:work redis --queue=default --sleep=3 --tries=3 --max-time=3600 --memory=256
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user=www-data
numprocs=2
redirect_stderr=true
stdout_logfile=/var/www/mamadil.team/storage/logs/worker.log
stopwaitsecs=3600
```

**Step 3: Start workers**
```bash
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start laravel-worker:*
```

**Step 4: Verify**
```bash
sudo supervisorctl status
```

You should see:
```
laravel-worker:00    RUNNING   pid 12345, uptime 0:00:05
laravel-worker:01    RUNNING   pid 12346, uptime 0:00:05
```

**Done!** Queue workers will now run forever, even after server reboots.

---

## 🔄 Updating Code Workflow

When you upload new files to live server:

```bash
# 1. Upload files via FTP/Git
git pull

# 2. Clear caches
php artisan cache:clear
php artisan config:clear

# 3. Restart queue workers (IMPORTANT!)
sudo supervisorctl restart laravel-worker:*

# 4. Verify workers restarted
sudo supervisorctl status
```

**This ensures workers use the latest code!**

---

## 📊 Expected Resource Usage

### Your Server (4 sellers, ~500 orders/day):

**With 2 Workers:**
- **CPU:** < 5% average (spikes to 20% during sync)
- **RAM:** ~160 MB (very low)
- **Disk I/O:** Minimal

**Impact:** Almost none! Your server won't even notice.

---

## ✅ Summary

| Aspect | Impact |
|--------|--------|
| **Setup Difficulty** | Easy (15 minutes) |
| **Server Load** | Negligible (< 5% CPU) |
| **RAM Usage** | ~160 MB (2 workers) |
| **Reliability** | 99.9% uptime with Supervisor |
| **Maintenance** | Almost zero |

**Conclusion:** Just use Supervisor! It's the professional solution and uses barely any resources.

---

**Status:** Ready to implement on live server
