# Production Deployment Guide

This guide covers the steps needed when deploying to a Linux production server.

## Table of Contents
- [Queue Management with Laravel Horizon](#queue-management-with-laravel-horizon)
- [Environment Configuration](#environment-configuration)
- [Google Sheets Sync](#google-sheets-sync)
- [Supervisor Setup](#supervisor-setup)

---

## Queue Management with Laravel Horizon

### Why Horizon?

Laravel Horizon provides:
- Beautiful dashboard to monitor queue jobs
- Auto-scaling workers based on load
- Metrics for job throughput and runtime
- Failed job management
- Better performance than basic queue workers

### Installation Steps

#### 1. Install Horizon via Composer

```bash
composer require laravel/horizon
```

**Note:** Horizon requires `ext-pcntl` and `ext-posix` PHP extensions (available on Linux only).

#### 2. Publish Horizon Assets

```bash
php artisan horizon:install
php artisan migrate
```

This creates:
- `config/horizon.php` - Configuration file
- Database tables for job metrics

#### 3. Configure Horizon

Edit `config/horizon.php`:

```php
'environments' => [
    'production' => [
        'supervisor-1' => [
            'connection' => 'redis',
            'queue' => ['default'],
            'balance' => 'auto',
            'maxProcesses' => 10,
            'balanceMaxShift' => 1,
            'balanceCooldown' => 3,
            'tries' => 3,
            'timeout' => 300,
        ],
    ],

    'local' => [
        'supervisor-1' => [
            'connection' => 'redis',
            'queue' => ['default'],
            'balance' => 'simple',
            'maxProcesses' => 3,
            'tries' => 3,
            'timeout' => 300,
        ],
    ],
],
```

#### 4. Update Production .env

Change queue connection from `sync` to `redis`:

```env
# Queue Configuration
QUEUE_CONNECTION=redis

# Redis Configuration (required for Horizon)
REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379
REDIS_CLIENT=phpredis
```

#### 5. Start Horizon

```bash
php artisan horizon
```

**Important:** Horizon must keep running. Use Supervisor to manage it (see below).

---

## Supervisor Setup

Supervisor ensures Horizon stays running and auto-restarts if it crashes.

### 1. Install Supervisor

```bash
sudo apt-get install supervisor
```

### 2. Create Horizon Configuration

Create file: `/etc/supervisor/conf.d/horizon.conf`

```ini
[program:horizon]
process_name=%(program_name)s
command=php /var/www/multistore/artisan horizon
autostart=true
autorestart=true
user=www-data
redirect_stderr=true
stdout_logfile=/var/www/multistore/storage/logs/horizon.log
stopwaitsecs=3600
```

**Adjust paths:**
- `/var/www/multistore` → Your actual project path
- `www-data` → Your web server user

### 3. Start Supervisor

```bash
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start horizon
```

### 4. Check Horizon Status

```bash
sudo supervisorctl status horizon
```

Should show:
```
horizon    RUNNING   pid 12345, uptime 0:01:00
```

### 5. Restart Horizon After Code Deployment

After deploying code updates, restart Horizon:

```bash
php artisan horizon:terminate
```

Supervisor will automatically restart it with the new code.

---

## Environment Configuration

### Development (Windows/Local)

```env
APP_ENV=local
APP_DEBUG=true
QUEUE_CONNECTION=sync
```

- `sync` = Jobs run instantly, no worker needed
- Perfect for local testing on Windows (Horizon not compatible)

### Production (Linux Server)

```env
APP_ENV=production
APP_DEBUG=false
QUEUE_CONNECTION=redis
```

- `redis` = Fast, reliable queue backend
- Requires Redis server installed
- Used with Laravel Horizon

---

## Google Sheets Sync

Google Sheets sync uses Laravel's queue system. Jobs are automatically queued when:
- New order created
- Order updated (status, tracking, etc.)

### Queue Jobs

Located in: `app/Jobs/SyncOrderToGoogleSheets.php`

**Processing:**
- **Local:** Runs instantly with `QUEUE_CONNECTION=sync`
- **Production:** Processed by Horizon with `QUEUE_CONNECTION=redis`

### Google OAuth Setup

Ensure production `.env` has:

```env
GOOGLE_CLIENT_ID=your_production_client_id
GOOGLE_CLIENT_SECRET=your_production_client_secret
GOOGLE_REDIRECT_URI=https://yourdomain.com/auth/google/callback
```

**Important:**
- Add production domain to Google Cloud Console OAuth redirect URIs
- Update from localhost to your actual domain
- Test OAuth flow after deployment

---

## Horizon Dashboard

Access Horizon dashboard at:
```
https://yourdomain.com/horizon
```

### Protect Horizon in Production

Add to `app/Providers/HorizonServiceProvider.php`:

```php
protected function gate()
{
    Gate::define('viewHorizon', function ($user) {
        return in_array($user->email, [
            'admin@yourdomain.com',
        ]) || $user->role === 'admin';
    });
}
```

This restricts access to admins only.

---

## Deployment Checklist

### Pre-Deployment

- [ ] Backup database
- [ ] Test all features locally
- [ ] Ensure .env.production is configured

### During Deployment

```bash
# 1. Pull latest code
git pull origin main

# 2. Install dependencies
composer install --no-dev --optimize-autoloader

# 3. Run migrations
php artisan migrate --force

# 4. Clear caches
php artisan config:clear
php artisan cache:clear
php artisan view:clear
php artisan route:clear

# 5. Optimize for production
php artisan config:cache
php artisan route:cache
php artisan view:cache

# 6. Restart Horizon
php artisan horizon:terminate
```

### Post-Deployment

- [ ] Test queue is processing: Check Horizon dashboard
- [ ] Test Google Sheets sync: Create/update an order
- [ ] Check logs: `storage/logs/laravel.log`
- [ ] Monitor Horizon: `sudo supervisorctl status horizon`

---

## Monitoring & Maintenance

### Check Horizon Logs

```bash
tail -f storage/logs/horizon.log
```

### Check Laravel Logs

```bash
tail -f storage/logs/laravel.log
```

### Check Failed Jobs

```bash
php artisan horizon:failed
```

### Retry Failed Jobs

```bash
php artisan queue:retry all
```

### Clear Failed Jobs

```bash
php artisan horizon:clear
```

---

## Troubleshooting

### Horizon Not Processing Jobs

**Check if running:**
```bash
sudo supervisorctl status horizon
```

**Restart:**
```bash
sudo supervisorctl restart horizon
```

**View logs:**
```bash
tail -f /var/www/multistore/storage/logs/horizon.log
```

### Google Sheets Sync Failing

**Check user connection:**
```bash
php artisan tinker
>>> $user = App\Models\User::find(4);
>>> $user->google_sheets_sync_enabled;
>>> $user->google_sheet_id;
```

**Test manual sync:**
```bash
php artisan tinker
>>> $order = App\Models\Order::first();
>>> $user = App\Models\User::find(4);
>>> $service = new App\Services\GoogleSheetsService();
>>> $service->syncOrder($order, $user);
```

### Redis Connection Failed

**Check Redis is running:**
```bash
redis-cli ping
```

Should return: `PONG`

**Start Redis:**
```bash
sudo service redis-server start
```

---

## Performance Tips

### 1. Use Redis for Cache & Sessions

In production `.env`:
```env
CACHE_STORE=redis
SESSION_DRIVER=redis
QUEUE_CONNECTION=redis
```

### 2. Enable OPcache

Add to `php.ini`:
```ini
opcache.enable=1
opcache.memory_consumption=256
opcache.interned_strings_buffer=16
opcache.max_accelerated_files=10000
```

### 3. Optimize Composer Autoloader

```bash
composer install --optimize-autoloader --no-dev
```

---

## Additional Resources

- [Laravel Horizon Docs](https://laravel.com/docs/11.x/horizon)
- [Supervisor Docs](http://supervisord.org/)
- [Redis Docs](https://redis.io/documentation)
- [Laravel Queue Docs](https://laravel.com/docs/11.x/queues)

---

**Last Updated:** 2025-11-20
