# OpenLiteSpeed Wildcard Domain Setup for Custom Domains

## Prerequisites
- Ubuntu server with OpenLiteSpeed installed
- Root/sudo access
- Domain with Cloudflare DNS (or similar)

## Step 1: Configure Virtual Host

Edit your virtual host config: `/usr/local/lsws/conf/vhosts/multistore/vhconf.conf`

```
docRoot                   /var/www/multistore/public
vhDomain                  multistore.test, *.multistore.test, $VH_NAME
enableGzip                1

context / {
  location                /var/www/multistore/public
  allowBrowse             1

  rewrite {
    enable                1
    autoLoadHtaccess      1
  }
}
```

**Key setting:** `vhDomain` must include `*.yourdomain.com, $VH_NAME`

## Step 2: Configure Listener

Edit listener: `/usr/local/lsws/conf/httpd_config.conf`

Find your listener (usually port 80/443):

```
listener Default {
  address                 *:80
  secure                  0
  map                     multistore.test *
}

listener SSL {
  address                 *:443
  secure                  1
  keyFile                 /path/to/ssl/key.pem
  certFile                /path/to/ssl/cert.pem
  map                     multistore.test *
}
```

**Key setting:** `map` should be `yourdomain.com *` (asterisk catches all subdomains)

## Step 3: DNS Configuration (Cloudflare)

Add these DNS records via Cloudflare API or dashboard:

```
Type: A
Name: @ (root)
Content: YOUR_SERVER_IP
Proxy: Enabled (orange cloud)

Type: A
Name: * (wildcard)
Content: YOUR_SERVER_IP
Proxy: Enabled (orange cloud)
```

**Important:** The wildcard `*` record catches all subdomains.

## Step 4: SSL Certificate (Wildcard)

For wildcard SSL, use Let's Encrypt with DNS challenge:

```bash
sudo apt install certbot
sudo certbot certonly --manual \
  --preferred-challenges=dns \
  --email your@email.com \
  --server https://acme-v02.api.letsencrypt.org/directory \
  --agree-tos \
  -d yourdomain.com \
  -d *.yourdomain.com
```

Follow prompts to add TXT record to Cloudflare DNS.

## Step 5: Laravel Configuration

Update `.env`:

```
APP_URL=https://yourdomain.com
SESSION_DOMAIN=.yourdomain.com
SANCTUM_STATEFUL_DOMAINS=yourdomain.com,*.yourdomain.com
```

## Step 6: Test Setup

1. **Restart OpenLiteSpeed:**
   ```bash
   sudo /usr/local/lsws/bin/lswsctrl restart
   ```

2. **Test DNS:**
   ```bash
   ping test.yourdomain.com
   ```

3. **Test in browser:**
   - Visit `https://seller1.yourdomain.com`
   - Should hit your Laravel app (middleware will route it)

## Step 7: Verify Middleware

Ensure `app/Http/Middleware/DetectCustomDomain.php` exists and is registered in `bootstrap/app.php` or `app/Http/Kernel.php`.

The middleware detects custom domains and sets attributes for routing.

## Troubleshooting

**Issue:** Domain not resolving
- Check DNS propagation: `dig @1.1.1.1 test.yourdomain.com`
- Wait 5-10 minutes for Cloudflare cache

**Issue:** 404 errors
- Check virtual host `docRoot` points to Laravel `public` folder
- Verify `.htaccess` exists in `public` folder
- Check `rewrite { autoLoadHtaccess 1 }` is enabled

**Issue:** SSL errors
- Verify wildcard cert includes both `yourdomain.com` and `*.yourdomain.com`
- Check cert paths in listener config
- Restart OpenLiteSpeed after updating certs

## Quick Reference

```bash
# Restart OpenLiteSpeed
sudo /usr/local/lsws/bin/lswsctrl restart

# Check virtual host config
sudo nano /usr/local/lsws/conf/vhosts/multistore/vhconf.conf

# Check listener config
sudo nano /usr/local/lsws/conf/httpd_config.conf

# Check Laravel logs
tail -f /var/www/multistore/storage/logs/laravel.log

# Check OpenLiteSpeed error log
tail -f /usr/local/lsws/logs/error.log
```

## Architecture Flow

1. User visits `seller1.yourdomain.com`
2. DNS resolves via wildcard `*` record to server IP
3. OpenLiteSpeed listener catches all domains via `map yourdomain.com *`
4. Virtual host serves Laravel app (all domains use same vhost)
5. Laravel middleware `DetectCustomDomain` checks if domain exists in `seller_domains` table
6. If found, routes to seller's sales pages; if not, returns 404

---

**Status:** This guide assumes your app code already has the middleware and routing logic from Phase 5.
