# 📚 Multistore Forecasting System - Documentation

**Complete documentation for mamadil.team forecasting system**

---

## 📋 Table of Contents

1. [System Overview](#system-overview)
2. [Features](#features)
3. [File Structure](#file-structure)
4. [API Routes](#api-routes)
5. [Database Schema](#database-schema)
6. [Configuration](#configuration)

---

## 🎯 System Overview

The Multistore Forecasting System is a comprehensive sales forecasting and business intelligence platform built with Laravel, featuring:

- **Sales Forecasting** with Prophet AI
- **Seller KPI Tracking** with team management
- **Marketing Strategy Analysis** with AI insights
- **Multi-channel Sales** tracking (Webstore + TikTok Shop)
- **Historical Data Management** with import/export

---

## ✨ Features

### 1. Sales Forecasting

**Location:** `/forecasting`

**Features:**
- Historical sales data management
- AI-powered forecasting using Facebook Prophet
- 6-month, 3-month, and 12-month projections
- Confidence intervals and trend analysis
- Import/export via CSV/XLSX
- Auto-calculation from orders table

**Controllers:**
- `ForecastingController.php`

**Models:**
- `HistoricalSale.php`
- `SalesTarget.php`

---

### 2. Seller KPI System

**Location:** `/forecasting/seller-kpi`

**Features:**
- Individual seller performance tracking
- Manager dashboard with team overview
- YTD (Year-to-Date) summaries
- Monthly performance breakdown
- Team member analytics
- Grade calculation (S, A, B, C, D, F)
- Multi-channel sales aggregation

**Controllers:**
- `SellerKpiController.php`

**Models:**
- `SellerKpi.php`
- `User.php`

**Key Routes:**
- `GET /forecasting/seller-kpi` - Seller listing
- `GET /forecasting/seller-kpi/{userId}` - Individual/Manager dashboard
- `POST /forecasting/seller-kpi/sync-sales` - Sync actual sales
- `GET /forecasting/seller-kpi/api/team-ytd/{userId}` - Team YTD API
- `GET /forecasting/seller-kpi/api/team-monthly/{userId}` - Team monthly API
- `GET /forecasting/seller-kpi/api/team-breakdown/{userId}` - Team breakdown API

---

### 3. Marketing Strategy

**Location:** `/marketing-strategy`

**Features:**
- Campaign planning and tracking
- ROI calculation with 80% Malaysian benchmark
- AI-powered recommendations via OpenAI
- Monthly breakdown with campaign notes
- Stokis/Agent channel-specific strategies

**Controllers:**
- `MarketingStrategyController.php`

**Models:**
- `MarketingStrategy.php`

---

### 4. Prophet AI Forecasting

**Technical Stack:**
- Python 3.9+
- Facebook Prophet 1.1.5
- cmdstanpy 1.2.0 (backend)
- NumPy 1.24.3
- Pandas 2.0.3

**Files:**
- `forecast_prophet.py` - Main forecasting script
- `INSTALL_PROPHET.sh` - Installation script
- `SETUP_GUIDE.md` - Setup documentation

**API Endpoint:**
- `GET /forecasting/prophet-forecast?periods=6`

**Returns:**
```json
{
  "success": true,
  "model": "Prophet by Meta",
  "data_points": 24,
  "forecast_periods": 6,
  "trend": "growing",
  "forecasts": [
    {
      "date": "2025-02-01",
      "year": 2025,
      "month": 2,
      "month_name": "February",
      "forecast_revenue": 85000.00,
      "lower_bound": 75000.00,
      "upper_bound": 95000.00,
      "confidence": "high"
    }
  ],
  "summary": {
    "next_month_revenue": 85000.00,
    "next_6_months_total": 500000.00,
    "average_monthly": 83333.33
  }
}
```

---

## 📁 File Structure

```
/var/www/mamadil.team/
├── app/
│   ├── Http/Controllers/
│   │   ├── ForecastingController.php     # Forecasting logic
│   │   ├── SellerKpiController.php       # KPI tracking
│   │   └── MarketingStrategyController.php # Marketing AI
│   ├── Models/
│   │   ├── HistoricalSale.php            # Historical data
│   │   ├── SellerKpi.php                 # KPI records
│   │   ├── MarketingStrategy.php         # Campaigns
│   │   └── User.php                      # Users/Sellers
│   └── ...
├── resources/views/
│   ├── forecasting/
│   │   ├── index.blade.php               # Main forecasting page
│   │   ├── import.blade.php              # Data import
│   │   └── expense-update.blade.php      # Expense tracking
│   ├── seller-kpi/
│   │   ├── index.blade.php               # KPI listing
│   │   └── show.blade.php                # Individual dashboard
│   └── marketing-strategy/
│       ├── index.blade.php               # Campaign listing
│       └── input.blade.php               # Campaign input
├── database/migrations/
│   ├── *_create_historical_sales_table.php
│   ├── *_create_seller_kpis_table.php
│   └── *_create_marketing_strategies_table.php
├── routes/
│   └── web.php                           # All routes
├── venv/                                 # Python virtual environment
│   └── bin/python3.9                     # Prophet runtime
├── forecast_prophet.py                   # Prophet script
├── INSTALL_PROPHET.sh                    # Prophet installer
├── SETUP_GUIDE.md                        # Prophet setup docs
├── DEPLOYMENT.md                         # Deployment guide
└── .env                                  # Environment config
```

---

## 🛣️ API Routes

### Forecasting Routes

```php
// Main forecasting page
GET /forecasting

// Import page
GET /forecasting/import
POST /forecasting/import

// Download template
GET /forecasting/template

// Expense management
GET /forecasting/expenses
POST /forecasting/expenses

// Channel sales update
POST /forecasting/update-channel-sales
POST /forecasting/update-orders

// Prophet AI
GET /forecasting/prophet-forecast?periods={6|3|12}

// AI Insights
GET /forecasting/ai-insights?periods={6|3|12}

// Multi-year trends
GET /forecasting/multi-year-data
POST /forecasting/project-future
```

### Seller KPI Routes

```php
// KPI listing
GET /forecasting/seller-kpi

// Individual dashboard
GET /forecasting/seller-kpi/{userId}

// Sync actual sales
POST /forecasting/seller-kpi/sync-sales

// API endpoints
GET /forecasting/seller-kpi/api/team-ytd/{userId}
GET /forecasting/seller-kpi/api/team-monthly/{userId}?year={2025}
GET /forecasting/seller-kpi/api/team-breakdown/{userId}?year={2025}&month={1}
```

### Marketing Strategy Routes

```php
// Campaign listing
GET /marketing-strategy

// Campaign input
GET /marketing-strategy/input/{year?}

// Save campaign
POST /marketing-strategy/save

// AI recommendations
POST /marketing-strategy/get-suggestions
```

---

## 🗄️ Database Schema

### `historical_sales` Table

```sql
CREATE TABLE `historical_sales` (
  `id` bigint unsigned NOT NULL AUTO_INCREMENT,
  `year` int NOT NULL,
  `month` int NOT NULL,
  `revenue` decimal(15,2) NOT NULL DEFAULT '0.00',
  `orders` int NOT NULL DEFAULT '0',
  `expenses` decimal(15,2) DEFAULT '0.00',
  `stokis_sales` decimal(15,2) DEFAULT '0.00',
  `agent_sales` decimal(15,2) DEFAULT '0.00',
  `hq_sales` decimal(15,2) DEFAULT '0.00',
  `notes` text,
  `import_source` varchar(50) DEFAULT NULL,
  `created_by` bigint unsigned DEFAULT NULL,
  `expense_updated_at` timestamp NULL DEFAULT NULL,
  `expense_updated_by` bigint unsigned DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT NULL,
  `updated_at` timestamp NULL DEFAULT NULL,
  PRIMARY KEY (`id`),
  UNIQUE KEY `year_month_unique` (`year`,`month`)
);
```

### `seller_kpis` Table

```sql
CREATE TABLE `seller_kpis` (
  `id` bigint unsigned NOT NULL AUTO_INCREMENT,
  `user_id` bigint unsigned NOT NULL,
  `year` int NOT NULL,
  `month` int NOT NULL,
  `target` decimal(15,2) NOT NULL DEFAULT '0.00',
  `actual` decimal(15,2) DEFAULT '0.00',
  `achievement_rate` decimal(5,2) DEFAULT '0.00',
  `grade` varchar(10) DEFAULT NULL,
  `notes` text,
  `synced_at` timestamp NULL DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT NULL,
  `updated_at` timestamp NULL DEFAULT NULL,
  PRIMARY KEY (`id`),
  UNIQUE KEY `user_year_month_unique` (`user_id`,`year`,`month`),
  KEY `user_id` (`user_id`)
);
```

### `marketing_strategies` Table

```sql
CREATE TABLE `marketing_strategies` (
  `id` bigint unsigned NOT NULL AUTO_INCREMENT,
  `year` int NOT NULL,
  `month` int NOT NULL,
  `budget` decimal(15,2) NOT NULL DEFAULT '0.00',
  `actual_spend` decimal(15,2) DEFAULT '0.00',
  `revenue_generated` decimal(15,2) DEFAULT '0.00',
  `roi` decimal(10,2) DEFAULT '0.00',
  `campaign_notes` text,
  `ai_suggestions` text,
  `created_at` timestamp NULL DEFAULT NULL,
  `updated_at` timestamp NULL DEFAULT NULL,
  PRIMARY KEY (`id`),
  UNIQUE KEY `year_month_unique` (`year`,`month`)
);
```

---

## ⚙️ Configuration

### Environment Variables

```env
# Application
APP_NAME="Multistore Forecasting"
APP_ENV=production
APP_DEBUG=false
APP_URL=https://mamadil.team

# Database
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=your_database
DB_USERNAME=your_username
DB_PASSWORD=your_password

# Prophet AI
PYTHON_PATH=/var/www/mamadil.team/venv/bin/python3.9
PROPHET_SCRIPT=/var/www/mamadil.team/forecast_prophet.py

# OpenAI API (for AI Insights)
OPENAI_API_KEY=sk-your-api-key-here

# Queue
QUEUE_CONNECTION=database

# Mail
MAIL_MAILER=smtp
MAIL_HOST=smtp.gmail.com
MAIL_PORT=587
MAIL_USERNAME=your-email@gmail.com
MAIL_PASSWORD=your-app-password
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS=noreply@mamadil.team
```

### Config Files

**Prophet Config (`config/prophet.php`):**
```php
<?php

return [
    'python_path' => env('PYTHON_PATH', '/var/www/mamadil.team/venv/bin/python3.9'),
    'script_path' => env('PROPHET_SCRIPT', base_path('forecast_prophet.py')),
];
```

---

## 🔐 User Roles & Permissions

### Role Types

1. **Admin** - Full access
2. **Manager** - Team overview + individual KPIs
3. **Seller** - Individual KPIs only

### Permission Logic

```php
// Check if user is manager
if ($user->role === 'manager' || $user->hasTeamMembers()) {
    // Show team dashboard
}

// Manager sees cumulative sales (direct + team)
$cumulativeSales = $user->direct_sales + $user->team_sales;
```

---

## 📊 Calculation Logic

### Grade Calculation

```php
function calculateGrade($achievementRate) {
    if ($achievementRate >= 120) return 'S';
    if ($achievementRate >= 100) return 'A';
    if ($achievementRate >= 80) return 'B';
    if ($achievementRate >= 60) return 'C';
    if ($achievementRate >= 40) return 'D';
    return 'F';
}
```

### ROI Calculation

```php
$roi = (($revenue - $spend) / $spend) * 100;

// Malaysian benchmark: 80% ROI (1.8x ratio)
$isGoodROI = $roi >= 80;
```

### Achievement Rate

```php
$achievementRate = ($actual / $target) * 100;
```

---

## 🔄 Data Sync Process

### Auto Sales Sync

**Trigger:** Monthly cron job or manual sync

**Process:**
1. Get orders from `orders` table (status: processing, approval, printed, completed)
2. Get TikTok Shop transactions from `tiktok_transactions` table
3. Aggregate by user, year, month
4. Update `seller_kpis.actual`
5. Recalculate achievement rate and grade

**Code:**
```php
// ForecastingController.php
public function syncActualSales($year, $month) {
    // Orders table
    $orders = Order::whereYear('date_created', $year)
        ->whereMonth('date_created', $month)
        ->whereIn('status', ['processing', 'approval', 'printed', 'completed'])
        ->sum('total');

    // TikTok Shop
    $tiktok = TikTokTransaction::whereYear('settled_date', $year)
        ->whereMonth('settled_date', $month)
        ->where('type', 'Order')
        ->sum('total_revenue');

    return $orders + $tiktok;
}
```

---

## 🆘 Troubleshooting

See respective guides:
- **Prophet Issues:** SETUP_GUIDE.md
- **Deployment Issues:** DEPLOYMENT.md
- **Feature-Specific:** FEATURES_REFERENCE.md

---

## 📞 Quick Commands

```bash
# Clear all caches
php artisan optimize:clear

# Sync sales data
php artisan tinker
>>> app(App\Http\Controllers\SellerKpiController::class)->syncActualSales();

# Test Prophet
echo '[{"year":2023,"month":1,"revenue":50000}]' | venv/bin/python3.9 forecast_prophet.py 6

# Check logs
tail -f storage/logs/laravel.log
```

---

**Last Updated:** 2025-01-15
**Version:** 2.0
**For:** mamadil.team
