# Debug Google Sheets Sync - Live Server Commands

**Date:** 2025-11-20
**Purpose:** Troubleshoot why Google Sheets sync jobs complete but don't send data

---

## 🔍 Step 1: Check User's Google Sheets Configuration

Run this on your **live server**:

```bash
php artisan tinker
```

Then copy and paste this into tinker:

```php
$user = User::find(2);
echo "User: " . $user->name . "\n";
echo "Email: " . $user->email . "\n";
echo "Sync Enabled: " . ($user->google_sheets_sync_enabled ? 'YES' : 'NO') . "\n";
echo "Sheet ID: " . ($user->google_sheet_id ?? 'NULL') . "\n";
echo "Google Email: " . ($user->google_email ?? 'NULL') . "\n";
echo "Access Token: " . ($user->google_access_token ? 'EXISTS' : 'NULL') . "\n";
echo "Refresh Token: " . ($user->google_refresh_token ? 'EXISTS' : 'NULL') . "\n";
echo "Token Expires: " . ($user->google_token_expires_at ?? 'NULL') . "\n";
exit
```

**To exit tinker:** Just type `exit` and press Enter.

---

## 📊 Expected Output

### ✅ Good Configuration:
```
User: Team Muslim
Email: mamadilhq@gmail.com
Sync Enabled: YES
Sheet ID: 10BbPmt3xss6dWR6Nc_Y...
Google Email: mamadilhq@gmail.com
Access Token: EXISTS
Refresh Token: EXISTS
Token Expires: 2025-11-20 22:30:00
```

### ❌ Bad Configuration:
```
User: Team Muslim
Email: mamadilhq@gmail.com
Sync Enabled: NO    <-- PROBLEM!
Sheet ID: NULL      <-- PROBLEM!
Google Email: NULL
Access Token: NULL
Refresh Token: NULL
Token Expires: NULL
```

---

## 🔍 Step 2: Check All Sellers

To check all sellers at once:

```bash
php artisan tinker
```

```php
$sellers = User::where('role', 'seller')->get();
foreach ($sellers as $seller) {
    echo "\n=== " . $seller->name . " (ID: " . $seller->id . ") ===\n";
    echo "Sync Enabled: " . ($seller->google_sheets_sync_enabled ? 'YES' : 'NO') . "\n";
    echo "Sheet ID: " . ($seller->google_sheet_id ? 'EXISTS' : 'NULL') . "\n";
    echo "Token Valid: " . ($seller->google_token_expires_at && $seller->google_token_expires_at->isFuture() ? 'YES' : 'NO') . "\n";
}
exit
```

---

## 🔍 Step 3: Test Manual Sync for One Order

```bash
php artisan tinker
```

```php
$user = User::find(2);
$order = Order::where('created_by', 2)->latest()->first();
echo "Testing sync for Order #" . $order->id . " (" . $order->order_number . ")\n";
\App\Jobs\SyncOrderToGoogleSheets::dispatch($user, $order);
echo "Job dispatched! Now run: php artisan queue:work --once\n";
exit
```

Then in **another terminal**, run:
```bash
php artisan queue:work --once
```

---

## 🔍 Step 4: Check Laravel Logs

```bash
tail -50 storage/logs/laravel.log | grep -i "google"
```

Look for these messages:

### ✅ Success Messages:
- `Google Sheets batch sync completed`
- `Order synced to Google Sheets`

### ❌ Problem Messages:
- `Google Sheets sync skipped - not enabled`
- `Failed to sync order to Google Sheets`
- `Google Sheets sync job failed`

---

## 🔍 Step 5: Check Failed Queue Jobs

```bash
php artisan queue:failed
```

If you see failed jobs, retry them:
```bash
php artisan queue:retry all
```

---

## 🛠️ Common Fixes

### Fix 1: Token Expired
**Symptom:** Sync enabled = YES, but token expired

**Solution:**
1. Go to `/settings/google-sheets`
2. Click "Disconnect Google Account"
3. Click "Connect Google Account"
4. Authorize with Google
5. Try sync again

---

### Fix 2: Sync Not Enabled
**Symptom:** Sync enabled = NO

**Solution:**
```bash
php artisan tinker
```

```php
$user = User::find(2);
$user->google_sheets_sync_enabled = true;
$user->save();
echo "Enabled Google Sheets sync for " . $user->name . "\n";
exit
```

---

### Fix 3: Missing Sheet ID
**Symptom:** Sheet ID = NULL

**Solution:** Reconnect Google account (it will create a new sheet)
1. Go to `/settings/google-sheets`
2. Click "Connect Google Account"
3. Authorize with Google

---

## 🎯 Quick Diagnostic Command

Run this to get a full diagnostic:

```bash
php artisan googlesheets:diagnose
```

This shows the status of all sellers' Google Sheets connections.

---

## 🚀 Start Queue Worker Permanently

If jobs are queuing but not processing:

**Option 1: Background Process**
```bash
nohup php artisan queue:work --sleep=3 --tries=3 >> storage/logs/queue.log 2>&1 &
```

**Option 2: Check if Already Running**
```bash
ps aux | grep "queue:work"
```

**Option 3: Kill Existing Queue Worker**
```bash
killall -9 php
```

Then start a new one:
```bash
nohup php artisan queue:work --sleep=3 --tries=3 >> storage/logs/queue.log 2>&1 &
```

---

## 📝 How to Exit Tinker

Just type:
```
exit
```

And press **Enter**.

---

## 🎯 Summary

1. **Run diagnostic:** `php artisan googlesheets:diagnose`
2. **Check user config:** Use tinker to check `google_sheets_sync_enabled` and `google_sheet_id`
3. **Check logs:** `tail -50 storage/logs/laravel.log | grep -i "google"`
4. **Test manual sync:** Dispatch a job and process it with `queue:work --once`
5. **Fix:** Reconnect Google account if needed

---

**Status:** Ready to debug on live server
