# 🔄 Cache Clearing Guide - Omnia PWA

**Problem:** Users seeing old content after uploading new files
**Solution:** Follow this guide to clear all caches

---

## 🎯 What We Fixed

### 1. Service Worker Cache Strategy ✅
**Before:** Cache-first (serves old content)
**After:** Network-first for HTML (always fresh content)

### 2. Cache Version ✅
**Before:** `multistore-v1` (never updated)
**After:** `omnia-v2-2025-11-19` (version bumped)

### 3. Browser Cache Headers ✅
**Before:** No cache control
**After:** Added .htaccess headers to prevent HTML caching

---

## 🚨 How to Clear Cache After Deployment

### Step 1: Clear Server-Side Cache (Laravel)

SSH into your server and run:

```bash
cd /var/www/your-domain

# Clear all Laravel caches
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear

# Rebuild caches (optional, for performance)
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

### Step 2: Force Service Worker Update

The service worker will automatically update because we changed the version:
- **Old:** `multistore-v1`
- **New:** `omnia-v2-2025-11-19`

When users reload the page, the old service worker will be replaced automatically.

### Step 3: Clear Browser Cache (For Testing)

**Chrome/Edge:**
1. Press `F12` to open DevTools
2. Go to **Application** tab
3. Click **Storage** in left sidebar
4. Click **Clear site data** button
5. Check all boxes
6. Click **Clear data**
7. Close DevTools
8. Hard reload: `Ctrl + Shift + R` (Windows) or `Cmd + Shift + R` (Mac)

**Firefox:**
1. Press `F12` to open DevTools
2. Go to **Storage** tab
3. Right-click **Service Workers**
4. Click **Unregister**
5. Right-click **Cache Storage**
6. Click **Delete All**
7. Close DevTools
8. Hard reload: `Ctrl + Shift + R`

**Safari:**
1. Go to Safari menu → **Preferences** → **Advanced**
2. Check "Show Develop menu"
3. Go to **Develop** → **Empty Caches**
4. Hold `Shift` and click reload button

---

## 🔧 For End Users (Simple Instructions)

Tell your users to do a **hard refresh**:

- **Windows:** `Ctrl + Shift + R` or `Ctrl + F5`
- **Mac:** `Cmd + Shift + R`
- **Mobile:** Clear browser cache in settings

---

## 🤖 Automatic Cache Updates

With our fixes, future deployments will automatically update:

1. **Service Worker Auto-Update:**
   - Service worker checks for updates every page load
   - New version number forces cache refresh
   - Old cache is automatically deleted

2. **HTML Always Fresh:**
   - `.htaccess` prevents browser caching of HTML/PHP
   - Service worker uses network-first for HTML
   - Users always get latest content

3. **Static Assets Cached:**
   - CSS, JS, images cached for 1 year
   - Only re-downloaded when filename changes
   - Faster loading for repeat visits

---

## 📝 How to Update Service Worker Version (Future Deployments)

Every time you deploy changes, update the version in `public/service-worker.js`:

```javascript
// Before deployment:
const CACHE_NAME = 'omnia-v2-2025-11-19';

// After deployment (update date or increment version):
const CACHE_NAME = 'omnia-v3-2025-11-20';
// OR
const CACHE_NAME = 'omnia-v2025-11-20';
// OR
const CACHE_NAME = 'omnia-v2.1.0';
```

**Important:** Always change the version string when deploying updates!

---

## 🔍 How to Verify Cache is Cleared

### Check Service Worker Version

1. Open browser DevTools (`F12`)
2. Go to **Application** → **Service Workers**
3. Look for "Source" - should show updated file
4. Check Console for: `[ServiceWorker] Loaded successfully`
5. Version in cache storage should be: `omnia-v2-2025-11-19`

### Check Cache Storage

1. DevTools → **Application** → **Cache Storage**
2. Expand the cache
3. Old cache (`multistore-v1`) should be gone
4. New cache (`omnia-v2-2025-11-19`) should exist

### Check Network Tab

1. DevTools → **Network** tab
2. Check "Disable cache" checkbox (for testing only)
3. Reload page
4. Look at Size column:
   - HTML files: should show size (not "from cache")
   - CSS/JS: can be "from cache" or "from service worker"

---

## ⚠️ Common Issues & Solutions

### Issue 1: Users Still See Old Content

**Cause:** Service worker not updated yet

**Solution:**
```bash
# Tell users to:
1. Hard refresh: Ctrl + Shift + R
2. Or close all browser tabs/windows
3. Reopen browser and visit site
```

### Issue 2: Service Worker Won't Update

**Cause:** Browser cache is holding old service-worker.js

**Solution:**
```bash
# Server-side - add timestamp to service worker registration
# Edit resources/views/layouts/base.blade.php:

<script>
navigator.serviceWorker.register('/service-worker.js?v={{ time() }}')
</script>
```

### Issue 3: Changes Not Visible Immediately

**Cause:** Network-first requires network connection

**Solution:**
- Ensure server is responding with updated files
- Check .htaccess cache headers are working
- Verify file permissions (644 for files)

---

## 🎯 Best Practices for Future Deployments

### 1. Always Update Service Worker Version
```javascript
const CACHE_NAME = 'omnia-v2025-11-20'; // Update this!
```

### 2. Clear Laravel Caches
```bash
php artisan optimize:clear
```

### 3. Version Static Assets
Use Laravel Mix/Vite versioning:
```bash
npm run build
# Generates: app.css?id=abc123
```

### 4. Test Before Production
- Test in incognito/private window
- Check DevTools Console for errors
- Verify service worker updates
- Check cache storage

### 5. Communicate with Users
Send notification:
```
"We've updated the system. Please refresh your browser (Ctrl+Shift+R) to see the latest changes."
```

---

## 📊 Cache Strategy Summary

| Content Type | Strategy | Cache Duration |
|--------------|----------|----------------|
| HTML Pages | Network-first | No cache |
| PHP Files | Network-first | No cache |
| Service Worker | Network-first | No cache |
| CSS Files | Cache-first | 1 year |
| JavaScript | Cache-first | 1 year |
| Images | Cache-first | 1 year |
| Fonts | Cache-first | 1 year |
| Sounds (MP3) | Cache-first | 7 days |
| API Calls | No caching | - |
| Admin Pages | No caching | - |
| Livewire | No caching | - |

---

## 🚀 Quick Commands Reference

```bash
# Clear all caches
php artisan optimize:clear

# Clear specific caches
php artisan cache:clear        # Application cache
php artisan config:clear       # Config cache
php artisan route:clear        # Route cache
php artisan view:clear         # Blade templates

# Rebuild caches (production)
php artisan optimize

# Check cache status
php artisan config:cache --show
php artisan route:list
```

---

## 📞 Emergency Cache Clear (Nuclear Option)

If nothing else works:

```bash
# 1. SSH to server
cd /var/www/your-domain

# 2. Delete all cache files
rm -rf storage/framework/cache/data/*
rm -rf storage/framework/views/*
rm -rf bootstrap/cache/*.php

# 3. Rebuild caches
php artisan optimize

# 4. Restart web server
sudo systemctl restart apache2
# OR for OpenLiteSpeed:
sudo systemctl restart lsws

# 5. Restart queue workers
sudo supervisorctl restart omnia-worker:*
```

---

## ✅ Post-Deployment Checklist

After deploying:

- [ ] Update service worker version number
- [ ] Run `php artisan optimize:clear`
- [ ] Test in incognito window
- [ ] Check DevTools Console (no errors)
- [ ] Verify service worker updated
- [ ] Test PWA notifications
- [ ] Test on different browsers
- [ ] Notify users to hard refresh
- [ ] Monitor error logs

---

## 📚 Additional Resources

- **Service Worker Lifecycle:** https://web.dev/service-worker-lifecycle/
- **Cache Strategies:** https://web.dev/offline-cookbook/
- **PWA Best Practices:** https://web.dev/pwa-checklist/

---

**Document Created:** 2025-11-19
**Last Updated:** 2025-11-19
**Version:** 1.0
