# ✅ Deployment Checklist

**Quick reference checklist for deploying Multistore Management System**

---

## 📦 Pre-Deployment Preparation

### Local Development
- [ ] Test all features locally
- [ ] Run `npm run build` to compile production assets
- [ ] Test PWA notifications work locally
- [ ] Test Prophet AI forecasting works
- [ ] Backup local database (if migrating data)
- [ ] Commit all code changes to Git

### Server Preparation
- [ ] Server provisioned (Ubuntu 20.04+)
- [ ] Domain DNS pointed to server IP
- [ ] SSH access confirmed
- [ ] Root/sudo access available

---

## 🖥️ Server Setup

### System Requirements Installation
- [ ] PHP 8.2+ installed with required extensions
  ```bash
  php -v  # Verify version
  php -m | grep -E "curl|mbstring|mysql|xml|zip|gd|intl|bcmath|gmp"
  ```
- [ ] MySQL/MariaDB installed
  ```bash
  mysql --version
  ```
- [ ] Composer 2.x installed
  ```bash
  composer --version
  ```
- [ ] Node.js 18+ installed
  ```bash
  node --version
  npm --version
  ```
- [ ] Python 3.9+ installed (for Prophet)
  ```bash
  python3.9 --version
  ```
- [ ] Web server installed (OpenLiteSpeed/Apache/Nginx)
- [ ] Supervisor installed (for queue workers)
  ```bash
  sudo apt install supervisor
  ```

---

## 📁 File Upload

### Upload Methods
- [ ] Files uploaded via Git clone OR
- [ ] Files uploaded via SFTP/FTP
- [ ] All files present in `/var/www/mamadil.team/`

### Verify Key Directories Exist
```bash
cd /var/www/mamadil.team
ls -la app config database public resources routes storage
```

---

## 🔧 Installation & Configuration

### PHP Dependencies
- [ ] Run `composer install --optimize-autoloader --no-dev`
- [ ] Verify vendor folder created
- [ ] Check for any composer errors

### Node Dependencies
- [ ] Run `npm install`
- [ ] Run `npm run build`
- [ ] Verify `public/build/` folder created

### Environment File
- [ ] Copy `.env.example` to `.env`
  ```bash
  cp .env.example .env
  ```
- [ ] Configure APP settings (name, env, debug, url)
- [ ] Configure database credentials
- [ ] Generate APP_KEY
  ```bash
  php artisan key:generate
  ```
- [ ] Generate VAPID keys for PWA
  ```bash
  php artisan webpush:vapid
  ```
- [ ] Add VAPID keys to .env
- [ ] Configure Python paths for Prophet
- [ ] Add OpenAI API key (if using)
- [ ] Configure mail settings
- [ ] Set queue connection to `database`

### File Permissions
- [ ] Set ownership to www-data
  ```bash
  sudo chown -R www-data:www-data .
  ```
- [ ] Set directory permissions (755)
  ```bash
  sudo find . -type d -exec chmod 755 {} \;
  ```
- [ ] Set file permissions (644)
  ```bash
  sudo find . -type f -exec chmod 644 {} \;
  ```
- [ ] Set storage writable (775)
  ```bash
  sudo chmod -R 775 storage bootstrap/cache
  ```
- [ ] Make artisan executable
  ```bash
  sudo chmod +x artisan
  ```

---

## 🗄️ Database Setup

### Database Creation
- [ ] Create database
  ```sql
  CREATE DATABASE your_database CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
  ```
- [ ] Create database user
- [ ] Grant privileges
- [ ] Test connection
  ```bash
  mysql -u username -p database_name -e "SELECT 1"
  ```

### Migrations
- [ ] Run migrations
  ```bash
  php artisan migrate --force
  ```
- [ ] Verify 88 migrations completed
  ```bash
  php artisan migrate:status
  ```
- [ ] Check `push_subscriptions` table exists
- [ ] Check `notifications` table exists
- [ ] Seed initial data (if needed)

---

## 🔔 PWA Push Notifications

### VAPID Configuration
- [ ] VAPID_PUBLIC_KEY set in .env
- [ ] VAPID_PRIVATE_KEY set in .env
- [ ] VAPID_SUBJECT set in .env (email format)

### PWA Files
- [ ] `public/service-worker.js` exists
- [ ] `public/sounds/kaching.mp3` uploaded
- [ ] Icons uploaded to `public/icons/`:
  - [ ] icon-72.png
  - [ ] icon-96.png
  - [ ] icon-128.png
  - [ ] icon-144.png
  - [ ] icon-152.png
  - [ ] icon-192.png
  - [ ] icon-384.png
  - [ ] icon-512.png

### PWA Configuration
- [ ] `config/pwa.php` configured for your domain
- [ ] Manifest route works: `/manifest.json`
- [ ] VAPID public key route works: `/api/vapid-public-key`

---

## 🤖 Prophet AI Setup

### Python Environment
- [ ] Python 3.9+ installed
- [ ] Virtual environment created
  ```bash
  python3.9 -m venv venv
  ```
- [ ] venv ownership set
  ```bash
  sudo chown -R $(whoami):$(whoami) venv
  ```

### Prophet Installation
- [ ] Run installation script
  ```bash
  bash INSTALL_PROPHET.sh
  ```
  OR manually:
- [ ] Activate venv
- [ ] Install pip packages (numpy, pandas, cmdstanpy, prophet)
- [ ] Install CmdStan compiler
- [ ] Test Prophet import

### Configuration
- [ ] PYTHON_PATH in .env points to venv Python
- [ ] PROPHET_SCRIPT in .env points to forecast_prophet.py
- [ ] forecast_prophet.py is executable
  ```bash
  chmod +x forecast_prophet.py
  ```

### Testing
- [ ] Test Prophet CLI works
  ```bash
  echo '[...]' | venv/bin/python3.9 forecast_prophet.py 6
  ```

---

## 🌐 Web Server Configuration

### OpenLiteSpeed
- [ ] Virtual host configured
- [ ] Document root set to `/var/www/mamadil.team/public`
- [ ] Rewrite rules enabled
- [ ] PHP LSAPI configured
- [ ] Web server restarted
  ```bash
  sudo systemctl restart lsws
  ```

### SSL Certificate
- [ ] Certbot installed
- [ ] SSL certificate obtained
  ```bash
  sudo certbot --apache -d mamadil.team -d www.mamadil.team
  ```
- [ ] Auto-renewal configured
- [ ] HTTPS redirect enabled

---

## ⚙️ Background Services

### Queue Workers (Supervisor)
- [ ] Supervisor config created
  `/etc/supervisor/conf.d/mamadil-worker.conf`
- [ ] Config loaded
  ```bash
  sudo supervisorctl reread
  sudo supervisorctl update
  ```
- [ ] Workers started
  ```bash
  sudo supervisorctl start mamadil-worker:*
  ```
- [ ] Workers running
  ```bash
  sudo supervisorctl status
  ```

### Cron Jobs
- [ ] Crontab configured for www-data user
- [ ] Laravel scheduler added
  ```cron
  * * * * * cd /var/www/mamadil.team && php artisan schedule:run >> /dev/null 2>&1
  ```

---

## 🧪 Testing & Verification

### Basic Tests
- [ ] Website accessible via HTTPS
  ```bash
  curl -I https://mamadil.team
  ```
- [ ] Login page loads
- [ ] Login works
- [ ] Dashboard loads

### PWA Push Notifications Tests
- [ ] "Enable Notifications" button appears on dashboard
- [ ] Clicking button requests browser permission
- [ ] Permission granted successfully
- [ ] Subscription saved to database
  ```sql
  SELECT COUNT(*) FROM push_subscriptions;
  ```
- [ ] Create test order
- [ ] Push notification received
- [ ] Notification sound plays (kaching.mp3)
- [ ] Notification shows correct format
- [ ] Settings page loads: `/settings`
- [ ] PWA settings section visible
- [ ] Can customize notification templates
- [ ] Preview notification button works
- [ ] Settings save successfully

### Prophet AI Tests
- [ ] Forecasting page loads: `/forecasting`
- [ ] "Forecast (AI)" button visible
- [ ] Click button opens modal
- [ ] Select 6 months
- [ ] Forecast generates (wait 10-30s)
- [ ] Forecast table displays
- [ ] Data looks reasonable
- [ ] No errors in Laravel log

### Feature Tests
- [ ] Orders page works
- [ ] Can create manual order
- [ ] Stores page works
- [ ] Reports generate
- [ ] Sales KPI page works
- [ ] Payment settings work
- [ ] Poslaju settings work
- [ ] WhatsApp settings work

### Log Checks
- [ ] No errors in Laravel log
  ```bash
  tail -100 storage/logs/laravel.log
  ```
- [ ] No errors in queue worker log
  ```bash
  tail -100 storage/logs/worker.log
  ```
- [ ] No errors in web server log

---

## 🔒 Security Checklist

### Application Security
- [ ] APP_DEBUG=false in .env
- [ ] APP_ENV=production in .env
- [ ] Strong database password
- [ ] .env file not publicly accessible
- [ ] storage/ not publicly accessible

### Server Security
- [ ] SSH key-based auth enabled
- [ ] Firewall configured (UFW)
- [ ] Fail2Ban installed (optional)
- [ ] Regular backups scheduled

---

## 📊 Performance Optimization

### Caching
- [ ] Config cached
  ```bash
  php artisan config:cache
  ```
- [ ] Routes cached
  ```bash
  php artisan route:cache
  ```
- [ ] Views cached
  ```bash
  php artisan view:cache
  ```

### Database
- [ ] Indexes created (via migrations)
- [ ] Query performance tested

---

## 🔄 Post-Deployment

### Documentation
- [ ] Admin credentials documented securely
- [ ] VAPID keys backed up securely
- [ ] Database credentials backed up
- [ ] Server access details documented

### Monitoring
- [ ] Setup error monitoring (optional)
- [ ] Setup uptime monitoring (optional)
- [ ] Log rotation configured

### Backups
- [ ] Database backup script created
- [ ] File backup strategy defined
- [ ] Test restore procedure

### Training
- [ ] Admin user trained
- [ ] User documentation provided
- [ ] Support contact established

---

## ✅ Final Sign-Off

- [ ] All checklist items completed
- [ ] All tests passing
- [ ] Admin can access system
- [ ] PWA notifications working
- [ ] Prophet AI forecasting working
- [ ] No critical errors in logs
- [ ] Backup strategy in place
- [ ] Deployment documented
- [ ] Handover complete

---

## 📝 Deployment Notes

**Deployment Date:** _______________

**Deployed By:** _______________

**Server IP:** _______________

**Domain:** _______________

**Database:** _______________

**Issues Encountered:**
-
-
-

**Solutions Applied:**
-
-
-

**Special Notes:**
-
-
-

---

## 📞 Emergency Contacts

**Technical Support:** _______________

**Server Provider:** _______________

**Database Admin:** _______________

**Emergency Procedures:**
1. Check Laravel logs: `tail -f storage/logs/laravel.log`
2. Check queue workers: `sudo supervisorctl status`
3. Restart services: `sudo systemctl restart lsws && sudo supervisorctl restart mamadil-worker:*`
4. Clear caches: `php artisan optimize:clear`

---

**Document Version:** 1.0
**Last Updated:** 2025-11-18
**For:** Multistore Management System v2.0
