# 🚀 Complete Deployment Guide - Multistore Management System

**Last Updated:** 2025-11-18
**Version:** 2.0 (with PWA Push Notifications & Prophet AI)

---

## 📋 Table of Contents

1. [System Overview](#system-overview)
2. [Server Requirements](#server-requirements)
3. [Files to Upload](#files-to-upload)
4. [Installation Steps](#installation-steps)
5. [Environment Configuration](#environment-configuration)
6. [Database Setup](#database-setup)
7. [PWA Push Notifications Setup](#pwa-push-notifications-setup)
8. [Prophet AI Forecasting Setup](#prophet-ai-forecasting-setup)
9. [Post-Deployment Configuration](#post-deployment-configuration)
10. [Testing & Verification](#testing--verification)
11. [Troubleshooting](#troubleshooting)

---

## 🎯 System Overview

### Core Features
- ✅ Multi-store order management
- ✅ WooCommerce integration
- ✅ **PWA Push Notifications** for real-time order alerts
- ✅ **Prophet AI** for sales forecasting
- ✅ Payment gateway integration (BayarCash, FPX, DuitNow)
- ✅ Shipping integration (Poslaju)
- ✅ WhatsApp notifications
- ✅ TikTok Shop integration
- ✅ Sales KPI tracking & analytics

### Technology Stack
- **Backend:** Laravel 12 (PHP 8.2+)
- **Frontend:** Alpine.js, Tailwind CSS, Livewire 3
- **Database:** MySQL 8.0+ / MariaDB 10.6+
- **AI:** Python 3.9+ with Prophet (Meta/Facebook)
- **PWA:** Service Workers, Web Push API
- **Web Server:** OpenLiteSpeed (recommended) / Apache / Nginx

---

## 💻 Server Requirements

### Operating System
- Ubuntu 20.04 LTS or 22.04 LTS (recommended)
- CentOS 7/8 (compatible)
- Debian 10/11 (compatible)

### PHP Requirements
```bash
PHP 8.2 or higher

# Required PHP Extensions:
- php8.2-cli
- php8.2-common
- php8.2-curl
- php8.2-mbstring
- php8.2-mysql
- php8.2-xml
- php8.2-zip
- php8.2-gd
- php8.2-intl
- php8.2-bcmath
- php8.2-gmp (for Web Push VAPID)
```

### Database
- MySQL 8.0+ OR MariaDB 10.6+
- Minimum 512MB allocated memory
- InnoDB storage engine

### Python (For Prophet AI Forecasting)
- Python 3.9 or higher
- Virtual environment (venv)
- Packages: prophet, pandas, numpy, cmdstanpy

### Node.js (For Asset Compilation)
- Node.js 18+
- npm 9+

### Composer
- Composer 2.x

---

## 📦 Files to Upload

### Complete File List

Upload **ALL** files and folders to your server path (e.g., `/var/www/mamadil.team/`):

```
multistore/
├── app/                          # Laravel application code
├── bootstrap/                    # Laravel bootstrap files
├── config/                       # Configuration files
│   ├── pwa.php                  # ⭐ PWA configuration
│   ├── webpush.php              # ⭐ Web Push settings
│   └── ...
├── database/
│   ├── migrations/              # 88 migration files
│   └── seeders/
├── public/
│   ├── icons/                   # ⭐ PWA icons (upload your icons here)
│   ├── sounds/                  # ⭐ Notification sounds
│   │   └── kaching.mp3         # Notification sound file
│   ├── service-worker.js        # ⭐ PWA service worker
│   ├── index.php
│   └── ...
├── resources/
│   └── views/
│       ├── layouts/
│       │   └── base.blade.php  # ⭐ Includes PWA setup
│       └── ...
├── routes/
│   └── web.php                 # ⭐ Includes PWA routes
├── storage/                     # Must be writable (777/775)
├── vendor/                      # Will be generated by composer
├── .env.example                # Copy to .env and configure
├── composer.json               # ⭐ Includes webpush package
├── package.json                # Frontend dependencies
├── forecast_prophet.py          # ⭐ Prophet AI script
├── INSTALL_PROPHET.sh          # Prophet installation script
└── ...
```

### Critical Files for PWA Push Notifications
```
✅ public/service-worker.js
✅ public/sounds/kaching.mp3
✅ config/pwa.php
✅ config/webpush.php
✅ app/Http/Controllers/ManifestController.php
✅ app/Http/Controllers/PushSubscriptionController.php
✅ app/Notifications/OrderCreatedNotification.php
✅ resources/views/components/pwa-notification-button.blade.php
✅ database/migrations/2025_11_18_154436_create_push_subscriptions_table.php
✅ database/migrations/2025_11_18_163041_create_notifications_table.php
```

---

## 🛠️ Installation Steps

### Step 1: Upload Files to Server

**Via SFTP/FTP:**
```bash
Upload all files to: /var/www/mamadil.team/
```

**Via Git (Recommended):**
```bash
cd /var/www/
git clone <your-repo-url> mamadil.team
cd mamadil.team
```

---

### Step 2: Install PHP Dependencies

```bash
cd /var/www/mamadil.team

# Install Composer dependencies
composer install --optimize-autoloader --no-dev

# Expected packages include:
# - laravel/framework: ^12.0
# - laravel-notification-channels/webpush: ^10.2
# - webimpian/bayarcash-php-sdk: ^2.0
# - automattic/woocommerce: ^3.1
# - maatwebsite/excel: ^3.1
# - barryvdh/laravel-dompdf: ^3.1
# - etc.
```

---

### Step 3: Install Node Dependencies & Build Assets

```bash
cd /var/www/mamadil.team

# Install Node packages
npm install

# Build production assets
npm run build

# This compiles Tailwind CSS, Alpine.js, and Vite assets
```

---

### Step 4: Set File Permissions

```bash
cd /var/www/mamadil.team

# Set ownership to web server user
sudo chown -R www-data:www-data .

# Set directory permissions
sudo find . -type d -exec chmod 755 {} \;

# Set file permissions
sudo find . -type f -exec chmod 644 {} \;

# Storage and cache must be writable
sudo chmod -R 775 storage bootstrap/cache

# Make artisan executable
sudo chmod +x artisan
```

---

## ⚙️ Environment Configuration

### Step 5: Configure .env File

```bash
cd /var/www/mamadil.team

# Copy environment file
cp .env.example .env

# Edit .env
nano .env
```

### Complete .env Configuration

```env
# ==============================================
# APP SETTINGS
# ==============================================
APP_NAME="Multistore Manager"
APP_ENV=production
APP_KEY=base64:GENERATE_THIS_WITH_ARTISAN_KEY_GENERATE
APP_DEBUG=false
APP_URL=https://mamadil.team
APP_TIMEZONE=Asia/Kuala_Lumpur

# ==============================================
# DATABASE
# ==============================================
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=your_database_name
DB_USERNAME=your_database_user
DB_PASSWORD=your_database_password

# ==============================================
# PWA PUSH NOTIFICATIONS (IMPORTANT!)
# ==============================================
# VAPID Keys for Web Push (Generate with: php artisan webpush:vapid)
VAPID_PUBLIC_KEY=your_public_key_here
VAPID_PRIVATE_KEY=your_private_key_here
VAPID_SUBJECT=mailto:admin@mamadil.team

# ==============================================
# PROPHET AI FORECASTING
# ==============================================
PYTHON_PATH=/var/www/mamadil.team/venv/bin/python3.9
PROPHET_SCRIPT=/var/www/mamadil.team/forecast_prophet.py
OPENAI_API_KEY=your-openai-api-key-here

# ==============================================
# PAYMENT GATEWAY
# ==============================================
# BayarCash
BAYARCASH_PORTAL_KEY=
BAYARCASH_PAT=
BAYARCASH_SECRET_KEY=
BAYARCASH_ENVIRONMENT=sandbox

# ==============================================
# QUEUE & CACHE
# ==============================================
QUEUE_CONNECTION=database
CACHE_STORE=database
SESSION_DRIVER=database

# ==============================================
# MAIL SETTINGS
# ==============================================
MAIL_MAILER=smtp
MAIL_HOST=your-smtp-host
MAIL_PORT=587
MAIL_USERNAME=your-email
MAIL_PASSWORD=your-password
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS=noreply@mamadil.team
MAIL_FROM_NAME="${APP_NAME}"

# ==============================================
# LOGGING
# ==============================================
LOG_CHANNEL=stack
LOG_DEPRECATIONS_CHANNEL=null
LOG_LEVEL=error
```

### Generate Application Key

```bash
php artisan key:generate
```

---

## 🗄️ Database Setup

### Step 6: Create Database

```bash
# Login to MySQL
mysql -u root -p

# Create database
CREATE DATABASE your_database_name CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

# Create user
CREATE USER 'your_database_user'@'localhost' IDENTIFIED BY 'your_password';

# Grant privileges
GRANT ALL PRIVILEGES ON your_database_name.* TO 'your_database_user'@'localhost';

# Flush privileges
FLUSH PRIVILEGES;

# Exit
EXIT;
```

### Step 7: Run Migrations

```bash
cd /var/www/mamadil.team

# Run all 88 migrations
php artisan migrate --force

# Expected migrations include:
# - Users, stores, orders (core tables)
# - Payment gateway settings
# - Poslaju shipping settings
# - WhatsApp settings
# - System settings
# - ⭐ push_subscriptions (PWA)
# - ⭐ notifications (PWA)
# - Historical sales, forecasts, KPIs
# - TikTok transactions
# - And more...
```

---

## 🔔 PWA Push Notifications Setup

### Step 8: Generate VAPID Keys

**IMPORTANT:** VAPID keys are required for Web Push notifications.

```bash
cd /var/www/mamadil.team

# Generate VAPID keys
php artisan webpush:vapid

# Output will show:
# Public key:  BM... (copy this)
# Private key: Rf... (copy this)

# Add to .env:
nano .env
```

Add these keys to `.env`:
```env
VAPID_PUBLIC_KEY=BMxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
VAPID_PRIVATE_KEY=Rfxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
VAPID_SUBJECT=mailto:admin@mamadil.team
```

### Step 9: Upload PWA Icon Files

Icons are required for PWA installation. Upload icon files to `public/icons/`:

```bash
# Required icon sizes:
public/icons/icon-72.png    (72x72px)
public/icons/icon-96.png    (96x96px)
public/icons/icon-128.png   (128x128px)
public/icons/icon-144.png   (144x144px)
public/icons/icon-152.png   (152x152px)
public/icons/icon-192.png   (192x192px)
public/icons/icon-384.png   (384x384px)
public/icons/icon-512.png   (512x512px)
```

**How to create icons:**
1. Create a square logo (512x512px recommended)
2. Use online tool: https://realfavicongenerator.net/
3. Generate all sizes
4. Upload to `public/icons/`

### Step 10: Configure PWA Settings

Edit `config/pwa.php` to customize for your domain:

```bash
nano config/pwa.php
```

```php
'domains' => [
    'mamadil.team' => [
        'name' => 'Mamadil Order Management',
        'short_name' => 'Mamadil',
        'description' => 'Real-time order notifications',
        'theme_color' => '#4F46E5',
        'background_color' => '#ffffff',
    ],
],
```

### Step 11: Clear Caches

```bash
php artisan config:clear
php artisan cache:clear
php artisan view:clear
php artisan route:clear
```

---

## 🤖 Prophet AI Forecasting Setup

### Step 12: Install Python & Create Virtual Environment

See **SETUP_GUIDE.md** for complete Prophet installation instructions.

**Quick Installation:**

```bash
cd /var/www/mamadil.team

# Install Python 3.9 (if not installed)
sudo apt update
sudo apt install python3.9 python3.9-dev python3.9-venv python3-pip build-essential -y

# Create virtual environment
python3.9 -m venv venv

# Change ownership
sudo chown -R $(whoami):$(whoami) venv

# Run automated installation
bash INSTALL_PROPHET.sh
```

Wait 5-10 minutes for installation to complete.

### Step 13: Test Prophet

```bash
cd /var/www/mamadil.team

# Test Prophet directly
echo '[{"year":2023,"month":1,"revenue":50000},{"year":2023,"month":2,"revenue":55000},{"year":2023,"month":3,"revenue":60000},{"year":2023,"month":4,"revenue":58000},{"year":2023,"month":5,"revenue":62000},{"year":2023,"month":6,"revenue":65000},{"year":2023,"month":7,"revenue":68000},{"year":2023,"month":8,"revenue":70000},{"year":2023,"month":9,"revenue":72000},{"year":2023,"month":10,"revenue":75000},{"year":2023,"month":11,"revenue":78000},{"year":2023,"month":12,"revenue":80000}]' | venv/bin/python3.9 forecast_prophet.py 6

# Should output JSON forecast data
```

---

## 🔧 Post-Deployment Configuration

### Step 14: Setup Queue Workers (Supervisor)

```bash
# Install Supervisor
sudo apt install supervisor -y

# Create worker config
sudo nano /etc/supervisor/conf.d/mamadil-worker.conf
```

Add:
```ini
[program:mamadil-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /var/www/mamadil.team/artisan queue:work --sleep=3 --tries=3 --max-time=3600
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user=www-data
numprocs=2
redirect_stderr=true
stdout_logfile=/var/www/mamadil.team/storage/logs/worker.log
stopwaitsecs=3600
```

```bash
# Start worker
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start mamadil-worker:*

# Check status
sudo supervisorctl status
```

### Step 15: Setup Cron Jobs

```bash
sudo crontab -e -u www-data
```

Add:
```cron
* * * * * cd /var/www/mamadil.team && php artisan schedule:run >> /dev/null 2>&1
```

### Step 16: Configure Web Server

**For OpenLiteSpeed:**

```bash
sudo nano /usr/local/lsws/conf/vhosts/mamadil.team.conf
```

Add:
```
docRoot                   /var/www/mamadil.team/public
enableGzip                1

rewrite  {
  enable                  1
  autoLoadHtaccess        1
}

context / {
  location                /var/www/mamadil.team/public
  allowBrowse             1

  rewrite {
    enable                1
    inherit               1
  }
}
```

```bash
# Restart OpenLiteSpeed
sudo systemctl restart lsws
```

### Step 17: Setup SSL Certificate

```bash
# Install Certbot
sudo apt install certbot python3-certbot-apache -y

# Get certificate
sudo certbot --apache -d mamadil.team -d www.mamadil.team

# Auto-renewal test
sudo certbot renew --dry-run
```

---

## ✅ Testing & Verification

### Step 18: Test Website

```bash
# Test homepage
curl -I https://mamadil.team

# Should return: HTTP/2 200

# Test database connection
php artisan tinker
>>> DB::connection()->getPdo();
>>> exit
```

### Step 19: Test PWA Push Notifications

1. **Navigate to:** https://mamadil.team/login
2. **Login** with your credentials
3. **Dashboard:** Look for "Enable Notifications" button (top right)
4. **Click** "Enable Notifications"
5. **Allow** browser notification permission
6. **Create test order** from Orders page
7. **Verify:** You should receive push notification with sound

**Admin Settings:**
- Go to: https://mamadil.team/settings
- Scroll to "PWA Push Notifications" section
- Customize notification templates, emojis, volume
- Click "Preview Notification" to test
- Save settings

### Step 20: Test Prophet AI Forecasting

1. **Navigate to:** https://mamadil.team/forecasting
2. **Click:** "Forecast (AI)" button
3. **Select:** 6 Months
4. **Wait:** 10-30 seconds for processing
5. **Verify:** Forecast table appears with predictions

---

## 🔍 Troubleshooting

### Issue: 500 Internal Server Error

**Check Laravel logs:**
```bash
tail -100 /var/www/mamadil.team/storage/logs/laravel.log
```

**Fix permissions:**
```bash
sudo chmod -R 775 /var/www/mamadil.team/storage
sudo chown -R www-data:www-data /var/www/mamadil.team/storage
```

---

### Issue: Push Notifications Not Working

**Check 1: VAPID keys configured**
```bash
cat /var/www/mamadil.team/.env | grep VAPID
```

**Check 2: Service Worker registered**
Open browser console (F12) and check for:
```
[PWA] ServiceWorker registration successful
```

**Check 3: HTTPS enabled**
Push notifications require HTTPS (or localhost).

**Check 4: Migrations ran**
```bash
php artisan migrate:status | grep push_subscriptions
```

---

### Issue: Prophet AI Not Working

**Check Python path:**
```bash
/var/www/mamadil.team/venv/bin/python3.9 --version
```

**Check Prophet installed:**
```bash
source /var/www/mamadil.team/venv/bin/activate
python -c "from prophet import Prophet; print('OK')"
deactivate
```

**Check .env:**
```bash
cat .env | grep PYTHON_PATH
cat .env | grep PROPHET_SCRIPT
```

---

### Issue: Queue Not Processing

**Check Supervisor status:**
```bash
sudo supervisorctl status mamadil-worker:*
```

**Restart workers:**
```bash
sudo supervisorctl restart mamadil-worker:*
```

**Check worker logs:**
```bash
tail -f /var/www/mamadil.team/storage/logs/worker.log
```

---

## 🔄 Updating Production

### Code Updates

```bash
cd /var/www/mamadil.team

# Backup database first!
mysqldump -u username -p database_name > backup_$(date +%Y%m%d_%H%M%S).sql

# Pull changes
git pull origin main

# Install dependencies
composer install --optimize-autoloader --no-dev
npm install && npm run build

# Run migrations
php artisan migrate --force

# Clear caches
php artisan config:clear
php artisan cache:clear
php artisan view:clear
php artisan route:clear

# Restart queue workers
sudo supervisorctl restart mamadil-worker:*
```

---

## 📊 Feature Checklist

After deployment, verify these features work:

- [ ] User login/registration
- [ ] Dashboard loads
- [ ] **PWA notifications enabled**
- [ ] **Test order creates notification with sound**
- [ ] **Notification settings page works**
- [ ] Orders management
- [ ] Stores management
- [ ] **Prophet AI forecasting works**
- [ ] Sales KPI tracking
- [ ] Payment gateway integration
- [ ] Poslaju shipping labels
- [ ] WhatsApp notifications
- [ ] TikTok Shop sync
- [ ] Reports generation
- [ ] Settings pages work

---

## 📞 Support & Logs

### Log Locations

```bash
# Laravel application logs
/var/www/mamadil.team/storage/logs/laravel.log

# Queue worker logs
/var/www/mamadil.team/storage/logs/worker.log

# Web server logs (OpenLiteSpeed)
/usr/local/lsws/logs/error.log

# Supervisor logs
/var/log/supervisor/supervisord.log
```

### Quick Commands

```bash
# Clear all caches
php artisan optimize:clear

# Rebuild caches
php artisan optimize

# Check queue status
php artisan queue:work --once

# View scheduled tasks
php artisan schedule:list

# Test VAPID keys
php artisan webpush:vapid --show
```

---

## 🎯 Quick Deployment Summary

```bash
# 1. Upload files
git clone <repo> /var/www/mamadil.team

# 2. Install dependencies
cd /var/www/mamadil.team
composer install --no-dev
npm install && npm run build

# 3. Configure environment
cp .env.example .env
nano .env
php artisan key:generate
php artisan webpush:vapid  # Save keys to .env

# 4. Database
mysql -u root -p -e "CREATE DATABASE your_db"
php artisan migrate --force

# 5. Install Prophet AI
bash INSTALL_PROPHET.sh

# 6. Set permissions
sudo chown -R www-data:www-data .
sudo chmod -R 775 storage bootstrap/cache

# 7. Setup services
sudo supervisorctl start mamadil-worker:*
sudo crontab -e -u www-data  # Add schedule:run

# 8. Upload PWA icons to public/icons/

# 9. Test!
curl -I https://mamadil.team
```

---

**🎉 Deployment Complete!**

Your Multistore Management System with PWA Push Notifications and Prophet AI is now live.

---

**Document Version:** 2.0
**Last Updated:** 2025-11-18
**Maintained By:** Development Team
