# Backup and Disaster Recovery Guide

## Overview

This guide covers backup strategies and disaster recovery for the Multistore system, ensuring minimal data loss and maximum uptime.

---

## 🎯 Backup Strategy Levels

### Level 1: Basic Daily Backups ⭐
**Cost:** Free
**Data Loss Risk:** Up to 24 hours
**Complexity:** Low

### Level 2: Incremental Backups ⭐⭐
**Cost:** Low ($5/month)
**Data Loss Risk:** Up to 1-6 hours
**Complexity:** Medium

### Level 3: Database Replication ⭐⭐⭐
**Cost:** Medium ($10-20/month)
**Data Loss Risk:** Minutes
**Complexity:** Medium-High

### Level 4: High Availability ⭐⭐⭐⭐
**Cost:** High ($100+/month)
**Data Loss Risk:** Near-zero
**Complexity:** High

---

## 🛡️ Current Protection: Google Sheets Sync

**YOU ALREADY HAVE THIS!** ✅

Every order is synced in real-time to Google Sheets, providing:
- ✅ Real-time backup of all orders
- ✅ 99.9% uptime (Google's infrastructure)
- ✅ Point-in-time recovery (Google Sheets version history)
- ✅ Accessible from anywhere
- ✅ Can export and re-import if database fails

**This is your first line of defense!**

---

## 📋 Recommended Implementation

### Phase 1: Immediate (Do This Now)

#### 1. Automated Daily Database Backups

**Windows (Laragon) - Local Development:**

Create: `C:\laragon\www\multistore\backup-database.bat`

```batch
@echo off
set DATE=%date:~-4,4%%date:~-10,2%%date:~-7,2%_%time:~0,2%%time:~3,2%%time:~6,2%
set DATE=%DATE: =0%
set BACKUP_DIR=C:\backups\mysql
set DB_NAME=multistore_manager

REM Create backup directory if not exists
if not exist "%BACKUP_DIR%" mkdir "%BACKUP_DIR%"

REM Create backup
C:\laragon\bin\mysql\mysql-8.0.30-winx64\bin\mysqldump.exe -u root %DB_NAME% > "%BACKUP_DIR%\backup_%DATE%.sql"

REM Keep only last 30 backups
forfiles /p "%BACKUP_DIR%" /m *.sql /d -30 /c "cmd /c del @path" 2>nul

echo Backup completed: %BACKUP_DIR%\backup_%DATE%.sql
```

**Schedule with Windows Task Scheduler:**
1. Open Task Scheduler
2. Create Task → Run daily at 2 AM
3. Action: Run `C:\laragon\www\multistore\backup-database.bat`

---

**Linux (Production Server):**

Create: `/usr/local/bin/backup-database.sh`

```bash
#!/bin/bash

# Configuration
DATE=$(date +%Y%m%d_%H%M%S)
BACKUP_DIR="/var/backups/mysql"
DB_NAME="multistore_manager"
DB_USER="root"
DB_PASS="your_password"
RETENTION_DAYS=30

# Create backup directory
mkdir -p $BACKUP_DIR

# Create backup
mysqldump -u $DB_USER -p$DB_PASS $DB_NAME | gzip > $BACKUP_DIR/backup_$DATE.sql.gz

# Upload to cloud storage (optional - requires rclone)
# rclone copy $BACKUP_DIR gs://your-bucket/backups/

# Delete old backups
find $BACKUP_DIR -name "*.sql.gz" -mtime +$RETENTION_DAYS -delete

echo "Backup completed: $BACKUP_DIR/backup_$DATE.sql.gz"
```

Make executable:
```bash
chmod +x /usr/local/bin/backup-database.sh
```

**Schedule with Cron:**
```bash
# Edit crontab
crontab -e

# Add daily backup at 2 AM
0 2 * * * /usr/local/bin/backup-database.sh >> /var/log/mysql-backup.log 2>&1
```

---

#### 2. Enable MySQL Binary Logs (Point-in-Time Recovery)

**Purpose:** Allows recovery to specific timestamp, not just daily backup.

**Configuration:**

Edit MySQL config:
- **Windows:** `C:\laragon\bin\mysql\mysql-8.0.30-winx64\my.ini`
- **Linux:** `/etc/mysql/my.cnf` or `/etc/mysql/mysql.conf.d/mysqld.cnf`

Add under `[mysqld]`:
```ini
[mysqld]
# Binary logging for point-in-time recovery
log_bin = mysql-bin
expire_logs_days = 7
max_binlog_size = 100M
binlog_format = ROW
```

Restart MySQL:
```bash
# Windows (Laragon)
# Stop/Start MySQL from Laragon

# Linux
sudo systemctl restart mysql
```

**Verify:**
```sql
SHOW VARIABLES LIKE 'log_bin';
-- Should show: log_bin = ON

SHOW BINARY LOGS;
-- Shows list of binary log files
```

---

#### 3. Cloud Backup Storage

**Options:**

**A. Google Drive (Using rclone)**

Install rclone: https://rclone.org/

Configure:
```bash
rclone config
# Follow prompts to add Google Drive

# Test upload
rclone copy /var/backups/mysql gdrive:backups/mysql
```

**B. AWS S3**

```bash
# Install AWS CLI
# Configure credentials
aws configure

# Upload backup
aws s3 cp /var/backups/mysql/ s3://your-bucket/backups/ --recursive
```

**C. Dropbox, OneDrive, etc.**
Similar setup using rclone.

---

### Phase 2: Medium Term (Production Setup)

#### 1. Database Replication (Master-Slave)

**Setup:**

**On Primary Server (Master):**

Edit `my.cnf`:
```ini
[mysqld]
server-id = 1
log_bin = mysql-bin
binlog_do_db = multistore_manager
bind-address = 0.0.0.0
```

Create replication user:
```sql
CREATE USER 'replicator'@'%' IDENTIFIED BY 'strong_password';
GRANT REPLICATION SLAVE ON *.* TO 'replicator'@'%';
FLUSH PRIVILEGES;

SHOW MASTER STATUS;
-- Note the File and Position values
```

**On Replica Server (Slave):**

Edit `my.cnf`:
```ini
[mysqld]
server-id = 2
relay-log = mysql-relay-bin
log_bin = mysql-bin
read_only = 1
```

Restore a backup from master, then:
```sql
CHANGE MASTER TO
  MASTER_HOST='master_ip',
  MASTER_USER='replicator',
  MASTER_PASSWORD='strong_password',
  MASTER_LOG_FILE='mysql-bin.000001',
  MASTER_LOG_POS=12345;

START SLAVE;
SHOW SLAVE STATUS\G
-- Check: Slave_IO_Running: Yes, Slave_SQL_Running: Yes
```

**Update Laravel .env for Read Replicas:**

```env
DB_CONNECTION=mysql

# Write connection (master)
DB_HOST=master_server_ip
DB_DATABASE=multistore_manager
DB_USERNAME=root
DB_PASSWORD=password

# Read connection (replica)
DB_HOST_READ=replica_server_ip
DB_DATABASE_READ=multistore_manager
DB_USERNAME_READ=root
DB_PASSWORD_READ=password
```

Update `config/database.php`:
```php
'mysql' => [
    'driver' => 'mysql',
    'url' => env('DB_URL'),
    'read' => [
        'host' => [env('DB_HOST_READ', env('DB_HOST'))],
    ],
    'write' => [
        'host' => [env('DB_HOST', '127.0.0.1')],
    ],
    // ... other config
],
```

---

#### 2. Monitoring & Alerts

**A. UptimeRobot (Free)**

1. Sign up: https://uptimerobot.com/
2. Add monitors:
   - HTTP(S) monitor for your site
   - Port monitor for MySQL (3306)
3. Configure alerts (email, SMS, Slack)

**B. Laravel Monitoring**

Install Telescope (dev only):
```bash
composer require laravel/telescope --dev
php artisan telescope:install
php artisan migrate
```

Access at: `/telescope`

**C. Server Monitoring**

Linux tools:
```bash
# Install Monit
sudo apt-get install monit

# Configure to restart services if they fail
# Edit /etc/monit/monitrc
```

---

### Phase 3: Advanced (Future High Availability)

#### 1. Load Balancer + Multiple App Servers

```
                 Internet
                    |
              Load Balancer
              (Cloudflare / AWS ELB)
                /       \
               /         \
        App Server 1   App Server 2
        (Laravel)      (Laravel)
              \         /
               \       /
            Database Cluster
         (Primary + Replicas)
```

**Requirements:**
- 2+ app servers (VPS or cloud instances)
- Shared session storage (Redis)
- Shared file storage (NFS, S3, or object storage)
- Database cluster with auto-failover

**Benefits:**
- Zero downtime deployments
- Auto-failover if server fails
- Load distribution

---

#### 2. Database Cluster (MySQL Group Replication)

Multi-master setup where all nodes can accept writes.

**Features:**
- Automatic failover
- No data loss
- Any node can be primary

**Tools:**
- MySQL Group Replication
- Galera Cluster
- Percona XtraDB Cluster

---

#### 3. Container Orchestration (Kubernetes)

For ultimate HA and auto-healing:

```yaml
# Kubernetes deployment example
apiVersion: apps/v1
kind: Deployment
metadata:
  name: multistore-app
spec:
  replicas: 3
  template:
    spec:
      containers:
      - name: app
        image: multistore:latest
        livenessProbe:
          httpGet:
            path: /health
            port: 80
          initialDelaySeconds: 30
          periodSeconds: 10
        readinessProbe:
          httpGet:
            path: /ready
            port: 80
```

**Benefits:**
- Auto-restart failed containers
- Auto-scale based on load
- Zero-downtime deployments
- Self-healing infrastructure

---

## 🔄 Disaster Recovery Procedures

### Scenario 1: Database Corruption

**If You Have Daily Backups:**

1. Stop application
2. Restore latest backup:
```bash
# Restore from backup
gunzip < backup_20251120.sql.gz | mysql -u root -p multistore_manager
```
3. Check Google Sheets for orders created after backup
4. Manually import missing orders from Google Sheets

**If You Have Binary Logs:**

1. Restore latest backup
2. Replay binary logs from backup time to incident time:
```bash
mysqlbinlog mysql-bin.000001 mysql-bin.000002 | mysql -u root -p multistore_manager
```

---

### Scenario 2: Primary Database Server Failure

**If You Have Replica:**

1. Promote replica to master:
```sql
# On replica
STOP SLAVE;
RESET SLAVE ALL;
SET GLOBAL read_only = OFF;
```

2. Update Laravel .env to point to new master:
```env
DB_HOST=replica_server_ip
```

3. Clear config cache:
```bash
php artisan config:clear
```

4. Application continues running!

---

### Scenario 3: Complete Server Failure

**Recovery Steps:**

1. **Provision new server**
2. **Install dependencies** (PHP, MySQL, etc.)
3. **Deploy application code**
4. **Restore database:**
   - From cloud backup
   - Or from replica
5. **Restore files** (uploads, if backed up)
6. **Import missing data from Google Sheets**
7. **Update DNS** to point to new server

**Downtime:** 1-4 hours (depending on preparation)

---

### Scenario 4: Data Loss (Orders Missing)

**Google Sheets to the Rescue!**

1. **Export Google Sheets** to CSV
2. **Create import script:**

```php
// app/Console/Commands/ImportOrdersFromGoogleSheets.php
namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Order;

class ImportOrdersFromGoogleSheets extends Command
{
    protected $signature = 'orders:import-from-sheets {file}';

    public function handle()
    {
        $file = $this->argument('file');
        $csv = array_map('str_getcsv', file($file));

        $header = array_shift($csv);

        foreach ($csv as $row) {
            $data = array_combine($header, $row);

            // Check if order exists
            if (Order::where('order_number', $data['Order Number'])->exists()) {
                continue;
            }

            // Create order from Google Sheets data
            Order::create([
                'order_number' => $data['Order Number'],
                'global_order_id' => $data['Global Order ID'],
                // ... map other fields
            ]);

            $this->info("Imported: {$data['Order Number']}");
        }
    }
}
```

3. **Run import:**
```bash
php artisan orders:import-from-sheets /path/to/exported_sheet.csv
```

---

## 📊 Backup Verification Checklist

**Weekly:**
- [ ] Verify latest backup file exists
- [ ] Check backup file size (not zero)
- [ ] Test backup restore on dev environment

**Monthly:**
- [ ] Full restore test on separate server
- [ ] Verify binary logs are being created
- [ ] Check cloud storage sync status
- [ ] Review replication lag (if using replicas)

**Quarterly:**
- [ ] Disaster recovery drill
- [ ] Review and update recovery procedures
- [ ] Test failover process (if HA setup)

---

## 💰 Cost Comparison

### Small Team (3-4 users) - Recommended

| Component | Cost/Month |
|-----------|------------|
| Daily backups to cloud | $5 |
| Google Sheets (FREE backup!) | $0 |
| Monitoring (UptimeRobot) | $0 |
| **Total** | **$5/month** |

**Data Loss Risk:** Up to 24 hours
**Uptime:** 99%+ (depends on single server)

---

### Medium Setup

| Component | Cost/Month |
|-----------|------------|
| Primary server | $20-50 |
| Replica server | $10-20 |
| Cloud backups | $10 |
| Monitoring | $10 |
| **Total** | **$50-90/month** |

**Data Loss Risk:** Minutes
**Uptime:** 99.5%+

---

### Enterprise Setup

| Component | Cost/Month |
|-----------|------------|
| Load balancer | $20 |
| App servers (2x) | $100 |
| Database cluster (3 nodes) | $150 |
| Cloud storage | $20 |
| Monitoring & logging | $30 |
| **Total** | **$320+/month** |

**Data Loss Risk:** Near-zero
**Uptime:** 99.9%+

---

## 🎯 Quick Start for Your Team

**Implement Today (15 minutes):**

1. ✅ **Enable automated daily backups** (see script above)
2. ✅ **Verify Google Sheets sync is working** (already done!)
3. ✅ **Enable MySQL binary logs** (edit my.cnf)
4. ✅ **Set up UptimeRobot monitoring** (free account)

**Result:**
- Maximum 24-hour data loss
- Orders backed up in Google Sheets in real-time
- Email alerts if site goes down

**This is 80% of the protection for 5% of the cost!**

---

## 📚 Additional Resources

- [MySQL Replication Guide](https://dev.mysql.com/doc/refman/8.0/en/replication.html)
- [Laravel Database Read/Write Connections](https://laravel.com/docs/11.x/database#read-and-write-connections)
- [Rclone Documentation](https://rclone.org/docs/)
- [UptimeRobot](https://uptimerobot.com/)

---

**Last Updated:** 2025-11-20
