<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Order;
use App\Models\Store;
use Carbon\Carbon;

class SampleOrdersSeeder extends Seeder
{
    public function run()
    {
        $this->command->info('🚀 Starting sample order generation...');

        // Get first active store (or create a test store)
        $store = Store::where('is_active', true)->first();

        if (!$store) {
            $this->command->error('❌ No active store found! Please create a store first.');
            return;
        }

        $this->command->info("📦 Using store: {$store->name} (ID: {$store->id})");

        // Sample products
        $products = [
            ['name' => 'Premium Coffee Beans 1kg', 'sku' => 'COFFEE-1KG', 'price' => 45.00],
            ['name' => 'Organic Green Tea 500g', 'sku' => 'TEA-GRN-500', 'price' => 32.00],
            ['name' => 'Chocolate Protein Bar', 'sku' => 'PROTEIN-CHOC', 'price' => 8.50],
            ['name' => 'Himalayan Pink Salt', 'sku' => 'SALT-PINK', 'price' => 12.00],
            ['name' => 'Extra Virgin Olive Oil', 'sku' => 'OIL-OLIVE', 'price' => 28.00],
            ['name' => 'Almond Butter 500g', 'sku' => 'ALMOND-BTR', 'price' => 22.00],
            ['name' => 'Organic Honey 1L', 'sku' => 'HONEY-1L', 'price' => 38.00],
            ['name' => 'Dark Chocolate 85%', 'sku' => 'CHOC-DARK-85', 'price' => 15.00],
            ['name' => 'Coconut Oil 500ml', 'sku' => 'OIL-COCONUT', 'price' => 18.00],
            ['name' => 'Chia Seeds 1kg', 'sku' => 'CHIA-1KG', 'price' => 25.00],
        ];

        // Sample customers
        $customers = [
            ['name' => 'Ahmad bin Ali', 'email' => 'ahmad.ali@example.com', 'phone' => '60123456789'],
            ['name' => 'Siti Nurhaliza', 'email' => 'siti.nur@example.com', 'phone' => '60129876543'],
            ['name' => 'Tan Wei Liang', 'email' => 'tan.wei@example.com', 'phone' => '60135551234'],
            ['name' => 'Priya Sharma', 'email' => 'priya.s@example.com', 'phone' => '60147778888'],
            ['name' => 'Muhammad Rizal', 'email' => 'rizal.m@example.com', 'phone' => '60159993333'],
            ['name' => 'Lim Mei Ling', 'email' => 'lim.mei@example.com', 'phone' => '60162224444'],
            ['name' => 'Raj Kumar', 'email' => 'raj.kumar@example.com', 'phone' => '60173335555'],
            ['name' => 'Fatimah Zahra', 'email' => 'fatimah.z@example.com', 'phone' => '60184446666'],
            ['name' => 'Chen Hui Ming', 'email' => 'chen.hui@example.com', 'phone' => '60195557777'],
            ['name' => 'Nurul Ain', 'email' => 'nurul.ain@example.com', 'phone' => '60101112222'],
        ];

        // Malaysian states
        $states = ['Selangor', 'Kuala Lumpur', 'Johor', 'Penang', 'Perak', 'Melaka', 'Sabah', 'Sarawak'];

        // Order statuses
        $statuses = ['completed', 'completed', 'completed', 'completed', 'processing', 'pending'];

        // Payment methods
        $paymentMethods = [
            ['method' => 'cod', 'title' => 'Cash on Delivery'],
            ['method' => 'online_banking', 'title' => 'Online Banking'],
            ['method' => 'credit_card', 'title' => 'Credit Card'],
            ['method' => 'bayarcash', 'title' => 'BayarCash'],
        ];

        $ordersCreated = 0;
        // Use high order IDs to avoid conflicts
        $startOrderId = 50000;

        // Generate orders for 3 months
        for ($month = 2; $month >= 0; $month--) {
            $monthStart = Carbon::now()->subMonths($month)->startOfMonth();
            $monthEnd = Carbon::now()->subMonths($month)->endOfMonth();

            // Generate 100 orders per month
            $ordersPerMonth = 100;

            $this->command->line("📅 Generating {$ordersPerMonth} orders for " . $monthStart->format('F Y') . "...");

            for ($i = 0; $i < $ordersPerMonth; $i++) {
                // Random date within the month
                $orderDate = Carbon::createFromTimestamp(
                    rand($monthStart->timestamp, $monthEnd->timestamp)
                );

                // Random customer
                $customer = $customers[array_rand($customers)];
                [$firstName, $lastName] = explode(' ', $customer['name'], 2);

                // Random number of items (1-4)
                $itemCount = rand(1, 4);
                $lineItems = [];
                $subtotal = 0;

                for ($j = 0; $j < $itemCount; $j++) {
                    $product = $products[array_rand($products)];
                    $quantity = rand(1, 3);
                    $itemTotal = $product['price'] * $quantity;
                    $subtotal += $itemTotal;

                    $lineItems[] = [
                        'name' => $product['name'],
                        'sku' => $product['sku'],
                        'quantity' => $quantity,
                        'price' => $product['price'],
                        'total' => $itemTotal,
                        'product_id' => rand(1, 100),
                    ];
                }

                // Random shipping
                $shippingTotal = rand(5, 15);
                $total = $subtotal + $shippingTotal;

                // Random status
                $status = $statuses[array_rand($statuses)];

                // Random payment method
                $payment = $paymentMethods[array_rand($paymentMethods)];

                // Random state
                $state = $states[array_rand($states)];

                // Build billing and shipping
                $billing = [
                    'first_name' => $firstName,
                    'last_name' => $lastName,
                    'email' => $customer['email'],
                    'phone' => $customer['phone'],
                    'address_1' => 'No. ' . rand(1, 999) . ', Jalan ' . chr(rand(65, 90)) . chr(rand(65, 90)),
                    'address_2' => 'Taman ' . chr(rand(65, 90)) . chr(rand(65, 90)) . chr(rand(65, 90)),
                    'city' => $state,
                    'state' => $state,
                    'postcode' => rand(10000, 99999),
                    'country' => 'MY',
                ];

                $shipping = $billing;

                // Random notes (30% chance)
                $notes = null;
                if (rand(1, 100) <= 30) {
                    $noteOptions = [
                        'Please deliver before 5pm',
                        'Call before delivery',
                        'Leave at guard house',
                        'Gift wrapping requested',
                        'Transaction ID: ' . strtoupper(uniqid('TXN')),
                    ];
                    $notes = $noteOptions[array_rand($noteOptions)];
                }

                // Create order
                Order::create([
                    'store_id' => $store->id,
                    'woo_order_id' => $startOrderId + $ordersCreated,
                    'order_number' => $startOrderId + $ordersCreated,
                    'status' => $status,
                    'total' => $total,
                    'subtotal' => $subtotal,
                    'shipping_total' => $shippingTotal,
                    'discount_total' => 0,
                    'currency' => 'MYR',
                    'payment_method' => $payment['method'],
                    'payment_method_title' => $payment['title'],
                    'billing' => $billing,
                    'shipping' => $shipping,
                    'line_items' => $lineItems,
                    'notes' => $notes,
                    'shipping_method_title' => 'Poslaju',
                    'date_created' => $orderDate,
                    'date_modified' => $orderDate,
                    'created_at' => $orderDate,
                    'updated_at' => $orderDate,
                ]);

                $ordersCreated++;

                // Show progress every 50 orders
                if ($ordersCreated % 50 == 0) {
                    $this->command->info("  ✓ {$ordersCreated} orders created...");
                }
            }

            $this->command->info("  ✓ Completed " . $monthStart->format('F Y'));
        }

        $this->command->newLine();
        $this->command->info("✅ Successfully created {$ordersCreated} sample orders!");
        $this->command->info("📊 Orders span from " . Carbon::now()->subMonths(2)->format('M Y') . " to " . Carbon::now()->format('M Y'));
    }
}
