<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\MalaysiaPostcode;
use Illuminate\Support\Facades\File;

class MalaysiaPostcodeSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $jsonPath = database_path('seeders/malaysia_postcodes.json');

        if (!File::exists($jsonPath)) {
            $this->command->error('malaysia_postcodes.json file not found!');
            $this->command->info('Creating postcode data from state ranges...');
            $this->seedFromStateRanges();
            return;
        }

        $postcodes = json_decode(File::get($jsonPath), true);

        $this->command->info('Seeding Malaysia postcodes...');

        foreach ($postcodes as $postcode) {
            MalaysiaPostcode::updateOrCreate(
                ['postcode' => $postcode['postcode']],
                [
                    'city' => $postcode['city'],
                    'state' => $postcode['state'],
                    'state_code' => $postcode['state_code']
                ]
            );
        }

        $this->command->info('Malaysia postcodes seeded successfully!');
    }

    /**
     * Seed from known postcode ranges for each state
     */
    protected function seedFromStateRanges()
    {
        $ranges = [
            // Perlis
            ['start' => '01000', 'end' => '02800', 'state' => 'Perlis', 'state_code' => 'PLS', 'city' => 'Perlis'],

            // Kedah
            ['start' => '05000', 'end' => '09810', 'state' => 'Kedah', 'state_code' => 'KDH', 'city' => 'Kedah'],

            // Pulau Pinang
            ['start' => '10000', 'end' => '14400', 'state' => 'Pulau Pinang', 'state_code' => 'PNG', 'city' => 'Pulau Pinang'],

            // Perak
            ['start' => '30000', 'end' => '36810', 'state' => 'Perak', 'state_code' => 'PRK', 'city' => 'Perak'],

            // Selangor
            ['start' => '40000', 'end' => '48970', 'state' => 'Selangor', 'state_code' => 'SGR', 'city' => 'Selangor'],

            // Kuala Lumpur
            ['start' => '50000', 'end' => '60000', 'state' => 'WP Kuala Lumpur', 'state_code' => 'KUL', 'city' => 'Kuala Lumpur'],

            // Putrajaya
            ['start' => '62000', 'end' => '62988', 'state' => 'WP Putrajaya', 'state_code' => 'PJY', 'city' => 'Putrajaya'],

            // Negeri Sembilan
            ['start' => '70000', 'end' => '73509', 'state' => 'Negeri Sembilan', 'state_code' => 'NSN', 'city' => 'Negeri Sembilan'],

            // Melaka
            ['start' => '75000', 'end' => '78309', 'state' => 'Melaka', 'state_code' => 'MLK', 'city' => 'Melaka'],

            // Johor
            ['start' => '79000', 'end' => '86900', 'state' => 'Johor', 'state_code' => 'JHR', 'city' => 'Johor'],

            // Pahang
            ['start' => '25000', 'end' => '28800', 'state' => 'Pahang', 'state_code' => 'PHG', 'city' => 'Pahang'],
            ['start' => '39000', 'end' => '39200', 'state' => 'Pahang', 'state_code' => 'PHG', 'city' => 'Pahang'],
            ['start' => '49000', 'end' => '49400', 'state' => 'Pahang', 'state_code' => 'PHG', 'city' => 'Pahang'],
            ['start' => '69000', 'end' => '69500', 'state' => 'Pahang', 'state_code' => 'PHG', 'city' => 'Pahang'],

            // Terengganu
            ['start' => '20000', 'end' => '24300', 'state' => 'Terengganu', 'state_code' => 'TRG', 'city' => 'Terengganu'],

            // Kelantan
            ['start' => '15000', 'end' => '19800', 'state' => 'Kelantan', 'state_code' => 'KTN', 'city' => 'Kelantan'],

            // Sarawak
            ['start' => '93000', 'end' => '98859', 'state' => 'Sarawak', 'state_code' => 'SWK', 'city' => 'Sarawak'],

            // Sabah
            ['start' => '87000', 'end' => '91309', 'state' => 'Sabah', 'state_code' => 'SBH', 'city' => 'Sabah'],

            // Labuan
            ['start' => '87000', 'end' => '87033', 'state' => 'WP Labuan', 'state_code' => 'LBN', 'city' => 'Labuan'],
        ];

        $this->command->info('Generating postcodes from state ranges...');
        $bar = $this->command->getOutput()->createProgressBar(count($ranges));

        foreach ($ranges as $range) {
            $start = intval($range['start']);
            $end = intval($range['end']);

            // Generate sample postcodes for the range (not all, but major ones)
            // We'll create entries for xxx00, xxx50 patterns to keep database smaller
            for ($i = $start; $i <= $end; $i += 50) {
                $postcode = str_pad($i, 5, '0', STR_PAD_LEFT);

                MalaysiaPostcode::updateOrCreate(
                    ['postcode' => $postcode],
                    [
                        'city' => $range['city'],
                        'state' => $range['state'],
                        'state_code' => $range['state_code']
                    ]
                );
            }

            // Also add the exact start and end
            MalaysiaPostcode::updateOrCreate(
                ['postcode' => $range['start']],
                [
                    'city' => $range['city'],
                    'state' => $range['state'],
                    'state_code' => $range['state_code']
                ]
            );

            MalaysiaPostcode::updateOrCreate(
                ['postcode' => $range['end']],
                [
                    'city' => $range['city'],
                    'state' => $range['state'],
                    'state_code' => $range['state_code']
                ]
            );

            $bar->advance();
        }

        $bar->finish();
        $this->command->newLine();
        $this->command->info('Postcode ranges generated successfully!');
    }
}
