<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\MalaysiaPostcode;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;

class CompletePostcodeSeeder extends Seeder
{
    /**
     * State code mapping
     */
    protected $stateCodes = [
        'Wp Kuala Lumpur' => 'KUL',
        'Wp Labuan' => 'LBN',
        'Wp Putrajaya' => 'PJY',
        'Johor' => 'JHR',
        'Kedah' => 'KDH',
        'Kelantan' => 'KTN',
        'Melaka' => 'MLK',
        'Negeri Sembilan' => 'NSN',
        'Pahang' => 'PHG',
        'Perak' => 'PRK',
        'Perlis' => 'PLS',
        'Pulau Pinang' => 'PNG',
        'Sabah' => 'SBH',
        'Sarawak' => 'SWK',
        'Selangor' => 'SGR',
        'Terengganu' => 'TRG',
    ];

    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $jsonPath = database_path('seeders/malaysia_postcodes_complete.json');

        if (!File::exists($jsonPath)) {
            $this->command->error('malaysia_postcodes_complete.json file not found!');
            $this->command->info('Please download it from: https://raw.githubusercontent.com/AsyrafHussin/malaysia-postcodes/main/all.json');
            return;
        }

        $this->command->info('Loading complete Malaysia postcode database...');
        $data = json_decode(File::get($jsonPath), true);

        if (!isset($data['state'])) {
            $this->command->error('Invalid JSON structure!');
            return;
        }

        $this->command->info('Clearing existing postcodes...');
        DB::table('malaysia_postcodes')->delete();

        $totalPostcodes = 0;

        $this->command->info('Processing postcodes...');
        $bar = $this->command->getOutput()->createProgressBar(count($data['state']));

        foreach ($data['state'] as $state) {
            $stateName = $state['name'];
            $stateCode = $this->stateCodes[$stateName] ?? 'UNK';

            foreach ($state['city'] as $city) {
                $cityName = $city['name'];

                foreach ($city['postcode'] as $postcode) {
                    // Use updateOrCreate to avoid duplicates
                    MalaysiaPostcode::updateOrCreate(
                        ['postcode' => $postcode],
                        [
                            'city' => $cityName,
                            'state' => $stateName,
                            'state_code' => $stateCode,
                        ]
                    );

                    $totalPostcodes++;
                }
            }

            $bar->advance();
        }

        $bar->finish();
        $this->command->newLine();
        $this->command->info("✓ Successfully seeded {$totalPostcodes} postcodes!");

        // Show summary by state
        $this->command->newLine();
        $this->command->info('Postcodes by State:');

        $summary = MalaysiaPostcode::select('state', DB::raw('COUNT(*) as count'))
            ->groupBy('state')
            ->orderBy('state')
            ->get();

        foreach ($summary as $item) {
            $this->command->line("  {$item->state}: {$item->count} postcodes");
        }
    }
}
