<?php

use App\Models\TikTokShop;
use App\Models\User;
use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Step 1: Create default TikTok Shop for existing data
        $defaultShop = TikTokShop::create([
            'name' => 'Main TikTok Shop',
            'description' => 'Default shop for existing transactions (migrated data)',
            'is_active' => true,
        ]);

        echo "✓ Created default TikTok Shop (ID: {$defaultShop->id})\n";

        // Step 2: Assign all existing tiktok_transactions to default shop
        $transactionsUpdated = DB::table('tiktok_transactions')
            ->whereNull('tiktok_shop_id')
            ->update(['tiktok_shop_id' => $defaultShop->id]);

        echo "✓ Migrated {$transactionsUpdated} TikTok transactions to default shop\n";

        // Step 3: Assign all existing tiktok_ads_transactions to default shop
        $adsUpdated = DB::table('tiktok_ads_transactions')
            ->whereNull('tiktok_shop_id')
            ->update(['tiktok_shop_id' => $defaultShop->id]);

        echo "✓ Migrated {$adsUpdated} TikTok ads transactions to default shop\n";

        // Step 4: Assign all managers with TikTok permission to default shop
        $managers = User::where('role', 'manager')
            ->where('can_manage_tiktok_shop', 1)
            ->get();

        foreach ($managers as $manager) {
            $manager->tiktokShops()->attach($defaultShop->id);
            echo "✓ Assigned {$manager->name} to default TikTok Shop\n";
        }

        echo "\n✓ Data migration completed successfully!\n";
        echo "  - Default TikTok Shop created\n";
        echo "  - {$transactionsUpdated} transactions migrated\n";
        echo "  - {$adsUpdated} ads transactions migrated\n";
        echo "  - {$managers->count()} managers assigned\n";
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Remove default shop assignment (set back to NULL)
        DB::table('tiktok_transactions')
            ->where('tiktok_shop_id', function($query) {
                $query->select('id')
                    ->from('tiktok_shops')
                    ->where('name', 'Main TikTok Shop')
                    ->limit(1);
            })
            ->update(['tiktok_shop_id' => null]);

        DB::table('tiktok_ads_transactions')
            ->where('tiktok_shop_id', function($query) {
                $query->select('id')
                    ->from('tiktok_shops')
                    ->where('name', 'Main TikTok Shop')
                    ->limit(1);
            })
            ->update(['tiktok_shop_id' => null]);

        // Delete the default shop (will cascade delete pivot entries)
        TikTokShop::where('name', 'Main TikTok Shop')->delete();

        echo "✓ Rolled back data migration\n";
    }
};
