<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations - Add performance indexes for MariaDB/MySQL
     */
    public function up(): void
    {
        // Optimize users table
        Schema::table('users', function (Blueprint $table) {
            if (!$this->indexExists('users', 'idx_users_role_email')) {
                $table->index(['role', 'email'], 'idx_users_role_email');
            }
            if (!$this->indexExists('users', 'idx_users_url_slug')) {
                $table->index('url_slug', 'idx_users_url_slug');
            }
        });

        // Optimize store_user pivot table
        if (Schema::hasTable('store_user')) {
            Schema::table('store_user', function (Blueprint $table) {
                if (!$this->indexExists('store_user', 'idx_store_user_user')) {
                    $table->index('user_id', 'idx_store_user_user');
                }
                if (!$this->indexExists('store_user', 'idx_store_user_store')) {
                    $table->index('store_id', 'idx_store_user_store');
                }
            });
        }

        // Optimize manager_seller pivot table
        if (Schema::hasTable('manager_seller')) {
            Schema::table('manager_seller', function (Blueprint $table) {
                if (!$this->indexExists('manager_seller', 'idx_manager_seller_manager')) {
                    $table->index('manager_id', 'idx_manager_seller_manager');
                }
                if (!$this->indexExists('manager_seller', 'idx_manager_seller_seller')) {
                    $table->index('seller_id', 'idx_manager_seller_seller');
                }
            });
        }

        // Optimize orders table - additional indexes
        Schema::table('orders', function (Blueprint $table) {
            if (!$this->indexExists('orders', 'idx_orders_created_by')) {
                $table->index('created_by', 'idx_orders_created_by');
            }
            if (!$this->indexExists('orders', 'idx_orders_is_manual')) {
                $table->index('is_manual', 'idx_orders_is_manual');
            }
            if (!$this->indexExists('orders', 'idx_orders_webhook_source')) {
                $table->index('webhook_source_id', 'idx_orders_webhook_source');
            }
            if (!$this->indexExists('orders', 'idx_orders_customer')) {
                $table->index('customer_id', 'idx_orders_customer');
            }
            if (!$this->indexExists('orders', 'idx_orders_tracking')) {
                $table->index(['tracking_number', 'courier'], 'idx_orders_tracking');
            }
        });

        // Optimize sales_pages table
        if (Schema::hasTable('sales_pages')) {
            Schema::table('sales_pages', function (Blueprint $table) {
                if (!$this->indexExists('sales_pages', 'idx_sales_pages_user_active')) {
                    $table->index(['user_id', 'is_active'], 'idx_sales_pages_user_active');
                }
                if (!$this->indexExists('sales_pages', 'idx_sales_pages_slug')) {
                    $table->index('slug', 'idx_sales_pages_slug');
                }
            });
        }

        // Optimize sales_page_orders table
        if (Schema::hasTable('sales_page_orders')) {
            Schema::table('sales_page_orders', function (Blueprint $table) {
                if (!$this->indexExists('sales_page_orders', 'idx_sp_orders_page_created')) {
                    $table->index(['sales_page_id', 'created_at'], 'idx_sp_orders_page_created');
                }
                if (!$this->indexExists('sales_page_orders', 'idx_sp_orders_status')) {
                    $table->index('status', 'idx_sp_orders_status');
                }
                if (!$this->indexExists('sales_page_orders', 'idx_sp_orders_phone')) {
                    $table->index('customer_phone', 'idx_sp_orders_phone');
                }
            });
        }

        // Optimize products table
        if (Schema::hasTable('products')) {
            Schema::table('products', function (Blueprint $table) {
                if (!$this->indexExists('products', 'idx_products_user')) {
                    $table->index('user_id', 'idx_products_user');
                }
                if (!$this->indexExists('products', 'idx_products_sku')) {
                    $table->index('sku', 'idx_products_sku');
                }
            });
        }

        // Optimize user_product pivot table
        if (Schema::hasTable('user_product')) {
            Schema::table('user_product', function (Blueprint $table) {
                if (!$this->indexExists('user_product', 'idx_user_product_user')) {
                    $table->index('user_id', 'idx_user_product_user');
                }
                if (!$this->indexExists('user_product', 'idx_user_product_product')) {
                    $table->index('product_id', 'idx_user_product_product');
                }
                if (!$this->indexExists('user_product', 'idx_user_product_enabled')) {
                    $table->index('is_enabled', 'idx_user_product_enabled');
                }
            });
        }

        // Optimize customers table
        if (Schema::hasTable('customers')) {
            Schema::table('customers', function (Blueprint $table) {
                if (!$this->indexExists('customers', 'idx_customers_phone')) {
                    $table->index('phone', 'idx_customers_phone');
                }
                if (!$this->indexExists('customers', 'idx_customers_email')) {
                    $table->index('email', 'idx_customers_email');
                }
            });
        }

        // Optimize webhook_sources table
        if (Schema::hasTable('webhook_sources')) {
            Schema::table('webhook_sources', function (Blueprint $table) {
                if (!$this->indexExists('webhook_sources', 'idx_webhook_sources_active')) {
                    $table->index(['is_active', 'type'], 'idx_webhook_sources_active');
                }
                if (!$this->indexExists('webhook_sources', 'idx_webhook_sources_token')) {
                    $table->index('webhook_token', 'idx_webhook_sources_token');
                }
            });
        }

        // Optimize whatsapp_settings table
        if (Schema::hasTable('whatsapp_settings')) {
            Schema::table('whatsapp_settings', function (Blueprint $table) {
                if (!$this->indexExists('whatsapp_settings', 'idx_whatsapp_settings_user_active')) {
                    $table->index(['user_id', 'is_active'], 'idx_whatsapp_settings_user_active');
                }
            });
        }

        // Optimize seller_domains table
        if (Schema::hasTable('seller_domains')) {
            Schema::table('seller_domains', function (Blueprint $table) {
                if (!$this->indexExists('seller_domains', 'idx_seller_domains_domain')) {
                    $table->index('domain', 'idx_seller_domains_domain');
                }
                if (!$this->indexExists('seller_domains', 'idx_seller_domains_user_active')) {
                    $table->index(['user_id', 'is_active'], 'idx_seller_domains_user_active');
                }
            });
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Drop all indexes created in up()
        Schema::table('users', function (Blueprint $table) {
            $this->dropIndexIfExists($table, 'idx_users_role_email');
            $this->dropIndexIfExists($table, 'idx_users_url_slug');
        });

        if (Schema::hasTable('store_user')) {
            Schema::table('store_user', function (Blueprint $table) {
                $this->dropIndexIfExists($table, 'idx_store_user_user');
                $this->dropIndexIfExists($table, 'idx_store_user_store');
            });
        }

        if (Schema::hasTable('manager_seller')) {
            Schema::table('manager_seller', function (Blueprint $table) {
                $this->dropIndexIfExists($table, 'idx_manager_seller_manager');
                $this->dropIndexIfExists($table, 'idx_manager_seller_seller');
            });
        }

        Schema::table('orders', function (Blueprint $table) {
            $this->dropIndexIfExists($table, 'idx_orders_created_by');
            $this->dropIndexIfExists($table, 'idx_orders_is_manual');
            $this->dropIndexIfExists($table, 'idx_orders_webhook_source');
            $this->dropIndexIfExists($table, 'idx_orders_customer');
            $this->dropIndexIfExists($table, 'idx_orders_tracking');
        });

        if (Schema::hasTable('sales_pages')) {
            Schema::table('sales_pages', function (Blueprint $table) {
                $this->dropIndexIfExists($table, 'idx_sales_pages_user_active');
                $this->dropIndexIfExists($table, 'idx_sales_pages_slug');
            });
        }

        if (Schema::hasTable('sales_page_orders')) {
            Schema::table('sales_page_orders', function (Blueprint $table) {
                $this->dropIndexIfExists($table, 'idx_sp_orders_page_created');
                $this->dropIndexIfExists($table, 'idx_sp_orders_status');
                $this->dropIndexIfExists($table, 'idx_sp_orders_phone');
            });
        }

        if (Schema::hasTable('products')) {
            Schema::table('products', function (Blueprint $table) {
                $this->dropIndexIfExists($table, 'idx_products_user');
                $this->dropIndexIfExists($table, 'idx_products_sku');
            });
        }

        if (Schema::hasTable('user_product')) {
            Schema::table('user_product', function (Blueprint $table) {
                $this->dropIndexIfExists($table, 'idx_user_product_user');
                $this->dropIndexIfExists($table, 'idx_user_product_product');
                $this->dropIndexIfExists($table, 'idx_user_product_enabled');
            });
        }

        if (Schema::hasTable('customers')) {
            Schema::table('customers', function (Blueprint $table) {
                $this->dropIndexIfExists($table, 'idx_customers_phone');
                $this->dropIndexIfExists($table, 'idx_customers_email');
            });
        }

        if (Schema::hasTable('webhook_sources')) {
            Schema::table('webhook_sources', function (Blueprint $table) {
                $this->dropIndexIfExists($table, 'idx_webhook_sources_active');
                $this->dropIndexIfExists($table, 'idx_webhook_sources_token');
            });
        }

        if (Schema::hasTable('whatsapp_settings')) {
            Schema::table('whatsapp_settings', function (Blueprint $table) {
                $this->dropIndexIfExists($table, 'idx_whatsapp_settings_user_active');
            });
        }

        if (Schema::hasTable('seller_domains')) {
            Schema::table('seller_domains', function (Blueprint $table) {
                $this->dropIndexIfExists($table, 'idx_seller_domains_domain');
                $this->dropIndexIfExists($table, 'idx_seller_domains_user_active');
            });
        }
    }

    /**
     * Check if index exists (MariaDB/MySQL compatible)
     */
    private function indexExists(string $table, string $index): bool
    {
        try {
            $indexes = DB::select("SHOW INDEX FROM {$table} WHERE Key_name = ?", [$index]);
            return count($indexes) > 0;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Drop index if it exists
     */
    private function dropIndexIfExists(Blueprint $table, string $index): void
    {
        try {
            $table->dropIndex($index);
        } catch (\Exception $e) {
            // Index doesn't exist, ignore
        }
    }
};
