<?php

if (!function_exists('formatPhoneNumber')) {
    /**
     * Format phone number for display with WhatsApp integration
     * 
     * @param string $phone
     * @param string $countryCode Default country code (without +)
     * @param bool $includeWhatsApp Whether to include WhatsApp link
     * @return string
     */
    function formatPhoneNumber($phone, $countryCode = '60', $includeWhatsApp = true)
    {
        if (empty($phone)) {
            return '';
        }

        // Clean the phone number
        $cleanPhone = preg_replace('/[^0-9+]/', '', $phone);
        
        // If phone already has country code, use as is
        if (strpos($cleanPhone, '+') === 0) {
            $formattedPhone = $cleanPhone;
            $whatsappNumber = str_replace('+', '', $cleanPhone);
        } else {
            // Remove leading zero if present
            if (strpos($cleanPhone, '0') === 0) {
                $cleanPhone = substr($cleanPhone, 1);
            }
            
            $formattedPhone = '+' . $countryCode . $cleanPhone;
            $whatsappNumber = $countryCode . $cleanPhone;
        }

        if ($includeWhatsApp && !empty($cleanPhone)) {
            $whatsappUrl = generateWhatsAppUrl($whatsappNumber);
            return '<a href="' . $whatsappUrl . '" class="whatsapp-link" target="_blank" title="Chat on WhatsApp">' . 
                   $formattedPhone . 
                   '<svg width="12" height="12" viewBox="0 0 24 24" fill="currentColor" style="margin-left: 4px;">' .
                   '<path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893A11.821 11.821 0 0020.885 3.488"/>' .
                   '</svg></a>';
        }

        return $formattedPhone;
    }
}

if (!function_exists('generateWhatsAppUrl')) {
    /**
     * Generate WhatsApp URL for given phone number
     * 
     * @param string $phone Phone number (without + sign)
     * @param string $message Optional pre-filled message
     * @return string
     */
    function generateWhatsAppUrl($phone, $message = '')
    {
        $baseUrl = 'https://wa.me/';
        $cleanPhone = preg_replace('/[^0-9]/', '', $phone);
        
        $url = $baseUrl . $cleanPhone;
        
        if (!empty($message)) {
            $url .= '?text=' . urlencode($message);
        }
        
        return $url;
    }
}

if (!function_exists('detectCountryFromPhone')) {
    /**
     * Detect country from phone number
     * 
     * @param string $phone
     * @return array ['code' => 'MY', 'country_code' => '60', 'name' => 'Malaysia']
     */
    function detectCountryFromPhone($phone)
    {
        $countryMap = [
            '60' => ['code' => 'MY', 'country_code' => '60', 'name' => 'Malaysia'],
            '65' => ['code' => 'SG', 'country_code' => '65', 'name' => 'Singapore'],
            '1' => ['code' => 'US', 'country_code' => '1', 'name' => 'United States'],
            '44' => ['code' => 'GB', 'country_code' => '44', 'name' => 'United Kingdom'],
            '86' => ['code' => 'CN', 'country_code' => '86', 'name' => 'China'],
            '81' => ['code' => 'JP', 'country_code' => '81', 'name' => 'Japan'],
            '82' => ['code' => 'KR', 'country_code' => '82', 'name' => 'South Korea'],
            '91' => ['code' => 'IN', 'country_code' => '91', 'name' => 'India'],
            '62' => ['code' => 'ID', 'country_code' => '62', 'name' => 'Indonesia'],
            '66' => ['code' => 'TH', 'country_code' => '66', 'name' => 'Thailand'],
            '84' => ['code' => 'VN', 'country_code' => '84', 'name' => 'Vietnam'],
        ];

        $cleanPhone = preg_replace('/[^0-9+]/', '', $phone);
        
        if (strpos($cleanPhone, '+') === 0) {
            $cleanPhone = substr($cleanPhone, 1);
        }
        
        // Check for matches
        foreach ($countryMap as $code => $info) {
            if (strpos($cleanPhone, $code) === 0) {
                return $info;
            }
        }
        
        // Default to Malaysia if no match
        return ['code' => 'MY', 'country_code' => '60', 'name' => 'Malaysia'];
    }
}

if (!function_exists('validatePhoneNumber')) {
    /**
     * Validate phone number format
     * 
     * @param string $phone
     * @return bool
     */
    function validatePhoneNumber($phone)
    {
        if (empty($phone)) {
            return false;
        }
        
        $cleanPhone = preg_replace('/[^0-9+]/', '', $phone);
        
        // Must be at least 7 digits (without country code)
        if (strlen(preg_replace('/[^0-9]/', '', $cleanPhone)) < 7) {
            return false;
        }
        
        // Must not exceed 15 digits (international standard)
        if (strlen(preg_replace('/[^0-9]/', '', $cleanPhone)) > 15) {
            return false;
        }
        
        return true;
    }
}

if (!function_exists('formatPhoneForDisplay')) {
    /**
     * Format phone number for display only (no links)
     * 
     * @param string $phone
     * @param string $countryCode
     * @return string
     */
    function formatPhoneForDisplay($phone, $countryCode = '60')
    {
        return formatPhoneNumber($phone, $countryCode, false);
    }
}

if (!function_exists('getPhoneWithWhatsApp')) {
    /**
     * Get phone number with WhatsApp link and custom message
     * 
     * @param string $phone
     * @param string $message
     * @param string $countryCode
     * @return string
     */
    function getPhoneWithWhatsApp($phone, $message = '', $countryCode = '60')
    {
        if (empty($phone)) {
            return '';
        }

        $cleanPhone = preg_replace('/[^0-9+]/', '', $phone);
        
        if (strpos($cleanPhone, '+') === 0) {
            $formattedPhone = $cleanPhone;
            $whatsappNumber = str_replace('+', '', $cleanPhone);
        } else {
            if (strpos($cleanPhone, '0') === 0) {
                $cleanPhone = substr($cleanPhone, 1);
            }
            $formattedPhone = '+' . $countryCode . $cleanPhone;
            $whatsappNumber = $countryCode . $cleanPhone;
        }

        $whatsappUrl = generateWhatsAppUrl($whatsappNumber, $message);
        
        return '<a href="' . $whatsappUrl . '" class="whatsapp-link" target="_blank" title="Chat on WhatsApp">' . 
               $formattedPhone . 
               '<svg width="12" height="12" viewBox="0 0 24 24" fill="currentColor" style="margin-left: 4px;">' .
               '<path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893A11.821 11.821 0 0020.885 3.488"/>' .
               '</svg></a>';
    }
}
