<?php

namespace App\Services;

use App\Models\Store;
use App\Models\Order;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class WooCommerceService
{
    protected $store;
    protected $baseUrl;
    protected $auth;

    public function __construct(Store $store)
    {
        $this->store = $store;
        $this->baseUrl = rtrim($store->url, '/') . '/wp-json/wc/v3';
        $this->auth = [
            $store->consumer_key,
            $store->consumer_secret
        ];
    }

    public function getOrders($params = [], $retries = 3)
    {
        $defaultParams = [
            'per_page' => 100,
            'orderby' => 'date',
            'order' => 'desc'
        ];

        $params = array_merge($defaultParams, $params);

        $attempt = 0;
        while ($attempt < $retries) {
            try {
                $response = Http::timeout(60) // 60 second timeout
                    ->withBasicAuth($this->auth[0], $this->auth[1])
                    ->get("{$this->baseUrl}/orders", $params);

                if ($response->successful()) {
                    return $response->json();
                }

                // Log unsuccessful response
                Log::warning("Failed to fetch orders from {$this->store->name} (attempt " . ($attempt + 1) . "/{$retries})", [
                    'status' => $response->status(),
                    'body' => $response->body(),
                    'params' => $params
                ]);

                // If 401 Unauthorized or 403 Forbidden, don't retry
                if (in_array($response->status(), [401, 403])) {
                    Log::error("Authentication failed for {$this->store->name} - Check API credentials");
                    return [];
                }

                // If 404, the endpoint doesn't exist
                if ($response->status() == 404) {
                    Log::error("WooCommerce API endpoint not found for {$this->store->name} - Check store URL");
                    return [];
                }

            } catch (\Exception $e) {
                Log::warning("Error fetching orders from {$this->store->name} (attempt " . ($attempt + 1) . "/{$retries})", [
                    'error' => $e->getMessage(),
                    'params' => $params
                ]);
            }

            $attempt++;

            // Wait before retry (exponential backoff)
            if ($attempt < $retries) {
                sleep(2 * $attempt); // Wait 2s, 4s, 6s...
            }
        }

        // All retries failed
        Log::error("Failed to fetch orders from {$this->store->name} after {$retries} attempts");
        return [];
    }

    public function getAllOrders($dateAfter = null, $statuses = ['completed', 'processing'])
    {
        $allOrders = [];

        // Fetch orders for each status separately (WooCommerce limitation)
        foreach ($statuses as $status) {
            $page = 1;
            $perPage = 100;

            do {
                $params = [
                    'status' => $status,
                    'page' => $page,
                    'per_page' => $perPage,
                    'orderby' => 'date',
                    'order' => 'desc'
                ];

                // Add date filter if specified
                if ($dateAfter) {
                    $params['after'] = $dateAfter;
                }

                $orders = $this->getOrders($params);

                if (empty($orders)) {
                    break;
                }

                $allOrders = array_merge($allOrders, $orders);
                $page++;

                // Continue until we get less than perPage results
            } while (count($orders) === $perPage);
        }

        return $allOrders;
    }

    public function updateOrder($orderId, $data)
    {
        try {
            $response = Http::withBasicAuth($this->auth[0], $this->auth[1])
                ->put("{$this->baseUrl}/orders/{$orderId}", $data);

            if ($response->successful()) {
                return $response->json();
            }

            Log::error("Failed to update order {$orderId} in {$this->store->name}", [
                'status' => $response->status(),
                'body' => $response->body()
            ]);

            return false;
        } catch (\Exception $e) {
            Log::error("Error updating order {$orderId} in {$this->store->name}", [
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Update order tracking information in WooCommerce
     * This ensures tracking numbers persist even if store is unlinked/re-linked
     */
    public function updateOrderTracking($orderId, $trackingNumber, $courier = null, $trackingUrl = null)
    {
        try {
            $data = [
                'meta_data' => [
                    [
                        'key' => '_omnia_tracking_number',
                        'value' => $trackingNumber
                    ],
                    [
                        'key' => '_omnia_courier',
                        'value' => $courier ?? 'Pos Laju'
                    ],
                    [
                        'key' => '_omnia_tracking_url',
                        'value' => $trackingUrl ?? ''
                    ],
                    [
                        'key' => '_omnia_awb_generated_at',
                        'value' => now()->toIso8601String()
                    ]
                ]
            ];

            $response = Http::withBasicAuth($this->auth[0], $this->auth[1])
                ->put("{$this->baseUrl}/orders/{$orderId}", $data);

            if ($response->successful()) {
                Log::info("Tracking number synced to WooCommerce", [
                    'store' => $this->store->name,
                    'order_id' => $orderId,
                    'tracking_number' => $trackingNumber,
                    'courier' => $courier
                ]);
                return $response->json();
            }

            Log::error("Failed to sync tracking to WooCommerce order {$orderId}", [
                'store' => $this->store->name,
                'status' => $response->status(),
                'body' => $response->body()
            ]);

            return false;
        } catch (\Exception $e) {
            Log::error("Error syncing tracking to WooCommerce order {$orderId}", [
                'store' => $this->store->name,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Add note to WooCommerce order
     * Useful for recording processing status and internal notes
     */
    public function addOrderNote($orderId, $note, $isCustomerNote = false)
    {
        try {
            $data = [
                'note' => $note,
                'customer_note' => $isCustomerNote
            ];

            $response = Http::withBasicAuth($this->auth[0], $this->auth[1])
                ->post("{$this->baseUrl}/orders/{$orderId}/notes", $data);

            if ($response->successful()) {
                Log::info("Note added to WooCommerce order {$orderId}", [
                    'store' => $this->store->name,
                    'note_preview' => substr($note, 0, 50)
                ]);
                return $response->json();
            }

            Log::error("Failed to add note to WooCommerce order {$orderId}", [
                'store' => $this->store->name,
                'status' => $response->status()
            ]);

            return false;
        } catch (\Exception $e) {
            Log::error("Error adding note to WooCommerce order {$orderId}", [
                'store' => $this->store->name,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Move WooCommerce order to trash (soft delete)
     * This is triggered when order is deleted in our system
     */
    public function trashOrder($orderId)
    {
        try {
            // WooCommerce API: DELETE with force=false moves to trash
            $response = Http::withBasicAuth($this->auth[0], $this->auth[1])
                ->delete("{$this->baseUrl}/orders/{$orderId}", ['force' => false]);

            if ($response->successful()) {
                Log::info("Order moved to trash in WooCommerce", [
                    'store' => $this->store->name,
                    'order_id' => $orderId
                ]);
                return $response->json();
            }

            Log::error("Failed to trash order {$orderId} in WooCommerce", [
                'store' => $this->store->name,
                'status' => $response->status()
            ]);

            return false;
        } catch (\Exception $e) {
            Log::error("Error trashing order {$orderId} in WooCommerce", [
                'store' => $this->store->name,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

public function syncOrders($fullSync = false, $dateAfter = null)
{
    // Only sync completed and processing orders to save time
    $statuses = ['completed', 'processing'];

    // For full sync, get all orders; otherwise get recent ones
    if ($fullSync) {
        $orders = $this->getAllOrders($dateAfter, $statuses);
    } else {
        // Regular sync - get recent orders (last 30 days)
        $recentDate = now()->subDays(30)->toISOString();
        $orders = $this->getAllOrders($recentDate, $statuses);
    }

    $syncedCount = 0;
    $skippedCount = 0;

    // Get existing order IDs to avoid re-syncing
    $existingOrderIds = Order::where('store_id', $this->store->id)
        ->pluck('woo_order_id')
        ->toArray();

    foreach ($orders as $wooOrder) {
        // Skip if order already exists and hasn't been modified recently
        if (in_array($wooOrder['id'], $existingOrderIds)) {
            $existingOrder = Order::where('store_id', $this->store->id)
                ->where('woo_order_id', $wooOrder['id'])
                ->first();

            // Only skip if the order hasn't been modified since last sync
            if ($existingOrder && $existingOrder->date_modified >= $wooOrder['date_modified']) {
                $skippedCount++;
                continue;
            }
        }

        try {
            $order = Order::updateOrCreate(
                [
                    'store_id' => $this->store->id,
                    'woo_order_id' => $wooOrder['id']
                ],
                [
                    'order_number' => $wooOrder['number'],
                    // global_order_id will auto-generate from Order model
                    'status' => $wooOrder['status'],
                    'total' => $wooOrder['total'],
                    'currency' => $wooOrder['currency'],
                    'payment_method' => $wooOrder['payment_method'],
                    'payment_method_title' => $wooOrder['payment_method_title'],
                    'billing' => $wooOrder['billing'],
                    'shipping' => $wooOrder['shipping'],
                    'line_items' => $wooOrder['line_items'],
                    'meta_data' => $wooOrder['meta_data'],
                    'notes' => $wooOrder['customer_note'] ?? null,
                    'date_created' => $wooOrder['date_created'],
                    'date_modified' => $wooOrder['date_modified'],
                ]
            );
            $syncedCount++;
        } catch (\Exception $e) {
            \Log::error("Failed to sync order {$wooOrder['id']} from store {$this->store->name}: " . $e->getMessage());
            // Continue with other orders
        }
    }

    \Log::info("Store {$this->store->name}: Synced {$syncedCount} orders, skipped {$skippedCount} unchanged orders");

    $this->store->update(['last_sync' => now()]);

    return $syncedCount;
}

public function syncHistoricalOrders($monthsBack = 12)
{
    $dateAfter = now()->subMonths($monthsBack)->toISOString();
    return $this->syncOrders(true, $dateAfter);
}

/**
 * Create webhooks for real-time order sync
 */
public function createWebhooks($webhookUrl, $secret)
{
    $webhooks = [];
    $topics = ['order.created', 'order.updated', 'order.deleted'];

    foreach ($topics as $topic) {
        $webhookData = [
            'name' => "OMNIA - {$topic}",
            'topic' => $topic,
            'delivery_url' => $webhookUrl,
            'secret' => $secret,
            'status' => 'active'
        ];

        try {
            $response = Http::withBasicAuth($this->auth[0], $this->auth[1])
                ->post("{$this->baseUrl}/webhooks", $webhookData);

            if ($response->successful()) {
                $webhook = $response->json();
                $webhooks[$topic] = $webhook['id'];
                Log::info("Created webhook for {$this->store->name}: {$topic}", ['webhook_id' => $webhook['id']]);
            } else {
                Log::error("Failed to create webhook for {$this->store->name}: {$topic}", [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
            }
        } catch (\Exception $e) {
            Log::error("Error creating webhook for {$this->store->name}: {$topic}", [
                'error' => $e->getMessage()
            ]);
        }
    }

    return $webhooks;
}

/**
 * Delete webhooks
 */
public function deleteWebhooks(array $webhookIds)
{
    foreach ($webhookIds as $topic => $webhookId) {
        try {
            $response = Http::withBasicAuth($this->auth[0], $this->auth[1])
                ->delete("{$this->baseUrl}/webhooks/{$webhookId}", ['force' => true]);

            if ($response->successful()) {
                Log::info("Deleted webhook for {$this->store->name}: {$topic}", ['webhook_id' => $webhookId]);
            } else {
                Log::error("Failed to delete webhook for {$this->store->name}: {$topic}", [
                    'webhook_id' => $webhookId,
                    'status' => $response->status()
                ]);
            }
        } catch (\Exception $e) {
            Log::error("Error deleting webhook for {$this->store->name}: {$topic}", [
                'webhook_id' => $webhookId,
                'error' => $e->getMessage()
            ]);
        }
    }
}

public function syncHistoricalOrdersWithTimeLimit($monthsBack = 12, $maxExecutionTime = 540, $startTime = null)
{
    if (!$startTime) {
        $startTime = time();
    }

    $dateAfter = now()->subMonths($monthsBack)->toISOString();
    $statuses = ['completed', 'processing'];
    $syncedCount = 0;
    $timeoutReached = false;

    // Get existing order IDs for this store to avoid re-syncing
    $existingOrderIds = Order::where('store_id', $this->store->id)
        ->pluck('woo_order_id')
        ->toArray();

    foreach ($statuses as $status) {
        if ((time() - $startTime) > $maxExecutionTime) {
            $timeoutReached = true;
            break;
        }

        $page = 1;
        $perPage = 100;

        do {
            // Check time limit before each API call
            if ((time() - $startTime) > $maxExecutionTime) {
                $timeoutReached = true;
                break;
            }

            $params = [
                'status' => $status,
                'page' => $page,
                'per_page' => $perPage,
                'orderby' => 'date',
                'order' => 'desc',
                'after' => $dateAfter
            ];

            $orders = $this->getOrders($params);

            if (empty($orders)) {
                break;
            }

            foreach ($orders as $wooOrder) {
                // Check time limit during processing
                if ((time() - $startTime) > $maxExecutionTime) {
                    $timeoutReached = true;
                    break;
                }

                // Skip if order already exists and hasn't been modified
                if (in_array($wooOrder['id'], $existingOrderIds)) {
                    $existingOrder = Order::where('store_id', $this->store->id)
                        ->where('woo_order_id', $wooOrder['id'])
                        ->first();

                    if ($existingOrder && $existingOrder->date_modified >= $wooOrder['date_modified']) {
                        continue; // Skip unchanged order
                    }
                }

                try {
                    Order::updateOrCreate(
                        [
                            'store_id' => $this->store->id,
                            'woo_order_id' => $wooOrder['id']
                        ],
                        [
                            'order_number' => $wooOrder['number'],
                            // global_order_id will auto-generate from Order model
                            'status' => $wooOrder['status'],
                            'total' => $wooOrder['total'],
                            'currency' => $wooOrder['currency'],
                            'payment_method' => $wooOrder['payment_method'],
                            'payment_method_title' => $wooOrder['payment_method_title'],
                            'billing' => $wooOrder['billing'],
                            'shipping' => $wooOrder['shipping'],
                            'line_items' => $wooOrder['line_items'],
                            'meta_data' => $wooOrder['meta_data'],
                            'notes' => $wooOrder['customer_note'] ?? null,
                            'date_created' => $wooOrder['date_created'],
                            'date_modified' => $wooOrder['date_modified'],
                        ]
                    );
                    $syncedCount++;
                } catch (\Exception $e) {
                    \Log::error("Failed to sync order {$wooOrder['id']} from store {$this->store->name}: " . $e->getMessage());
                }
            }

            if ($timeoutReached) {
                break;
            }

            $page++;

        } while (count($orders) === $perPage && !$timeoutReached);

        if ($timeoutReached) {
            \Log::info("Time limit reached for store {$this->store->name}. Synced {$syncedCount} orders before timeout.");
            break;
        }
    }

    $this->store->update(['last_sync' => now()]);

    return $syncedCount;
}

/**
 * Major sync for all order statuses (for migration to new system)
 * This syncs ALL order statuses including pending, failed, cancelled, etc.
 * Used when seller wants complete historical records.
 */
public function syncMajorHistoricalOrders($monthsBack = 24, $maxExecutionTime = 900, $startTime = null)
{
    if (!$startTime) {
        $startTime = time();
    }

    // FIXED: Changed from toISOString() to toIso8601String() for WooCommerce compatibility
    $dateAfter = now()->subMonths($monthsBack)->toIso8601String();

    // ALL statuses for major sync - complete migration
    $statuses = [
        'processing',
        'completed',
        'pending',
        'failed',
        'on-hold',
        'cancelled',
        'refunded'
    ];

    $syncedCount = 0;
    $timeoutReached = false;

    // Get existing order IDs for this store to avoid re-syncing
    $existingOrderIds = Order::where('store_id', $this->store->id)
        ->pluck('woo_order_id')
        ->toArray();

    Log::info("Starting major sync for store {$this->store->name}: {$monthsBack} months, all statuses");
    Log::info("Date range: From {$dateAfter} to now ({$monthsBack} months back)");
    Log::info("Syncing statuses: " . implode(', ', $statuses));

    foreach ($statuses as $status) {
        if ((time() - $startTime) > $maxExecutionTime) {
            $timeoutReached = true;
            Log::warning("Time limit reached at status: {$status}");
            break;
        }

        $page = 1;
        $perPage = 100;

        do {
            // Check time limit before each API call
            if ((time() - $startTime) > $maxExecutionTime) {
                $timeoutReached = true;
                break;
            }

            $params = [
                'status' => $status,
                'page' => $page,
                'per_page' => $perPage,
                'orderby' => 'date',
                'order' => 'desc',
                'after' => $dateAfter
            ];

            // Log API request details (only first page and every 10th page to avoid log spam)
            if ($page == 1 || $page % 10 == 0) {
                Log::info("Fetching {$status} orders: page {$page}, after {$dateAfter}");
            }

            $orders = $this->getOrders($params);

            if (empty($orders)) {
                if ($page == 1) {
                    Log::warning("No {$status} orders found from {$dateAfter} - check WooCommerce date format or no orders in this status");
                }
                break;
            }

            // Log progress every 10 pages
            if ($page % 10 == 0) {
                Log::info("Progress: Fetched page {$page} ({$status}), got " . count($orders) . " orders");
            }

            foreach ($orders as $wooOrder) {
                // Check time limit during processing
                if ((time() - $startTime) > $maxExecutionTime) {
                    $timeoutReached = true;
                    break;
                }

                // Skip if order already exists and hasn't been modified
                if (in_array($wooOrder['id'], $existingOrderIds)) {
                    $existingOrder = Order::where('store_id', $this->store->id)
                        ->where('woo_order_id', $wooOrder['id'])
                        ->first();

                    if ($existingOrder && $existingOrder->date_modified >= $wooOrder['date_modified']) {
                        continue; // Skip unchanged order
                    }
                }

                try {
                    Order::updateOrCreate(
                        [
                            'store_id' => $this->store->id,
                            'woo_order_id' => $wooOrder['id']
                        ],
                        [
                            'order_number' => $wooOrder['number'],
                            // global_order_id will auto-generate from Order model
                            'status' => $wooOrder['status'],
                            'total' => $wooOrder['total'],
                            'currency' => $wooOrder['currency'],
                            'payment_method' => $wooOrder['payment_method'],
                            'payment_method_title' => $wooOrder['payment_method_title'],
                            'billing' => $wooOrder['billing'],
                            'shipping' => $wooOrder['shipping'],
                            'line_items' => $wooOrder['line_items'],
                            'meta_data' => $wooOrder['meta_data'],
                            'notes' => $wooOrder['customer_note'] ?? null,
                            'date_created' => $wooOrder['date_created'],
                            'date_modified' => $wooOrder['date_modified'],
                        ]
                    );
                    $syncedCount++;
                } catch (\Exception $e) {
                    Log::error("Failed to sync order {$wooOrder['id']} from store {$this->store->name}: " . $e->getMessage());
                }
            }

            if ($timeoutReached) {
                break;
            }

            $page++;

        } while (count($orders) === $perPage && !$timeoutReached);

        if ($timeoutReached) {
            Log::info("Time limit reached for store {$this->store->name}. Synced {$syncedCount} orders before timeout.");
            break;
        }
    }

    $this->store->update(['last_sync' => now()]);

    Log::info("Major sync completed for store {$this->store->name}: {$syncedCount} orders synced");

    return $syncedCount;
}
}