<?php

namespace App\Services;

use App\Models\Order;
use App\Models\WhatsAppSetting;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Exception;

class WhatsAppNotificationService
{
    protected $settings;

    public function __construct($userId = null)
    {
        // Get settings from database
        if ($userId) {
            $this->settings = WhatsAppSetting::where('user_id', $userId)->where('is_active', true)->first();
        } else {
            // Try to get settings from auth user or first active setting
            $this->settings = auth()->check()
                ? WhatsAppSetting::where('user_id', auth()->id())->where('is_active', true)->first()
                : WhatsAppSetting::where('is_active', true)->first();
        }
    }

    /**
     * Send WhatsApp notification for order event
     */
    public function sendOrderNotification(Order $order, string $event)
    {
        if (!$this->settings || !$this->settings->is_active) {
            Log::info('WhatsApp notifications disabled', ['order_id' => $order->id]);
            return false;
        }

        // Check if this event should trigger notification
        $shouldNotify = match($event) {
            'order_created' => $this->settings->notify_on_order_created,
            'order_processing' => $this->settings->notify_on_order_processing,
            'order_completed' => $this->settings->notify_on_order_completed,
            'order_cancelled' => $this->settings->notify_on_order_cancelled,
            'awb_generated' => $this->settings->notify_on_awb_generated,
            default => false,
        };

        if (!$shouldNotify) {
            Log::info('Notification disabled for event', ['event' => $event, 'order_id' => $order->id]);
            return false;
        }

        // Get customer phone number
        $customerPhone = $this->getCustomerPhone($order);
        if (!$customerPhone) {
            Log::warning('No customer phone number found', ['order_id' => $order->id]);
            return false;
        }

        // Get message template
        $message = $this->getMessageForEvent($order, $event);
        if (!$message) {
            Log::warning('No message template for event', ['event' => $event]);
            return false;
        }

        // Send via appropriate provider
        return $this->sendMessage($customerPhone, $message, $order);
    }

    /**
     * Get customer phone number from order
     */
    protected function getCustomerPhone(Order $order): ?string
    {
        $billing = $order->billing ?? [];
        $shipping = $order->shipping ?? [];

        $phone = $billing['phone'] ?? $shipping['phone'] ?? null;

        if (!$phone) {
            return null;
        }

        // Clean phone number (already formatted by Order model)
        return $phone;
    }

    /**
     * Get message for specific event
     */
    protected function getMessageForEvent(Order $order, string $event): ?string
    {
        $templateField = 'template_' . $event;
        $template = $this->settings->$templateField ?? WhatsAppSetting::getDefaultTemplates()[$event] ?? null;

        if (!$template) {
            return null;
        }

        // Replace variables
        $billing = $order->billing ?? [];
        $shipping = $order->shipping ?? $billing;
        // Support both 'name' (from sales page) and 'first_name'/'last_name' (from WooCommerce)
        $customerName = $billing['name'] ?? trim(($billing['first_name'] ?? '') . ' ' . ($billing['last_name'] ?? '')) ?: 'Customer';

        // Build full address
        // Support both 'address' (from sales page) and 'address_1'/'address_2' (from WooCommerce)
        $addressParts = array_filter([
            $shipping['address'] ?? $shipping['address_1'] ?? $billing['address'] ?? $billing['address_1'] ?? '',
            $shipping['address_2'] ?? $billing['address_2'] ?? '',
            $shipping['postcode'] ?? $billing['postcode'] ?? '',
            $shipping['city'] ?? $billing['city'] ?? '',
            $shipping['state'] ?? $billing['state'] ?? '',
        ]);
        $fullAddress = implode(', ', $addressParts) ?: 'N/A';

        // Build product list
        $productList = '';
        if (is_array($order->line_items) && count($order->line_items) > 0) {
            $products = [];
            foreach ($order->line_items as $item) {
                $qty = $item['quantity'] ?? 1;
                $name = $item['name'] ?? 'Product';
                $products[] = "- {$name} (x{$qty})";
            }
            $productList = implode("\n", $products);
        } else {
            $productList = 'N/A';
        }

        $replacements = [
            '{customer_name}' => $customerName,
            '{order_number}' => $order->order_number,
            '{global_order_id}' => $order->global_order_id,
            '{status}' => ucfirst($order->status),
            '{total}' => number_format($order->total, 2),
            '{currency}' => $order->currency,
            '{address}' => $fullAddress,
            '{customer_contact}' => $billing['phone'] ?? 'N/A',
            '{email}' => $billing['email'] ?? 'N/A',
            '{product_list}' => $productList,
            '{tracking_number}' => $order->tracking_number ?? 'N/A',
            '{courier}' => $order->courier ?? 'N/A',
            '{tracking_url}' => $order->tracking_number ? $this->getTrackingUrl($order) : 'N/A',
            '{store_name}' => $order->store->name ?? ($this->settings->sender_name ?? 'Our Store'),
            '{date}' => now()->format('d M Y'),
        ];

        return str_replace(array_keys($replacements), array_values($replacements), $template);
    }

    /**
     * Get tracking URL for courier
     */
    protected function getTrackingUrl(Order $order): string
    {
        if (!$order->tracking_number) {
            return '';
        }

        // Default to Pos Laju tracking
        if ($order->courier === 'Pos Laju' || !$order->courier) {
            return 'https://tracking.pos.com.my/tracking/' . $order->tracking_number;
        }

        // Add more courier tracking URLs here
        return match(strtolower($order->courier)) {
            'j&t' => 'https://www.jtexpress.my/tracking?billno=' . $order->tracking_number,
            'dhl' => 'https://www.dhl.com/my-en/home/tracking.html?tracking-id=' . $order->tracking_number,
            'fedex' => 'https://www.fedex.com/fedextrack/?tracknumbers=' . $order->tracking_number,
            default => 'https://tracking.pos.com.my/tracking/' . $order->tracking_number,
        };
    }

    /**
     * Send message via configured provider
     */
    protected function sendMessage(string $phone, string $message, Order $order): bool
    {
        try {
            $provider = $this->settings->provider;

            $result = match($provider) {
                'official' => $this->sendViaWhatsAppOfficial($phone, $message),
                'wasapbot' => $this->sendViaWasapBot($phone, $message),
                'twilio' => $this->sendViaTwilio($phone, $message),
                'wati' => $this->sendViaWati($phone, $message),
                'webhook' => $this->sendViaWebhook($phone, $message, $order),
                default => $this->sendViaWhatsAppOfficial($phone, $message),
            };

            if ($result) {
                Log::info('WhatsApp notification sent successfully', [
                    'order_id' => $order->id,
                    'phone' => $phone,
                    'provider' => $provider
                ]);
            }

            return $result;

        } catch (Exception $e) {
            Log::error('WhatsApp notification failed', [
                'order_id' => $order->id,
                'phone' => $phone,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send via WhatsApp Official Business API
     */
    protected function sendViaWhatsAppOfficial(string $phone, string $message): bool
    {
        if (!$this->settings->phone_number_id || !$this->settings->access_token) {
            Log::error('WhatsApp Official API credentials not configured');
            return false;
        }

        $url = "https://graph.facebook.com/v18.0/{$this->settings->phone_number_id}/messages";

        $response = Http::withToken($this->settings->access_token)
            ->post($url, [
                'messaging_product' => 'whatsapp',
                'to' => $phone,
                'type' => 'text',
                'text' => [
                    'body' => $message
                ]
            ]);

        return $response->successful();
    }

    /**
     * Send via WasapBot.my (Unofficial WhatsApp API)
     */
    protected function sendViaWasapBot(string $phone, string $message): bool
    {
        if (!$this->settings->access_token || !$this->settings->wasapbot_instance_id) {
            Log::error('WasapBot credentials not configured');
            return false;
        }

        // Clean phone number - WasapBot expects format: 60123456789 (no + or @s.whatsapp.net)
        $cleanPhone = str_replace(['+', '@s.whatsapp.net'], '', $phone);

        // WasapBot API endpoint
        $url = "https://dash.wasapbot.my/api/send";

        $payload = [
            'number' => $cleanPhone,
            'type' => 'text',
            'message' => $message,
            'instance_id' => $this->settings->wasapbot_instance_id,
            'access_token' => $this->settings->access_token,
        ];

        Log::info('WasapBot sending message', [
            'phone' => $cleanPhone,
            'url' => $url,
            'payload' => array_merge($payload, ['access_token' => '***hidden***'])
        ]);

        $response = Http::timeout(30)
            ->asForm()
            ->post($url, $payload);

        if ($response->successful()) {
            $responseData = $response->json();

            Log::info('WasapBot message sent successfully', [
                'phone' => $cleanPhone,
                'response' => $responseData
            ]);

            return true;
        }

        Log::error('WasapBot message failed', [
            'phone' => $cleanPhone,
            'status' => $response->status(),
            'body' => $response->body()
        ]);

        return false;
    }

    /**
     * Send via Twilio WhatsApp API
     */
    protected function sendViaTwilio(string $phone, string $message): bool
    {
        if (!$this->settings->api_key || !$this->settings->api_secret) {
            Log::error('Twilio credentials not configured');
            return false;
        }

        $accountSid = $this->settings->api_key;
        $authToken = $this->settings->api_secret;
        $twilioPhone = $this->settings->phone_number_id; // Format: whatsapp:+601234567890

        $url = "https://api.twilio.com/2010-04-01/Accounts/{$accountSid}/Messages.json";

        $response = Http::asForm()
            ->withBasicAuth($accountSid, $authToken)
            ->post($url, [
                'From' => $twilioPhone,
                'To' => 'whatsapp:' . $phone,
                'Body' => $message
            ]);

        return $response->successful();
    }

    /**
     * Send via WATI (WhatsApp Team Inbox)
     */
    protected function sendViaWati(string $phone, string $message): bool
    {
        if (!$this->settings->api_url || !$this->settings->access_token) {
            Log::error('WATI credentials not configured');
            return false;
        }

        $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->settings->access_token,
                'Content-Type' => 'application/json',
            ])
            ->post($this->settings->api_url . '/api/v1/sendSessionMessage/' . $phone, [
                'messageText' => $message
            ]);

        return $response->successful();
    }

    /**
     * Send via custom webhook endpoint
     * This allows integration with any 3rd party service that accepts webhooks
     */
    protected function sendViaWebhook(string $phone, string $message, Order $order): bool
    {
        if (!$this->settings->api_url) {
            Log::error('Webhook URL not configured');
            return false;
        }

        $payload = [
            'phone' => $phone,
            'message' => $message,
            'order' => [
                'id' => $order->id,
                'order_number' => $order->order_number,
                'global_order_id' => $order->global_order_id,
                'status' => $order->status,
                'total' => $order->total,
                'currency' => $order->currency,
                'tracking_number' => $order->tracking_number,
                'courier' => $order->courier,
            ],
            'timestamp' => now()->toIso8601String(),
        ];

        // Add authentication if configured
        $request = Http::timeout(30);

        if ($this->settings->api_key) {
            // Support different auth methods
            if ($this->settings->api_secret) {
                // Bearer token
                $request = $request->withToken($this->settings->api_key);
            } else {
                // API key in header
                $request = $request->withHeaders([
                    'X-API-Key' => $this->settings->api_key,
                ]);
            }
        }

        $response = $request->post($this->settings->api_url, $payload);

        Log::info('Webhook sent', [
            'url' => $this->settings->api_url,
            'status' => $response->status(),
            'response' => $response->body()
        ]);

        return $response->successful();
    }

    /**
     * Test connection to WhatsApp provider
     */
    public function testConnection(): array
    {
        if (!$this->settings || !$this->settings->is_active) {
            return [
                'success' => false,
                'message' => 'WhatsApp notifications are not configured or disabled',
            ];
        }

        try {
            // Send a test message to yourself (use your own number)
            $testPhone = $this->settings->phone_number_id ?? '+60123456789';
            $testMessage = "Test message from OMNIA\n\nWhatsApp integration is working! ✅\n\nProvider: {$this->settings->provider}\nTime: " . now()->format('d M Y H:i:s');

            $result = $this->sendMessage($testPhone, $testMessage, new Order());

            return [
                'success' => $result,
                'message' => $result ? 'Test message sent successfully!' : 'Failed to send test message',
                'provider' => $this->settings->provider,
            ];

        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Error: ' . $e->getMessage(),
            ];
        }
    }
}
