<?php

namespace App\Services;

use App\Models\TikTokTransaction;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Carbon\Carbon;

class TikTokImportService
{
    /**
     * Parse TikTok Excel file and extract simplified data
     */
    public function parseExcelFile($filePath)
    {
        $spreadsheet = IOFactory::load($filePath);
        $sheet = $spreadsheet->getActiveSheet();
        $rows = $sheet->toArray();

        // Find header row and get column indexes
        $headerRow = $rows[0];
        $columnIndexes = $this->getColumnIndexes($headerRow);

        // Parse data rows (skip header)
        $transactions = [];
        $skipped = 0;
        $errors = [];

        for ($i = 1; $i < count($rows); $i++) {
            $row = $rows[$i];

            try {
                $transaction = $this->parseRow($row, $columnIndexes);

                if ($transaction) {
                    $transactions[] = $transaction;
                }
            } catch (\Exception $e) {
                $skipped++;
                $errors[] = "Row " . ($i + 1) . ": " . $e->getMessage();
            }
        }

        return [
            'transactions' => $transactions,
            'total_rows' => count($rows) - 1,
            'parsed' => count($transactions),
            'skipped' => $skipped,
            'errors' => $errors,
            'summary' => $this->calculateSummary($transactions),
        ];
    }

    /**
     * Get column indexes from header row
     */
    private function getColumnIndexes($headerRow)
    {
        $indexes = [];

        foreach ($headerRow as $index => $header) {
            $header = trim($header);

            // Map important columns
            if (str_contains($header, 'Order/adjustment ID')) {
                $indexes['order_id'] = $index;
            } elseif ($header === 'Type') {
                $indexes['type'] = $index;
            } elseif (str_contains($header, 'Order created time')) {
                $indexes['order_date'] = $index;
            } elseif (str_contains($header, 'Order settled time')) {
                $indexes['settled_date'] = $index;
            } elseif ($header === 'Currency') {
                $indexes['currency'] = $index;
            } elseif ($header === 'Total Revenue') {
                $indexes['total_revenue'] = $index;
            } elseif ($header === 'Total Fees') {
                $indexes['total_fees'] = $index;
            } elseif ($header === 'Total settlement amount') {
                $indexes['net_settlement'] = $index;
            } elseif ($header === 'TikTok Shop commission fee') {
                $indexes['commission_fee'] = $index;
            } elseif ($header === 'Transaction fee') {
                $indexes['transaction_fee'] = $index;
            } elseif ($header === 'Actual shipping fee') {
                $indexes['shipping_fee'] = $index;
            }
        }

        return $indexes;
    }

    /**
     * Parse single row
     */
    private function parseRow($row, $columnIndexes)
    {
        // Skip empty rows
        if (empty($row[$columnIndexes['order_id'] ?? 0])) {
            return null;
        }

        // Parse order date (format: 2025/10/27 or similar)
        $orderDateRaw = $row[$columnIndexes['order_date']] ?? null;
        $orderDate = $this->parseDate($orderDateRaw);

        if (!$orderDate) {
            return null; // Skip if can't parse date
        }

        // Parse settled date
        $settledDateRaw = $row[$columnIndexes['settled_date'] ?? null] ?? null;
        $settledDate = $this->parseDate($settledDateRaw);

        return [
            'order_id' => trim($row[$columnIndexes['order_id']]),
            'type' => trim($row[$columnIndexes['type']] ?? 'Order'),
            'order_date' => $orderDate,
            'settled_date' => $settledDate,
            'currency' => trim($row[$columnIndexes['currency'] ?? 4] ?? 'MYR'),
            'total_revenue' => (float)($row[$columnIndexes['total_revenue']] ?? 0),
            'total_fees' => (float)($row[$columnIndexes['total_fees']] ?? 0),
            'net_settlement' => (float)($row[$columnIndexes['net_settlement']] ?? 0),
            'commission_fee' => (float)($row[$columnIndexes['commission_fee'] ?? null] ?? 0),
            'transaction_fee' => (float)($row[$columnIndexes['transaction_fee'] ?? null] ?? 0),
            'shipping_fee' => (float)($row[$columnIndexes['shipping_fee'] ?? null] ?? 0),
        ];
    }

    /**
     * Parse date from various formats
     */
    private function parseDate($dateString)
    {
        if (empty($dateString) || $dateString === '-' || $dateString === '/') {
            return null;
        }

        try {
            // Try common formats
            if (preg_match('/(\d{4})\/(\d{1,2})\/(\d{1,2})/', $dateString, $matches)) {
                return Carbon::createFromFormat('Y/m/d', $matches[0])->format('Y-m-d');
            }

            if (preg_match('/(\d{4})-(\d{1,2})-(\d{1,2})/', $dateString, $matches)) {
                return $dateString;
            }

            // Try Carbon parse as fallback
            return Carbon::parse($dateString)->format('Y-m-d');
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Calculate summary statistics
     */
    private function calculateSummary($transactions)
    {
        $orders = array_filter($transactions, fn($t) => $t['type'] === 'Order');
        $refunds = array_filter($transactions, fn($t) => $t['type'] === 'Refund');

        $totalRevenue = array_sum(array_column($orders, 'total_revenue'));
        $totalFees = array_sum(array_column($transactions, 'total_fees'));
        $totalSettlement = array_sum(array_column($transactions, 'net_settlement'));
        $totalRefunds = abs(array_sum(array_column($refunds, 'total_revenue')));

        // Get date range
        $dates = array_column($transactions, 'order_date');
        $minDate = !empty($dates) ? min($dates) : null;
        $maxDate = !empty($dates) ? max($dates) : null;

        return [
            'total_orders' => count($orders),
            'total_refunds' => count($refunds),
            'total_revenue' => $totalRevenue,
            'total_fees' => $totalFees,
            'total_settlement' => $totalSettlement,
            'total_refund_amount' => $totalRefunds,
            'avg_order_value' => count($orders) > 0 ? $totalRevenue / count($orders) : 0,
            'date_range' => [
                'from' => $minDate,
                'to' => $maxDate,
            ],
        ];
    }

    /**
     * Import transactions to database
     *
     * @param array $transactions Parsed transaction data
     * @param int $tiktokShopId The TikTok Shop ID these transactions belong to
     * @param int $uploadedBy The user ID who is uploading this data (for audit trail)
     * @return array Import result with success status, counts, and batch ID
     */
    public function importToDatabase($transactions, $tiktokShopId, $uploadedBy)
    {
        $batchId = Str::uuid()->toString();
        $imported = 0;
        $duplicates = 0;

        DB::beginTransaction();

        try {
            foreach ($transactions as $transaction) {
                // Check if already exists
                if (TikTokTransaction::where('order_id', $transaction['order_id'])->exists()) {
                    $duplicates++;
                    continue;
                }

                TikTokTransaction::create([
                    'tiktok_shop_id' => $tiktokShopId,
                    'uploaded_by' => $uploadedBy,
                    'import_batch_id' => $batchId,
                    ...$transaction,
                ]);

                $imported++;
            }

            DB::commit();

            return [
                'success' => true,
                'imported' => $imported,
                'duplicates' => $duplicates,
                'batch_id' => $batchId,
            ];
        } catch (\Exception $e) {
            DB::rollBack();

            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }
}
