<?php

namespace App\Services;

use App\Models\TikTokAdsTransaction;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Carbon\Carbon;

class TikTokAdsImportService
{
    /**
     * Parse TikTok Ads Transaction Excel file
     */
    public function parseExcelFile($filePath)
    {
        $spreadsheet = IOFactory::load($filePath);
        $sheet = $spreadsheet->getActiveSheet();
        $rows = $sheet->toArray();

        // Get header row and column indexes
        $headerRow = $rows[0];
        $columnIndexes = $this->getColumnIndexes($headerRow);

        // Parse data rows (skip header)
        $transactions = [];
        $skipped = 0;
        $errors = [];

        for ($i = 1; $i < count($rows); $i++) {
            $row = $rows[$i];

            try {
                $transaction = $this->parseRow($row, $columnIndexes);

                if ($transaction) {
                    $transactions[] = $transaction;
                }
            } catch (\Exception $e) {
                $skipped++;
                $errors[] = "Row " . ($i + 1) . ": " . $e->getMessage();
            }
        }

        return [
            'transactions' => $transactions,
            'total_rows' => count($rows) - 1,
            'parsed' => count($transactions),
            'skipped' => $skipped,
            'errors' => $errors,
            'summary' => $this->calculateSummary($transactions),
        ];
    }

    /**
     * Get column indexes from header row
     */
    private function getColumnIndexes($headerRow)
    {
        $indexes = [];

        foreach ($headerRow as $index => $header) {
            $header = trim($header);

            // Map columns based on Transaction_700.xlsx structure
            if ($header === 'Transaction time') {
                $indexes['transaction_time'] = $index;
            } elseif ($header === 'Transaction ID') {
                $indexes['transaction_id'] = $index;
            } elseif ($header === 'Payment method') {
                $indexes['payment_method'] = $index;
            } elseif ($header === 'Status') {
                $indexes['status'] = $index;
            } elseif ($header === 'Amount') {
                $indexes['amount'] = $index;
            } elseif ($header === 'Currency') {
                $indexes['currency'] = $index;
            } elseif ($header === 'Transaction type') {
                $indexes['transaction_type'] = $index;
            }
        }

        return $indexes;
    }

    /**
     * Parse a single row into transaction data
     */
    private function parseRow($row, $columnIndexes)
    {
        // Skip empty rows
        if (empty($row[0])) {
            return null;
        }

        // Extract data
        $transactionTime = $row[$columnIndexes['transaction_time']] ?? null;
        $transactionId = $row[$columnIndexes['transaction_id']] ?? null;
        $amount = $row[$columnIndexes['amount']] ?? null;

        // Validate required fields
        if (!$transactionTime || !$transactionId || !$amount) {
            throw new \Exception("Missing required fields");
        }

        // Parse amount (remove + sign and convert to float)
        $amount = str_replace(['+', ','], '', $amount);
        $amount = floatval($amount);

        // Parse transaction time
        try {
            $transactionTime = Carbon::parse($transactionTime);
        } catch (\Exception $e) {
            throw new \Exception("Invalid transaction time format");
        }

        return [
            'transaction_id' => trim($transactionId),
            'transaction_time' => $transactionTime,
            'payment_method' => $row[$columnIndexes['payment_method']] ?? null,
            'status' => $row[$columnIndexes['status']] ?? 'Success',
            'amount' => $amount,
            'currency' => $row[$columnIndexes['currency']] ?? 'MYR',
            'transaction_type' => $row[$columnIndexes['transaction_type']] ?? 'Bill payment',
        ];
    }

    /**
     * Calculate summary statistics
     */
    private function calculateSummary($transactions)
    {
        $totalSpent = 0;
        $transactionCount = count($transactions);
        $successfulCount = 0;

        foreach ($transactions as $transaction) {
            if ($transaction['status'] === 'Success') {
                $totalSpent += $transaction['amount'];
                $successfulCount++;
            }
        }

        return [
            'total_transactions' => $transactionCount,
            'successful_transactions' => $successfulCount,
            'total_spent' => $totalSpent,
            'currency' => $transactions[0]['currency'] ?? 'MYR',
        ];
    }

    /**
     * Import transactions to database
     *
     * @param array $transactions Parsed transaction data
     * @param int $tiktokShopId The TikTok Shop ID these transactions belong to
     * @param int $uploadedBy The user ID who is uploading this data (for audit trail)
     * @return array Import result with success status, counts, and batch ID
     */
    public function importTransactions($transactions, $tiktokShopId, $uploadedBy)
    {
        $batchId = Str::uuid()->toString();
        $imported = 0;
        $duplicates = 0;

        DB::beginTransaction();

        try {
            foreach ($transactions as $transaction) {
                // Check for duplicates
                $exists = TikTokAdsTransaction::where('transaction_id', $transaction['transaction_id'])->exists();

                if ($exists) {
                    $duplicates++;
                    continue;
                }

                // Create transaction
                TikTokAdsTransaction::create([
                    'tiktok_shop_id' => $tiktokShopId,
                    'uploaded_by' => $uploadedBy,
                    'transaction_id' => $transaction['transaction_id'],
                    'transaction_time' => $transaction['transaction_time'],
                    'payment_method' => $transaction['payment_method'],
                    'status' => $transaction['status'],
                    'amount' => $transaction['amount'],
                    'currency' => $transaction['currency'],
                    'transaction_type' => $transaction['transaction_type'],
                    'import_batch_id' => $batchId,
                ]);

                $imported++;
            }

            DB::commit();

            return [
                'success' => true,
                'imported' => $imported,
                'duplicates' => $duplicates,
                'batch_id' => $batchId,
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }
}
