<?php

namespace App\Services;

use App\Models\User;
use App\Models\Order;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class GoogleSheetsService
{
    protected $client;

    /**
     * Initialize Google Client for a specific user
     */
    public function authenticate(User $user)
    {
        if (!class_exists('\Google\Client')) {
            throw new \Exception('Google API Client library not installed. Run: composer require google/apiclient:^2.0');
        }

        $client = new \Google\Client();
        $client->setApplicationName(config('app.name') . ' Order Sync');
        $client->setScopes([
            \Google\Service\Sheets::SPREADSHEETS,
            \Google\Service\Drive::DRIVE_FILE,
        ]);
        $client->setClientId(config('services.google.client_id'));
        $client->setClientSecret(config('services.google.client_secret'));
        $client->setRedirectUri(config('services.google.redirect'));
        $client->setAccessType('offline');
        $client->setPrompt('consent');

        // Set access token if user has one
        if ($user->google_access_token) {
            try {
                $accessToken = Crypt::decryptString($user->google_access_token);
                $client->setAccessToken($accessToken);

                // Refresh token if expired
                if ($client->isAccessTokenExpired()) {
                    if ($user->google_refresh_token) {
                        $refreshToken = Crypt::decryptString($user->google_refresh_token);
                        $client->fetchAccessTokenWithRefreshToken($refreshToken);

                        // Save new access token
                        $newAccessToken = $client->getAccessToken();
                        $user->google_access_token = Crypt::encryptString(json_encode($newAccessToken));
                        $user->google_token_expires_at = Carbon::now()->addSeconds($newAccessToken['expires_in']);
                        $user->save();
                    } else {
                        throw new \Exception('Refresh token not found. Please reconnect your Google account.');
                    }
                }
            } catch (\Exception $e) {
                Log::error('Google Sheets authentication error', [
                    'user_id' => $user->id,
                    'error' => $e->getMessage()
                ]);
                throw new \Exception('Failed to authenticate with Google. Please reconnect your account.');
            }
        }

        $this->client = $client;
        return $client;
    }

    /**
     * Create a new Google Sheet for the user
     */
    public function createSheet(User $user)
    {
        $this->authenticate($user);

        $sheetsService = new \Google\Service\Sheets($this->client);
        $driveService = new \Google\Service\Drive($this->client);

        // Create spreadsheet
        $spreadsheet = new \Google\Service\Sheets\Spreadsheet([
            'properties' => [
                'title' => $user->name . ' - Sales Orders - ' . now()->format('Y-m-d')
            ]
        ]);

        $spreadsheet = $sheetsService->spreadsheets->create($spreadsheet);
        $spreadsheetId = $spreadsheet->spreadsheetId;

        // Add header row
        $this->addHeaderRow($user, $spreadsheetId);

        // Save sheet ID to user
        $user->google_sheet_id = $spreadsheetId;
        $user->save();

        return $spreadsheetId;
    }

    /**
     * Add header row to the sheet
     */
    protected function addHeaderRow(User $user, $spreadsheetId)
    {
        $this->authenticate($user);

        $sheetsService = new \Google\Service\Sheets($this->client);

        $headers = [
            'Global Order ID',
            'Order Number',
            'Date Created',
            'Customer Name',
            'Phone',
            'Email',
            'Address',
            'City',
            'State',
            'Postcode',
            'Products',
            'SKU',
            'Quantity',
            'Order Total',
            'Currency',
            'Payment Method',
            'Status',
            'Tracking Number',
            'Store',
            'Notes',
        ];

        $values = [$headers];
        $body = new \Google\Service\Sheets\ValueRange([
            'values' => $values
        ]);

        $params = [
            'valueInputOption' => 'RAW'
        ];

        $range = 'Sheet1!A1:T1'; // 20 columns (added Notes)
        $sheetsService->spreadsheets_values->update($spreadsheetId, $range, $body, $params);

        // Format header row (bold, background color)
        $requests = [
            new \Google\Service\Sheets\Request([
                'repeatCell' => [
                    'range' => [
                        'sheetId' => 0,
                        'startRowIndex' => 0,
                        'endRowIndex' => 1,
                    ],
                    'cell' => [
                        'userEnteredFormat' => [
                            'backgroundColor' => [
                                'red' => 0.2,
                                'green' => 0.6,
                                'blue' => 0.86
                            ],
                            'textFormat' => [
                                'foregroundColor' => [
                                    'red' => 1.0,
                                    'green' => 1.0,
                                    'blue' => 1.0
                                ],
                                'bold' => true
                            ]
                        ]
                    ],
                    'fields' => 'userEnteredFormat(backgroundColor,textFormat)'
                ]
            ]),
            // Freeze header row
            new \Google\Service\Sheets\Request([
                'updateSheetProperties' => [
                    'properties' => [
                        'sheetId' => 0,
                        'gridProperties' => [
                            'frozenRowCount' => 1
                        ]
                    ],
                    'fields' => 'gridProperties.frozenRowCount'
                ]
            ])
        ];

        $batchUpdateRequest = new \Google\Service\Sheets\BatchUpdateSpreadsheetRequest([
            'requests' => $requests
        ]);

        $sheetsService->spreadsheets->batchUpdate($spreadsheetId, $batchUpdateRequest);
    }

    /**
     * Sync an order to Google Sheets with retry logic
     */
    public function syncOrder(Order $order, User $user, int $attempt = 1, int $maxAttempts = 3)
    {
        // Check if user has Google Sheets enabled
        if (!$user->google_sheets_sync_enabled || !$user->google_sheet_id) {
            Log::warning('Google Sheets sync skipped - sync not enabled or no sheet ID', [
                'user_id' => $user->id,
                'sync_enabled' => $user->google_sheets_sync_enabled,
                'has_sheet_id' => !empty($user->google_sheet_id)
            ]);
            return false;
        }

        Log::info('Starting Google Sheets sync', [
            'order_id' => $order->id,
            'user_id' => $user->id,
            'order_number' => $order->order_number
        ]);

        try {
            Log::info('Authenticating with Google...', ['user_id' => $user->id]);
            $this->authenticate($user);
            Log::info('Authentication successful', ['user_id' => $user->id]);

            $sheetsService = new \Google\Service\Sheets($this->client);
            $spreadsheetId = $user->google_sheet_id;

            Log::info('Checking if order exists in sheet...', [
                'order_id' => $order->id,
                'global_order_id' => $order->global_order_id,
                'sheet_id' => $spreadsheetId
            ]);

            // Check if order already exists in sheet (using global_order_id for uniqueness)
            $existingRow = $this->findOrderRow($user, $order->global_order_id);

            if ($existingRow) {
                // Update existing row
                Log::info('Updating existing row', [
                    'order_id' => $order->id,
                    'row_number' => $existingRow
                ]);
                return $this->updateOrderRow($user, $order, $existingRow);
            } else {
                // Append new row
                Log::info('Appending new row', ['order_id' => $order->id]);
                $result = $this->appendOrder($user, $order);
                Log::info('Row appended successfully', [
                    'order_id' => $order->id,
                    'result' => $result
                ]);
                return $result;
            }
        } catch (\Google\Service\Exception $e) {
            // Check if it's a rate limit error (429) or quota exceeded (403)
            $statusCode = $e->getCode();
            $isRateLimitError = in_array($statusCode, [429, 403]);

            if ($isRateLimitError && $attempt < $maxAttempts) {
                // Calculate exponential backoff: 2^attempt seconds
                $delay = pow(2, $attempt);

                Log::warning('Google Sheets rate limit hit, retrying', [
                    'order_id' => $order->id,
                    'user_id' => $user->id,
                    'attempt' => $attempt,
                    'delay_seconds' => $delay,
                    'status_code' => $statusCode
                ]);

                // Wait before retrying
                sleep($delay);

                // Retry
                return $this->syncOrder($order, $user, $attempt + 1, $maxAttempts);
            }

            Log::error('Failed to sync order to Google Sheets', [
                'order_id' => $order->id,
                'user_id' => $user->id,
                'error' => $e->getMessage(),
                'status_code' => $statusCode,
                'attempt' => $attempt
            ]);
            return false;
        } catch (\Exception $e) {
            Log::error('Failed to sync order to Google Sheets', [
                'order_id' => $order->id,
                'user_id' => $user->id,
                'error' => $e->getMessage(),
                'attempt' => $attempt
            ]);
            return false;
        }
    }

    /**
     * Append order to Google Sheet
     */
    protected function appendOrder(User $user, Order $order)
    {
        $sheetsService = new \Google\Service\Sheets($this->client);
        $spreadsheetId = $user->google_sheet_id;

        $rowData = $this->formatOrderData($order);

        $body = new \Google\Service\Sheets\ValueRange([
            'values' => [$rowData]
        ]);

        $params = [
            'valueInputOption' => 'RAW'
        ];

        $range = 'Sheet1!A:T'; // Updated to include Notes column (T)
        $result = $sheetsService->spreadsheets_values->append($spreadsheetId, $range, $body, $params);

        // Update last sync timestamp
        $user->google_sheets_last_sync = now();
        $user->save();

        return true;
    }

    /**
     * Find order row in Google Sheet by global order ID (unique identifier)
     */
    protected function findOrderRow(User $user, $globalOrderId)
    {
        $sheetsService = new \Google\Service\Sheets($this->client);
        $spreadsheetId = $user->google_sheet_id;

        $range = 'Sheet1!A:A'; // Global Order ID column (Column A - first column)
        $response = $sheetsService->spreadsheets_values->get($spreadsheetId, $range);
        $values = $response->getValues();

        if ($values) {
            foreach ($values as $index => $row) {
                if (isset($row[0]) && $row[0] === $globalOrderId) {
                    return $index + 1; // Row number (1-indexed)
                }
            }
        }

        return null;
    }

    /**
     * Update existing order row in Google Sheet
     */
    protected function updateOrderRow(User $user, Order $order, $rowNumber)
    {
        $sheetsService = new \Google\Service\Sheets($this->client);
        $spreadsheetId = $user->google_sheet_id;

        $rowData = $this->formatOrderData($order);

        $body = new \Google\Service\Sheets\ValueRange([
            'values' => [$rowData]
        ]);

        $params = [
            'valueInputOption' => 'RAW'
        ];

        $range = "Sheet1!A{$rowNumber}:T{$rowNumber}"; // Updated to include Notes column (T)
        $result = $sheetsService->spreadsheets_values->update($spreadsheetId, $range, $body, $params);

        // Update last sync timestamp
        $user->google_sheets_last_sync = now();
        $user->save();

        return true;
    }

    /**
     * Format order data into array for Google Sheets
     */
    protected function formatOrderData(Order $order)
    {
        $billing = is_string($order->billing) ? json_decode($order->billing, true) : $order->billing;
        $shipping = is_string($order->shipping) ? json_decode($order->shipping, true) : $order->shipping;
        $lineItems = is_string($order->line_items) ? json_decode($order->line_items, true) : $order->line_items;

        // Customer info
        $customerName = trim(($billing['first_name'] ?? '') . ' ' . ($billing['last_name'] ?? ''));
        $phone = $billing['phone'] ?? '';
        $email = $billing['email'] ?? '';

        // Address
        $address = $shipping['address_1'] ?? $billing['address_1'] ?? '';
        $city = $shipping['city'] ?? $billing['city'] ?? '';
        $state = $shipping['state'] ?? $billing['state'] ?? '';
        $postcode = $shipping['postcode'] ?? $billing['postcode'] ?? '';

        // Products
        $products = [];
        $skus = [];
        $quantities = [];

        if (is_array($lineItems)) {
            foreach ($lineItems as $item) {
                $lineItemQuantity = isset($item['quantity']) ? intval($item['quantity']) : 1;

                // Check if this is a bundle item
                if (isset($item['bundle_items']) && is_array($item['bundle_items']) && count($item['bundle_items']) > 0) {
                    foreach ($item['bundle_items'] as $bundleItem) {
                        $products[] = $bundleItem['name'] ?? 'Unknown';
                        $skus[] = $bundleItem['sku'] ?? '';
                        $bundleItemQuantity = isset($bundleItem['quantity']) ? intval($bundleItem['quantity']) : 1;
                        $quantities[] = $lineItemQuantity * $bundleItemQuantity;
                    }
                } else {
                    $products[] = $item['name'] ?? 'Unknown';
                    $skus[] = $item['sku'] ?? '';
                    $quantities[] = $lineItemQuantity;
                }
            }
        }

        return [
            $order->global_order_id ?? '',           // Column A (swapped to first)
            $order->order_number,                     // Column B (swapped to second)
            $order->date_created ? $order->date_created->format('Y-m-d H:i:s') : '',
            $customerName,
            $phone,
            $email,
            $address,
            $city,
            $state,
            $postcode,
            implode(', ', $products),
            implode(', ', $skus),
            implode(', ', $quantities),
            (float)$order->total,
            $order->currency ?? 'MYR',
            $order->payment_method_title ?? $order->payment_method ?? '',
            $order->status,
            $order->tracking_number ?? '',
            $order->store->name ?? '',
            $order->notes ?? '',                      // Column T (new - notes from checkout)
        ];
    }

    /**
     * Batch sync all orders for a user
     */
    public function batchSyncOrders(User $user, $limit = null)
    {
        if (!$user->google_sheets_sync_enabled || !$user->google_sheet_id) {
            throw new \Exception('Google Sheets sync is not enabled for this user.');
        }

        // Get orders created by this user or from their stores
        $query = Order::query();

        if ($user->isSeller()) {
            $storeIds = $user->getStoreIds();
            $query->where(function($q) use ($storeIds, $user) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhere('created_by', $user->id);
            });
        }

        if ($limit) {
            $query->limit($limit);
        }

        $orders = $query->orderBy('date_created', 'desc')->get();

        $synced = 0;
        $failed = 0;

        foreach ($orders as $order) {
            if ($this->syncOrder($order, $user)) {
                $synced++;
            } else {
                $failed++;
            }

            // Delay to respect Google Sheets API rate limits
            // 1.2 seconds = ~50 requests/minute (stays under 60/min limit)
            usleep(1200000); // 1.2 seconds
        }

        return [
            'synced' => $synced,
            'failed' => $failed,
            'total' => $orders->count(),
        ];
    }

    /**
     * Revoke Google access token
     */
    public function revokeAccess(User $user)
    {
        try {
            $this->authenticate($user);
            $this->client->revokeToken();
        } catch (\Exception $e) {
            Log::warning('Failed to revoke Google token', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);
        }

        // Clear Google data from user
        $user->google_sheet_id = null;
        $user->google_access_token = null;
        $user->google_refresh_token = null;
        $user->google_token_expires_at = null;
        $user->google_sheets_sync_enabled = false;
        $user->google_sheets_last_sync = null;
        $user->google_email = null;
        $user->save();
    }
}
