<?php

namespace App\Services;

use App\Models\Order;
use App\Models\Store;
use App\Models\TikTokTransaction;
use App\Models\SellerForecast;
use App\Models\User;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardStatsService
{
    // Different cache TTLs for different data freshness needs
    private const CACHE_TTL_TODAY = 30;        // 30 seconds - Near real-time for today's stats
    private const CACHE_TTL_PERIOD = 300;      // 5 minutes - Historical data (last week/month/year)
    private const CACHE_TTL_ANALYTICS = 600;   // 10 minutes - Heavy analytics queries

    public function getPeriodStats($dateRange, $user = null)
    {
        $start = $dateRange['start'];
        $end = $dateRange['end'];
        $userKey = $user ? ($user->isSeller() ? 'seller_' . $user->id : ($user->isManager() ? 'manager_' . $user->id : 'admin')) : 'all';
        $cacheKey = "dashboard_period_stats_{$userKey}_{$start->format('Y-m-d_H-i')}_{$end->format('Y-m-d_H-i')}";

        // Use shorter cache for "today", longer for historical periods
        $isToday = $start->isToday() && $end->isToday();
        $cacheTTL = $isToday ? self::CACHE_TTL_TODAY : self::CACHE_TTL_PERIOD;

        return Cache::remember($cacheKey, $cacheTTL, function () use ($start, $end, $dateRange, $user) {
            return $this->calculatePeriodStats($start, $end, $dateRange, $user);
        });
    }

    private function calculatePeriodStats($start, $end, $dateRange, $user = null)
    {
        // Single query to get all order data we need - EXCLUDE cancelled and failed orders
        $query = Order::whereBetween('date_created', [$start, $end])
            ->whereNotIn('status', ['cancelled', 'failed'])
            ->select(['status', 'total', 'currency', 'payment_method', 'date_created', 'store_id', 'created_by']);

        // Filter by user's stores and created orders if seller or manager
        if ($user && $user->isSeller()) {
            $storeIds = $user->getStoreIds();
            $query->where(function($q) use ($storeIds, $user) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhere('created_by', $user->id);
            });
        } elseif ($user && $user->isManager()) {
            $storeIds = $user->getStoreIds();
            $sellerIds = $user->getManagedSellerIds();
            $query->where(function($q) use ($storeIds, $sellerIds) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhereIn('created_by', $sellerIds);
            });
        }

        $allOrders = $query->get();

        // Determine primary currency
        $primaryCurrency = $allOrders->first()?->currency ??
                          Order::orderBy('id', 'desc')->value('currency') ?? 'MYR';

        // Group orders efficiently in memory
        $ordersByStatus = $allOrders->groupBy('status');
        $completedOrders = $ordersByStatus->get('completed', collect());
        $processingOrders = $ordersByStatus->get('processing', collect());
        $approvalOrders = $ordersByStatus->get('approval', collect());
        $printedOrders = $ordersByStatus->get('printed', collect());
        $pendingOrders = $ordersByStatus->get('pending', collect());
        $onHoldOrders = $ordersByStatus->get('on-hold', collect());
        $cancelledOrders = $ordersByStatus->get('cancelled', collect());

        // Separate COD and non-COD orders
        $completedNonCod = $completedOrders->where('payment_method', '!=', 'cod');
        $completedCod = $completedOrders->where('payment_method', 'cod');
        $processingNonCod = $processingOrders->where('payment_method', '!=', 'cod');
        $processingCod = $processingOrders->where('payment_method', 'cod');

        // Get TikTok Shop data (Admin or Manager with permission) - Use settled_date for accurate revenue tracking
        $tiktokRevenue = 0;
        $tiktokOrders = 0;
        $canSeeTikTok = !$user || $user->isAdmin() || ($user->isManager() && $user->can_manage_tiktok_shop);

        if ($canSeeTikTok) {
            $tiktokQuery = TikTokTransaction::whereBetween('settled_date', [$start, $end]);

            // Filter by assigned TikTok Shops if manager
            if ($user && $user->isManager()) {
                $shopIds = $user->getTikTokShopIds();
                // Only show data if manager has assigned shops
                if (empty($shopIds)) {
                    $canSeeTikTok = false;
                } else {
                    $tiktokQuery->whereIn('tiktok_shop_id', $shopIds);
                }
            }

            if ($canSeeTikTok) {
                $tiktokOrders = (clone $tiktokQuery)->where('type', 'Order')->count();
                // Sum ALL net_settlement (includes refunds as negative values) = actual bank transfer amount
                $tiktokRevenue = (clone $tiktokQuery)->sum('net_settlement');
            }
        }

        $stats = [
            'period_label' => $dateRange['label'],
            'start_date' => $start->format('M d, Y'),
            'end_date' => $end->format('M d, Y'),
            'primary_currency' => $primaryCurrency,

            // Order counts
            'total_orders' => $allOrders->count(),
            'completed_orders' => $completedNonCod->count(),
            'completed_cod_orders' => $completedCod->count(),
            'processing_orders' => $processingOrders->count(),
            'pending_orders' => $pendingOrders->count(),
            'cancelled_orders' => $cancelledOrders->count(),

            // Revenue calculations
            'total_revenue' => $completedNonCod->sum('total'),
            'cod_revenue' => $completedCod->sum('total'),
            'processing_revenue' => $processingNonCod->sum('total'),
            'processing_cod_revenue' => $processingCod->sum('total'),

            // TikTok Shop stats (Admin/Manager only)
            'tiktok_orders' => $tiktokOrders,
            'tiktok_revenue' => $tiktokRevenue,

            // Combined stats (ALL orders except cancelled/failed + TikTok + Historical)
            // Since we already filtered out cancelled/failed in the query, sum ALL remaining orders
            'combined_orders' => $allOrders->count() + $tiktokOrders,
            'combined_revenue' => $allOrders->sum('total') + $tiktokRevenue,
        ];

        // Add historical sales for sellers (from SellerForecast table)
        $stats['historical_revenue'] = 0;
        $stats['historical_orders'] = 0;

        if ($user && $user->isSeller()) {
            // Seller: Show only their historical data
            $historicalData = $this->getHistoricalData($user->id, $start, $end);
            $stats['historical_revenue'] = $historicalData['revenue'];
            $stats['historical_orders'] = $historicalData['orders'];
        } elseif ($user && $user->isManager()) {
            // Manager: Show sum of all managed sellers' historical data
            $sellerIds = $user->getManagedSellerIds();
            foreach ($sellerIds as $sellerId) {
                $historicalData = $this->getHistoricalData($sellerId, $start, $end);
                $stats['historical_revenue'] += $historicalData['revenue'];
                $stats['historical_orders'] += $historicalData['orders'];
            }
        } elseif (!$user || $user->isAdmin()) {
            // Admin: Show sum of ALL sellers' historical data
            $allSellers = User::where('role', 'seller')->pluck('id');
            foreach ($allSellers as $sellerId) {
                $historicalData = $this->getHistoricalData($sellerId, $start, $end);
                $stats['historical_revenue'] += $historicalData['revenue'];
                $stats['historical_orders'] += $historicalData['orders'];
            }
        }

        // Include historical data in combined totals
        $stats['combined_revenue'] += $stats['historical_revenue'];
        $stats['combined_orders'] += $stats['historical_orders'];

        // Calculate average order value (now includes historical data)
        $stats['average_order_value'] = $stats['combined_orders'] > 0
            ? $stats['combined_revenue'] / $stats['combined_orders']
            : 0;

        // Revenue breakdown for UI
        $stats['revenue_breakdown_legacy'] = [
            'completed' => $completedNonCod->sum('total'),
            'completed_cod' => $completedCod->sum('total'),
            'processing' => $processingNonCod->sum('total'),
            'processing_cod' => $processingCod->sum('total'),
            'pending' => $pendingOrders->where('payment_method', '!=', 'cod')->sum('total'),
            'pending_cod' => $pendingOrders->where('payment_method', 'cod')->sum('total'),
            'on_hold' => $onHoldOrders->where('payment_method', '!=', 'cod')->sum('total'),
            'on_hold_cod' => $onHoldOrders->where('payment_method', 'cod')->sum('total'),
        ];

        // Daily/Monthly breakdown
        $stats['daily_breakdown'] = $this->getTimeBreakdown($start, $end, $allOrders, $user);

        // Chart title
        $stats['chart_title'] = $this->getChartTitle($start, $end);

        return $stats;
    }

    private function getTimeBreakdown($start, $end, $allOrders, $user = null)
    {
        $isYearlyPeriod = $start->diffInDays($end) > 90;
        // Use ALL orders (already filtered to exclude cancelled/failed in the query)
        $relevantOrders = $allOrders;

        if ($isYearlyPeriod) {
            return $this->getMonthlyBreakdown($start, $end, $relevantOrders, $user);
        } else {
            return $this->getDailyBreakdown($start, $end, $relevantOrders, $user);
        }
    }

    private function getDailyBreakdown($start, $end, $orders, $user = null)
    {
        $breakdown = [];
        $current = $start->copy();
        $maxDays = 30;
        $dayCount = 0;

        // Group orders by date for efficiency
        $ordersByDate = $orders->groupBy(function ($order) {
            return Carbon::parse($order->date_created)->format('Y-m-d');
        });

        // Get TikTok Shop data grouped by date (Admin or Manager with permission) - Use settled_date
        $tiktokByDate = collect();
        $canSeeTikTok = !$user || $user->isAdmin() || ($user->isManager() && $user->can_manage_tiktok_shop);

        if ($canSeeTikTok) {
            $tiktokQuery = TikTokTransaction::whereBetween('settled_date', [$start, $end])
                ->selectRaw('DATE(settled_date) as date, COUNT(CASE WHEN type = "Order" THEN 1 END) as orders, SUM(net_settlement) as revenue')
                ->groupBy('date');

            // Filter by assigned TikTok Shops if manager
            if ($user && $user->isManager()) {
                $shopIds = $user->getTikTokShopIds();
                if (empty($shopIds)) {
                    $canSeeTikTok = false;
                } else {
                    $tiktokQuery->whereIn('tiktok_shop_id', $shopIds);
                }
            }

            if ($canSeeTikTok) {
                $tiktokByDate = $tiktokQuery->get()->keyBy('date');
            }
        }

        while ($current->lte($end) && $dayCount < $maxDays) {
            $dateKey = $current->format('Y-m-d');
            $dayOrders = $ordersByDate->get($dateKey, collect());
            $tiktokData = $tiktokByDate->get($dateKey);

            $breakdown[] = [
                'date' => $dateKey,
                'day' => $current->format('d'),
                'label' => $current->format('M d'),
                'full_date' => $current->format('M d, Y'),
                'orders' => $dayOrders->count() + ($tiktokData ? $tiktokData->orders : 0),
                'revenue' => (float) $dayOrders->sum('total') + ($tiktokData ? $tiktokData->revenue : 0),
            ];

            $current->addDay();
            $dayCount++;
        }

        return $breakdown;
    }

    private function getMonthlyBreakdown($start, $end, $orders, $user = null)
    {
        $breakdown = [];
        $current = $start->copy()->startOfMonth();
        $maxMonths = 12;
        $monthCount = 0;

        // Group orders by month for efficiency
        $ordersByMonth = $orders->groupBy(function ($order) {
            return Carbon::parse($order->date_created)->format('Y-m');
        });

        // Get TikTok Shop data grouped by month (Admin or Manager with permission) - Use settled_date
        $tiktokByMonth = collect();
        $canSeeTikTok = !$user || $user->isAdmin() || ($user->isManager() && $user->can_manage_tiktok_shop);

        if ($canSeeTikTok) {
            $tiktokQuery = TikTokTransaction::whereBetween('settled_date', [$start, $end])
                ->selectRaw('DATE_FORMAT(settled_date, "%Y-%m") as month, COUNT(CASE WHEN type = "Order" THEN 1 END) as orders, SUM(net_settlement) as revenue')
                ->groupBy('month');

            // Filter by assigned TikTok Shops if manager
            if ($user && $user->isManager()) {
                $shopIds = $user->getTikTokShopIds();
                if (empty($shopIds)) {
                    $canSeeTikTok = false;
                } else {
                    $tiktokQuery->whereIn('tiktok_shop_id', $shopIds);
                }
            }

            if ($canSeeTikTok) {
                $tiktokByMonth = $tiktokQuery->get()->keyBy('month');
            }
        }

        while ($current->lte($end) && $monthCount < $maxMonths) {
            $monthKey = $current->format('Y-m');
            $monthOrders = $ordersByMonth->get($monthKey, collect());
            $tiktokData = $tiktokByMonth->get($monthKey);

            $breakdown[] = [
                'date' => $monthKey,
                'day' => $current->format('n'),
                'label' => $current->format('M'),
                'full_date' => $current->format('M Y'),
                'orders' => $monthOrders->count() + ($tiktokData ? $tiktokData->orders : 0),
                'revenue' => (float) $monthOrders->sum('total') + ($tiktokData ? $tiktokData->revenue : 0),
            ];

            $current->addMonth();
            $monthCount++;
        }

        return $breakdown;
    }

    private function getChartTitle($start, $end)
    {
        if ($start->format('Y-m') === $end->format('Y-m')) {
            return $start->format('M Y');
        } elseif ($start->format('Y') === $end->format('Y')) {
            return $start->format('M') . ' - ' . $end->format('M Y');
        } else {
            return $start->format('M Y') . ' - ' . $end->format('M Y');
        }
    }

    /**
     * Get historical sales data from SellerForecast table
     * Includes manually entered past sales data (before system launch)
     * IMPORTANT: Excludes current month to avoid double counting with live orders
     *
     * @param int $userId Seller/Manager user ID
     * @param Carbon|null $start Optional start date filter
     * @param Carbon|null $end Optional end date filter
     * @return array ['revenue' => float, 'orders' => int]
     */
    private function getHistoricalData($userId, $start = null, $end = null)
    {
        $query = SellerForecast::where('user_id', $userId);

        // Apply date filters if provided
        if ($start && $end) {
            // Filter by year and month range
            $query->where(function($q) use ($start, $end) {
                $q->where(function($sq) use ($start, $end) {
                    // Include all months within the date range
                    $sq->where('year', '>=', $start->year)
                       ->where('year', '<=', $end->year);
                });
            });

            // Further filter by months if needed
            if ($start->year === $end->year) {
                $query->whereBetween('month', [$start->month, $end->month]);
            }
        }

        // Exclude current month to prevent double counting with live orders
        $currentYear = now()->year;
        $currentMonth = now()->month;
        $query->where(function($q) use ($currentYear, $currentMonth) {
            $q->where('year', '<', $currentYear)
              ->orWhere(function($sq) use ($currentYear, $currentMonth) {
                  $sq->where('year', '=', $currentYear)
                     ->where('month', '<', $currentMonth);
              });
        });

        return [
            'revenue' => $query->sum('actual_sales') ?? 0,
            'orders' => $query->sum('orders_count') ?? 0,
        ];
    }

    public function getTodayStats($user = null)
    {
        $userKey = $user ? ($user->isSeller() ? 'seller_' . $user->id : ($user->isManager() ? 'manager_' . $user->id : 'admin')) : 'all';
        $cacheKey = "dashboard_today_stats_{$userKey}_" . now()->format('Y-m-d_H-i'); // Include minute for more frequent cache busting

        return Cache::remember($cacheKey, self::CACHE_TTL_TODAY, function () use ($user) {
            // Base queries
            $todayOrdersQuery = Order::where(function($query) {
                $query->whereDate('created_at', today())
                      ->orWhereDate('date_created', today());
            });

            $processingQuery = Order::where('status', 'processing');

            $completedQuery = Order::where('status', 'completed')
                ->where(function($query) {
                    $query->whereDate('date_modified', today())
                          ->orWhereDate('created_at', today());
                });

            // Apply seller filtering
            if ($user && $user->isSeller()) {
                $storeIds = $user->getStoreIds();
                $todayOrdersQuery->where(function($q) use ($storeIds, $user) {
                    $q->whereIn('store_id', $storeIds)
                      ->orWhere('created_by', $user->id);
                });
                $processingQuery->where(function($q) use ($storeIds, $user) {
                    $q->whereIn('store_id', $storeIds)
                      ->orWhere('created_by', $user->id);
                });
                $completedQuery->where(function($q) use ($storeIds, $user) {
                    $q->whereIn('store_id', $storeIds)
                      ->orWhere('created_by', $user->id);
                });
            } elseif ($user && $user->isManager()) {
                $storeIds = $user->getStoreIds();
                $sellerIds = $user->getManagedSellerIds();
                $todayOrdersQuery->where(function($q) use ($storeIds, $sellerIds) {
                    $q->whereIn('store_id', $storeIds)
                      ->orWhereIn('created_by', $sellerIds);
                });
                $processingQuery->where(function($q) use ($storeIds, $sellerIds) {
                    $q->whereIn('store_id', $storeIds)
                      ->orWhereIn('created_by', $sellerIds);
                });
                $completedQuery->where(function($q) use ($storeIds, $sellerIds) {
                    $q->whereIn('store_id', $storeIds)
                      ->orWhereIn('created_by', $sellerIds);
                });
            }

            // Calculate stats
            $stats = [
                'total_orders' => $todayOrdersQuery->count(),
                'processing' => $processingQuery->count(),
                'completed' => $completedQuery->count(),
                'revenue' => (clone $completedQuery)->sum('total'),
            ];

            // Store count logic
            if ($user && $user->isSeller()) {
                $stats['stores_active'] = $user->stores()->where('is_active', true)->count();
            } else {
                $stats['stores_active'] = Store::where('is_active', true)->count();
            }

            return $stats;
        });
    }

    public function getShippingCostEstimation($dateRange, $user = null)
    {
        $start = $dateRange['start'];
        $end = $dateRange['end'];
        $userKey = $user ? ($user->isSeller() ? 'seller_' . $user->id : ($user->isManager() ? 'manager_' . $user->id : 'admin')) : 'all';
        $cacheKey = "dashboard_shipping_costs_{$userKey}_{$start->format('Y-m-d_H-i')}_{$end->format('Y-m-d_H-i')}";

        // Use shorter cache for "today", longer for historical periods
        $isToday = $start->isToday() && $end->isToday();
        $cacheTTL = $isToday ? self::CACHE_TTL_TODAY : self::CACHE_TTL_PERIOD;

        return Cache::remember($cacheKey, $cacheTTL, function () use ($start, $end, $user) {
            return $this->calculateShippingCosts($start, $end, $user);
        });
    }

    private function calculateShippingCosts($start, $end, $user = null)
    {
        // Get system settings
        $onlinePaymentRate = \App\Models\SystemSetting::get('online_payment_shipping_cost', 6.00);
        $codRateEastMalaysia = \App\Models\SystemSetting::get('cod_rate_east_malaysia', 20.00);
        $codRatePeninsular = \App\Models\SystemSetting::get('cod_rate_peninsular', 10.00);

        // Build query: Only orders with tracking numbers, excluding failed/cancelled
        $query = Order::whereBetween('date_created', [$start, $end])
            ->whereNotNull('tracking_number')
            ->where('tracking_number', '!=', '')
            ->whereNotIn('status', ['failed', 'cancelled']);

        // Filter by user's stores and created orders if seller or manager
        if ($user && $user->isSeller()) {
            $storeIds = $user->getStoreIds();
            $query->where(function($q) use ($storeIds, $user) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhere('created_by', $user->id);
            });
        } elseif ($user && $user->isManager()) {
            $storeIds = $user->getStoreIds();
            $sellerIds = $user->getManagedSellerIds();
            $query->where(function($q) use ($storeIds, $sellerIds) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhereIn('created_by', $sellerIds);
            });
        }

        // Get all eligible orders
        $orders = $query->select(['id', 'payment_method', 'status', 'shipping', 'billing'])->get();

        // Calculate shipping costs
        $totalCost = 0;
        $breakdown = [
            'completed' => ['count' => 0, 'cost' => 0],
            'printed' => ['count' => 0, 'cost' => 0],
            'approval' => ['count' => 0, 'cost' => 0],
            'processing' => ['count' => 0, 'cost' => 0],
        ];

        foreach ($orders as $order) {
            $cost = 0;
            $paymentMethod = strtolower($order->payment_method ?? '');

            // Determine shipping cost based on payment method
            if ($paymentMethod === 'cod') {
                // COD: Use state-based rates
                $shipping = is_string($order->shipping) ? json_decode($order->shipping, true) : $order->shipping;
                $billing = is_string($order->billing) ? json_decode($order->billing, true) : $order->billing;
                $stateRaw = $shipping['state'] ?? $billing['state'] ?? '';

                // Normalize state name for consistent matching
                $state = $this->normalizeStateName($stateRaw);

                // East Malaysia states (using normalized names)
                $eastMalaysiaStates = ['Sabah', 'Sarawak', 'Labuan'];
                if (in_array($state, $eastMalaysiaStates)) {
                    $cost = $codRateEastMalaysia;
                } else {
                    $cost = $codRatePeninsular;
                }
            } else {
                // Online Banking, DuitNow QR, etc.: Use flat rate
                $cost = $onlinePaymentRate;
            }

            $totalCost += $cost;

            // Track by status
            $status = $order->status;
            if (isset($breakdown[$status])) {
                $breakdown[$status]['count']++;
                $breakdown[$status]['cost'] += $cost;
            }
        }

        $totalOrders = $orders->count();
        $avgPerOrder = $totalOrders > 0 ? $totalCost / $totalOrders : 0;

        return [
            'total_cost' => $totalCost,
            'total_orders' => $totalOrders,
            'avg_per_order' => $avgPerOrder,
            'breakdown' => $breakdown,
            'rates' => [
                'online_payment' => $onlinePaymentRate,
                'cod_east_malaysia' => $codRateEastMalaysia,
                'cod_peninsular' => $codRatePeninsular,
            ],
        ];
    }

    /**
     * Normalize state names to standardized full names
     * Maps state abbreviations and variations to consistent full names
     */
    private function normalizeStateName($state)
    {
        if (empty($state)) {
            return 'Unknown';
        }

        // Convert to lowercase for comparison
        $stateLower = strtolower(trim($state));

        // State abbreviation and variation mapping
        $stateMapping = [
            // Johor
            'jhr' => 'Johor',
            'johor' => 'Johor',

            // Kedah
            'kdh' => 'Kedah',
            'kedah' => 'Kedah',

            // Kelantan
            'ktn' => 'Kelantan',
            'kelantan' => 'Kelantan',

            // Malacca / Melaka
            'mlk' => 'Melaka',
            'melaka' => 'Melaka',
            'malacca' => 'Melaka',

            // Negeri Sembilan
            'nsn' => 'Negeri Sembilan',
            'ns' => 'Negeri Sembilan',
            'negeri sembilan' => 'Negeri Sembilan',

            // Pahang
            'phg' => 'Pahang',
            'pahang' => 'Pahang',

            // Penang / Pulau Pinang
            'png' => 'Pulau Pinang',
            'penang' => 'Pulau Pinang',
            'pulau pinang' => 'Pulau Pinang',

            // Perak
            'prk' => 'Perak',
            'perak' => 'Perak',

            // Perlis
            'pls' => 'Perlis',
            'perlis' => 'Perlis',

            // Sabah
            'sbh' => 'Sabah',
            'sabah' => 'Sabah',

            // Sarawak
            'swk' => 'Sarawak',
            'sarawak' => 'Sarawak',
            'srwk' => 'Sarawak',

            // Selangor
            'sgr' => 'Selangor',
            'sel' => 'Selangor',
            'selangor' => 'Selangor',

            // Terengganu
            'trg' => 'Terengganu',
            'tganu' => 'Terengganu',
            'terengganu' => 'Terengganu',
            'trengganu' => 'Terengganu',

            // Federal Territories
            'kul' => 'Kuala Lumpur',
            'wp kuala lumpur' => 'Kuala Lumpur',
            'kuala lumpur' => 'Kuala Lumpur',
            'kl' => 'Kuala Lumpur',

            'lbn' => 'Labuan',
            'wp labuan' => 'Labuan',
            'labuan' => 'Labuan',

            'pjy' => 'Putrajaya',
            'wp putrajaya' => 'Putrajaya',
            'putrajaya' => 'Putrajaya',
        ];

        // Return mapped state name or original if not found
        return $stateMapping[$stateLower] ?? ucwords($state);
    }

    public function clearCache()
    {
        $patterns = [
            'dashboard_period_stats_*',
            'dashboard_today_stats_*',
            'dashboard_shipping_costs_*'
        ];

        foreach ($patterns as $pattern) {
            Cache::flush(); // For simplicity, flush all cache. In production, use more specific cache clearing
        }
    }
}