<?php

namespace App\Services;

use App\Models\SystemSetting;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class CloudflareService
{
    protected $apiToken;
    protected $zoneId;
    protected $email;
    protected $baseUrl = 'https://api.cloudflare.com/client/v4';

    public function __construct()
    {
        $this->apiToken = SystemSetting::getCloudflareApiToken();
        $this->zoneId = SystemSetting::getCloudflareZoneId();
        $this->email = SystemSetting::getCloudflareEmail();
    }

    /**
     * Check if Cloudflare is configured
     */
    public function isConfigured(): bool
    {
        return !empty($this->apiToken) && !empty($this->zoneId);
    }

    /**
     * Test Cloudflare API connection
     */
    public function testConnection(): array
    {
        if (!$this->isConfigured()) {
            return [
                'success' => false,
                'message' => 'Cloudflare API credentials not configured',
            ];
        }

        try {
            $response = Http::withHeaders($this->getHeaders())
                ->get("{$this->baseUrl}/zones/{$this->zoneId}");

            $data = $response->json();

            if ($response->successful() && isset($data['success']) && $data['success']) {
                return [
                    'success' => true,
                    'message' => 'Successfully connected to Cloudflare',
                    'zone_name' => $data['result']['name'] ?? null,
                ];
            }

            return [
                'success' => false,
                'message' => $data['errors'][0]['message'] ?? 'Failed to connect to Cloudflare',
            ];
        } catch (\Exception $e) {
            Log::error('Cloudflare connection test failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Connection error: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Create DNS A record for a domain
     */
    public function createDnsRecord(string $domain, string $ipAddress, bool $proxied = true): array
    {
        if (!$this->isConfigured()) {
            return [
                'success' => false,
                'message' => 'Cloudflare API not configured',
            ];
        }

        try {
            // Create @ (root) record
            $rootRecord = $this->createSingleDnsRecord('@', 'A', $ipAddress, $proxied, $domain);

            if (!$rootRecord['success']) {
                return $rootRecord;
            }

            // Create www CNAME record pointing to root domain
            $wwwRecord = $this->createSingleDnsRecord('www', 'CNAME', $domain, $proxied);

            return [
                'success' => true,
                'message' => 'DNS records created successfully',
                'root_record_id' => $rootRecord['record_id'] ?? null,
                'www_record_id' => $wwwRecord['record_id'] ?? null,
            ];
        } catch (\Exception $e) {
            Log::error('Failed to create DNS records: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Error creating DNS records: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Create a single DNS record
     */
    protected function createSingleDnsRecord(string $name, string $type, string $content, bool $proxied = true, ?string $domain = null): array
    {
        try {
            $payload = [
                'type' => $type,
                'name' => $name,
                'content' => $content,
                'ttl' => $proxied ? 1 : 3600, // 1 = Auto for proxied, 3600 for non-proxied
                'proxied' => $proxied,
            ];

            // For CNAME records pointing to external domain, append zone name if needed
            if ($type === 'CNAME' && $domain && strpos($content, '.') === false) {
                $payload['content'] = $content;
            }

            $response = Http::withHeaders($this->getHeaders())
                ->post("{$this->baseUrl}/zones/{$this->zoneId}/dns_records", $payload);

            $data = $response->json();

            if ($response->successful() && isset($data['success']) && $data['success']) {
                return [
                    'success' => true,
                    'record_id' => $data['result']['id'] ?? null,
                    'name' => $data['result']['name'] ?? null,
                ];
            }

            // Check if record already exists
            if (isset($data['errors'][0]['code']) && $data['errors'][0]['code'] === 81057) {
                return [
                    'success' => false,
                    'message' => 'DNS record already exists',
                    'error_code' => 'DUPLICATE',
                ];
            }

            return [
                'success' => false,
                'message' => $data['errors'][0]['message'] ?? 'Failed to create DNS record',
            ];
        } catch (\Exception $e) {
            Log::error("Failed to create {$type} record for {$name}: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Error: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Update DNS record
     */
    public function updateDnsRecord(string $recordId, string $ipAddress, bool $proxied = true): array
    {
        if (!$this->isConfigured()) {
            return [
                'success' => false,
                'message' => 'Cloudflare API not configured',
            ];
        }

        try {
            // First, get the existing record to preserve its details
            $response = Http::withHeaders($this->getHeaders())
                ->get("{$this->baseUrl}/zones/{$this->zoneId}/dns_records/{$recordId}");

            $data = $response->json();

            if (!$response->successful() || !isset($data['result'])) {
                return [
                    'success' => false,
                    'message' => 'DNS record not found',
                ];
            }

            $record = $data['result'];

            // Update the record
            $updateResponse = Http::withHeaders($this->getHeaders())
                ->put("{$this->baseUrl}/zones/{$this->zoneId}/dns_records/{$recordId}", [
                    'type' => $record['type'],
                    'name' => $record['name'],
                    'content' => $ipAddress,
                    'ttl' => $proxied ? 1 : 3600,
                    'proxied' => $proxied,
                ]);

            $updateData = $updateResponse->json();

            if ($updateResponse->successful() && isset($updateData['success']) && $updateData['success']) {
                return [
                    'success' => true,
                    'message' => 'DNS record updated successfully',
                ];
            }

            return [
                'success' => false,
                'message' => $updateData['errors'][0]['message'] ?? 'Failed to update DNS record',
            ];
        } catch (\Exception $e) {
            Log::error('Failed to update DNS record: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Error updating DNS record: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Delete DNS record
     */
    public function deleteDnsRecord(string $recordId): array
    {
        if (!$this->isConfigured()) {
            return [
                'success' => false,
                'message' => 'Cloudflare API not configured',
            ];
        }

        try {
            $response = Http::withHeaders($this->getHeaders())
                ->delete("{$this->baseUrl}/zones/{$this->zoneId}/dns_records/{$recordId}");

            $data = $response->json();

            if ($response->successful() && isset($data['success']) && $data['success']) {
                return [
                    'success' => true,
                    'message' => 'DNS record deleted successfully',
                ];
            }

            return [
                'success' => false,
                'message' => $data['errors'][0]['message'] ?? 'Failed to delete DNS record',
            ];
        } catch (\Exception $e) {
            Log::error('Failed to delete DNS record: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Error deleting DNS record: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * List all DNS records for the zone
     */
    public function listDnsRecords(string $type = null, string $name = null): array
    {
        if (!$this->isConfigured()) {
            return [
                'success' => false,
                'message' => 'Cloudflare API not configured',
                'records' => [],
            ];
        }

        try {
            $params = [];
            if ($type) $params['type'] = $type;
            if ($name) $params['name'] = $name;

            $response = Http::withHeaders($this->getHeaders())
                ->get("{$this->baseUrl}/zones/{$this->zoneId}/dns_records", $params);

            $data = $response->json();

            if ($response->successful() && isset($data['success']) && $data['success']) {
                return [
                    'success' => true,
                    'records' => $data['result'] ?? [],
                ];
            }

            return [
                'success' => false,
                'message' => $data['errors'][0]['message'] ?? 'Failed to list DNS records',
                'records' => [],
            ];
        } catch (\Exception $e) {
            Log::error('Failed to list DNS records: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Error listing DNS records: ' . $e->getMessage(),
                'records' => [],
            ];
        }
    }

    /**
     * Check if DNS record exists for a domain
     */
    public function dnsRecordExists(string $domain): array
    {
        $records = $this->listDnsRecords('A', $domain);

        if (!$records['success']) {
            return [
                'exists' => false,
                'message' => $records['message'],
            ];
        }

        $found = !empty($records['records']);

        return [
            'exists' => $found,
            'record' => $found ? $records['records'][0] : null,
        ];
    }

    /**
     * Verify DNS propagation
     */
    public function verifyDnsPropagation(string $domain, string $expectedIp): array
    {
        try {
            // Use DNS lookup to check if domain resolves to expected IP
            $resolvedIp = gethostbyname($domain);

            if ($resolvedIp === $domain) {
                // Domain didn't resolve
                return [
                    'verified' => false,
                    'message' => 'Domain does not resolve to any IP address',
                ];
            }

            if ($resolvedIp === $expectedIp) {
                return [
                    'verified' => true,
                    'message' => 'DNS propagation successful',
                    'resolved_ip' => $resolvedIp,
                ];
            }

            return [
                'verified' => false,
                'message' => 'Domain resolves to different IP',
                'resolved_ip' => $resolvedIp,
                'expected_ip' => $expectedIp,
            ];
        } catch (\Exception $e) {
            Log::error('DNS verification failed: ' . $e->getMessage());
            return [
                'verified' => false,
                'message' => 'Error verifying DNS: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get request headers for Cloudflare API
     */
    protected function getHeaders(): array
    {
        $headers = [
            'Authorization' => 'Bearer ' . $this->apiToken,
            'Content-Type' => 'application/json',
        ];

        if ($this->email) {
            $headers['X-Auth-Email'] = $this->email;
        }

        return $headers;
    }

    /**
     * Get zone details
     */
    public function getZoneDetails(): array
    {
        if (!$this->isConfigured()) {
            return [
                'success' => false,
                'message' => 'Cloudflare API not configured',
            ];
        }

        try {
            $response = Http::withHeaders($this->getHeaders())
                ->get("{$this->baseUrl}/zones/{$this->zoneId}");

            $data = $response->json();

            if ($response->successful() && isset($data['success']) && $data['success']) {
                return [
                    'success' => true,
                    'zone' => $data['result'],
                ];
            }

            return [
                'success' => false,
                'message' => $data['errors'][0]['message'] ?? 'Failed to get zone details',
            ];
        } catch (\Exception $e) {
            Log::error('Failed to get zone details: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Error: ' . $e->getMessage(),
            ];
        }
    }
}
