<?php

namespace App\Services;

use App\Models\SystemSetting;
use App\Models\User;
use Webimpian\BayarcashSdk\Bayarcash;

class BayarCashService
{
    protected Bayarcash $client;
    protected string $portalKey;
    protected string $secretKey;
    protected ?User $user;

    public function __construct(?User $user = null)
    {
        $this->user = $user;

        // Payment Gateway Hierarchy:
        // 1. Manager's own payment gateway (if they enabled it for themselves)
        // 2. Seller's manager payment gateway (if seller has manager with gateway enabled)
        // 3. Admin/System payment gateway (fallback)
        //
        // Note: Sellers cannot enable their own payment gateway (UI is hidden)
        // Only Managers can enable "Use My Own Payment Gateway"

        if ($user && $user->use_own_payment_gateway) {
            // Use Manager's own payment gateway (only managers can enable this)
            $pat = $user->bayarcash_pat;
            $this->portalKey = $user->bayarcash_portal_key;
            $this->secretKey = $user->bayarcash_secret_key;
            $environment = $user->bayarcash_environment ?? 'sandbox';

            if (!$pat) {
                throw new \Exception('Manager BayarCash PAT is not configured. Please configure your payment gateway in profile settings.');
            }
        } elseif ($user) {
            // Check if seller has a manager with own payment gateway enabled
            $manager = $user->managers()->where('use_own_payment_gateway', true)->first();

            if ($manager) {
                // Use seller's manager payment gateway
                $pat = $manager->bayarcash_pat;
                $this->portalKey = $manager->bayarcash_portal_key;
                $this->secretKey = $manager->bayarcash_secret_key;
                $environment = $manager->bayarcash_environment ?? 'sandbox';

                if (!$pat) {
                    throw new \Exception('Manager BayarCash PAT is not configured. Please ask your manager to configure their payment gateway.');
                }
            } else {
                // Use admin/system payment gateway
                $pat = SystemSetting::get('bayarcash_pat');
                $this->portalKey = SystemSetting::get('bayarcash_portal_key');
                $this->secretKey = SystemSetting::get('bayarcash_secret_key');
                $environment = SystemSetting::get('bayarcash_environment', 'sandbox');

                if (!$pat) {
                    throw new \Exception('BayarCash PAT is not configured in system settings.');
                }
            }
        } else {
            // Use admin/system payment gateway
            $pat = SystemSetting::get('bayarcash_pat');
            $this->portalKey = SystemSetting::get('bayarcash_portal_key');
            $this->secretKey = SystemSetting::get('bayarcash_secret_key');
            $environment = SystemSetting::get('bayarcash_environment', 'sandbox');

            if (!$pat) {
                throw new \Exception('BayarCash PAT is not configured in system settings.');
            }
        }

        $this->client = new Bayarcash($pat);

        if ($environment === 'sandbox') {
            $this->client->useSandbox();
        }
    }

    /**
     * Get available payment portals
     */
    public function getPortals(): array
    {
        return $this->client->getPortals();
    }

    /**
     * Get available payment channels for the configured portal
     */
    public function getChannels(): array
    {
        if (!$this->portalKey) {
            throw new \Exception('BayarCash Portal Key is not configured.');
        }

        return $this->client->getChannels($this->portalKey);
    }

    /**
     * Create a payment intent for FPX or DuitNow QR
     *
     * @param array $data Payment intent data
     * @return \Webimpian\BayarcashSdk\Resources\PaymentIntentResource
     */
    public function createPaymentIntent(array $data)
    {
        if (!$this->portalKey) {
            throw new \Exception('BayarCash Portal Key is not configured.');
        }

        // Add portal key to the payment intent data
        $data['portal_key'] = $this->portalKey;

        return $this->client->createPaymentIntent($data);
    }

    /**
     * Create FPX payment intent
     *
     * @param string $orderNumber Unique order number
     * @param float $amount Payment amount
     * @param string $payerName Customer name
     * @param string $payerEmail Customer email
     * @param string $payerPhone Customer phone
     * @param string $returnUrl URL to return after payment
     * @param string $callbackUrl URL for payment callback
     * @return \Webimpian\BayarcashSdk\Resources\PaymentIntentResource
     */
    public function createFpxPayment(
        string $orderNumber,
        float $amount,
        string $payerName,
        string $payerEmail,
        string $payerPhone,
        string $returnUrl,
        string $callbackUrl
    ) {
        return $this->createPaymentIntent([
            'order_number' => $orderNumber,
            'amount' => $amount,
            'payer_name' => $payerName,
            'payer_email' => $payerEmail,
            'payer_telephone_number' => $payerPhone,
            'payment_channel' => Bayarcash::FPX,
            'return_url' => $returnUrl,
            'callback_url' => $callbackUrl,
        ]);
    }

    /**
     * Create DuitNow QR payment intent
     *
     * @param string $orderNumber Unique order number
     * @param float $amount Payment amount
     * @param string $payerName Customer name
     * @param string $payerEmail Customer email
     * @param string $payerPhone Customer phone
     * @param string $returnUrl URL to return after payment
     * @param string $callbackUrl URL for payment callback
     * @return \Webimpian\BayarcashSdk\Resources\PaymentIntentResource
     */
    public function createDuitNowPayment(
        string $orderNumber,
        float $amount,
        string $payerName,
        string $payerEmail,
        string $payerPhone,
        string $returnUrl,
        string $callbackUrl
    ) {
        return $this->createPaymentIntent([
            'order_number' => $orderNumber,
            'amount' => $amount,
            'payer_name' => $payerName,
            'payer_email' => $payerEmail,
            'payer_telephone_number' => $payerPhone,
            'payment_channel' => Bayarcash::DUITNOW_QR,
            'return_url' => $returnUrl,
            'callback_url' => $callbackUrl,
        ]);
    }

    /**
     * Verify transaction callback data
     *
     * @param array $callbackData Callback data from BayarCash
     * @return bool
     */
    public function verifyCallback(array $callbackData): bool
    {
        if (!$this->secretKey) {
            throw new \Exception('BayarCash Secret Key is not configured.');
        }

        return $this->client->verifyTransactionCallbackData($callbackData, $this->secretKey);
    }

    /**
     * Verify return URL callback data
     *
     * @param array $callbackData Callback data from BayarCash
     * @return bool
     */
    public function verifyReturnCallback(array $callbackData): bool
    {
        if (!$this->secretKey) {
            throw new \Exception('BayarCash Secret Key is not configured.');
        }

        return $this->client->verifyReturnUrlCallbackData($callbackData, $this->secretKey);
    }

    /**
     * Get transaction details
     *
     * @param string $transactionId
     * @return \Webimpian\BayarcashSdk\Resources\TransactionResource
     */
    public function getTransaction(string $transactionId)
    {
        return $this->client->getTransaction($transactionId);
    }

    /**
     * Calculate COD fee based on state
     *
     * @param string $state Customer state
     * @return float COD fee amount
     */
    public static function calculateCodFee(string $state): float
    {
        $eastMalaysiaStates = ['Sabah', 'Sarawak', 'Wp Labuan'];

        if (in_array($state, $eastMalaysiaStates)) {
            return (float) SystemSetting::get('cod_rate_east_malaysia', 20.00);
        }

        return (float) SystemSetting::get('cod_rate_peninsular', 10.00);
    }

    /**
     * Calculate payment gateway fee based on user settings and payment method
     *
     * @param float $amount Order/payment amount
     * @param string $paymentMethod Payment method (fpx or duitnow)
     * @param User|null $user User to get fee settings from (follows payment gateway hierarchy)
     * @return array ['fee' => float, 'total' => float, 'details' => array]
     */
    public function calculatePaymentGatewayFee(float $amount, string $paymentMethod = 'fpx', ?User $user = null): array
    {
        $feeSettings = $this->getFeeSettings($paymentMethod, $user);

        // If no fee configured, return zero fee
        if ($feeSettings['fee_type'] === 'none') {
            return [
                'fee' => 0.00,
                'total' => $amount,
                'details' => [
                    'payment_method' => $paymentMethod,
                    'fee_type' => 'none',
                    'fixed_fee' => 0,
                    'percentage_fee' => 0,
                    'bearer' => $feeSettings['bearer'],
                ]
            ];
        }

        $fixedFee = 0;
        $percentageFee = 0;

        // Calculate based on fee type
        switch ($feeSettings['fee_type']) {
            case 'fixed':
                $fixedFee = floatval($feeSettings['fixed_fee']);
                break;

            case 'percentage':
                $percentageFee = $amount * (floatval($feeSettings['percentage_fee']) / 100);
                break;

            case 'both':
                $fixedFee = floatval($feeSettings['fixed_fee']);
                $percentageFee = $amount * (floatval($feeSettings['percentage_fee']) / 100);
                break;
        }

        $totalFee = $fixedFee + $percentageFee;

        // Round fee to 2 decimal places
        $totalFee = round($totalFee, 2);

        // If customer pays the fee, add to total
        $finalTotal = $feeSettings['bearer'] === 'customer'
            ? $amount + $totalFee
            : $amount;

        return [
            'fee' => $totalFee,
            'total' => $finalTotal,
            'details' => [
                'payment_method' => $paymentMethod,
                'fee_type' => $feeSettings['fee_type'],
                'fixed_fee' => $fixedFee,
                'percentage_fee' => $percentageFee,
                'bearer' => $feeSettings['bearer'],
                'original_amount' => $amount,
            ]
        ];
    }

    /**
     * Get fee settings following payment gateway hierarchy for specific payment method
     *
     * @param string $paymentMethod Payment method (fpx or duitnow)
     * @param User|null $user
     * @return array
     */
    public function getFeeSettings(string $paymentMethod, ?User $user): array
    {
        // Normalize payment method
        $method = strtolower($paymentMethod);
        $prefix = $method === 'duitnow' ? 'duitnow' : 'fpx';

        // Follow same hierarchy as payment gateway selection
        if ($user && $user->use_own_payment_gateway) {
            // Use Manager's own fee settings
            return [
                'fee_type' => $user->{$prefix . '_fee_type'} ?? 'none',
                'fixed_fee' => $user->{$prefix . '_fixed_fee'} ?? 0,
                'percentage_fee' => $user->{$prefix . '_percentage_fee'} ?? 0,
                'bearer' => $user->{$prefix . '_fee_bearer'} ?? 'customer',
            ];
        } elseif ($user) {
            // Check if seller has a manager with own payment gateway enabled
            $manager = $user->managers()->where('use_own_payment_gateway', true)->first();

            if ($manager) {
                // Use seller's manager fee settings
                return [
                    'fee_type' => $manager->{$prefix . '_fee_type'} ?? 'none',
                    'fixed_fee' => $manager->{$prefix . '_fixed_fee'} ?? 0,
                    'percentage_fee' => $manager->{$prefix . '_percentage_fee'} ?? 0,
                    'bearer' => $manager->{$prefix . '_fee_bearer'} ?? 'customer',
                ];
            }
        }

        // Use admin/system fee settings from system_settings table
        return [
            'fee_type' => SystemSetting::get($prefix . '_fee_type', 'none'),
            'fixed_fee' => SystemSetting::get($prefix . '_fixed_fee', 0),
            'percentage_fee' => SystemSetting::get($prefix . '_percentage_fee', 0),
            'bearer' => SystemSetting::get($prefix . '_fee_bearer', 'customer'),
        ];
    }
}
