<?php

namespace App\Observers;

use App\Models\Order;
use App\Models\User;
use App\Models\WebhookSource;
use App\Services\WhatsAppNotificationService;
use App\Notifications\OrderCreatedNotification;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class OrderObserver
{
    /**
     * Handle the Order "created" event.
     */
    public function created(Order $order): void
    {
        Log::info('New order created', [
            'order_id' => $order->id,
            'order_number' => $order->order_number,
            'is_manual' => $order->is_manual,
            'created_by' => $order->created_by,
        ]);

        // Only send "Order Created" notification for internally created orders
        // (Manual checkout and sales page orders)
        // Skip notifications for synced WooCommerce orders and webhook orders
        $isInternalOrder = $order->is_manual || $order->created_by;

        if (!$isInternalOrder) {
            Log::info('Skipping order created notification - external order (synced/webhook)', [
                'order_id' => $order->id,
                'order_number' => $order->order_number,
            ]);
            return;
        }

        // Send order created notification (WhatsApp)
        try {
            $userId = $this->getUserIdForOrder($order);
            $whatsappService = new WhatsAppNotificationService($userId);
            $whatsappService->sendOrderNotification($order, 'order_created');
        } catch (\Exception $e) {
            Log::error('Failed to send order created notification', [
                'order_id' => $order->id,
                'error' => $e->getMessage(),
            ]);
        }

        // Send PWA push notifications (queued to avoid blocking order creation)
        try {
            dispatch(function () use ($order) {
                $this->sendPushNotifications($order);
            })->afterResponse();

            Log::info('PWA push notification queued', [
                'order_id' => $order->id,
                'order_number' => $order->order_number,
            ]);
        } catch (\Exception $e) {
            Log::error('PWA push notification queue failed', [
                'order_id' => $order->id,
                'error' => $e->getMessage(),
            ]);
        }

        // Sync to Google Sheets (if seller has it enabled)
        $this->syncToGoogleSheets($order);
    }

    /**
     * Handle the Order "updated" event.
     * Detect status changes and send WhatsApp notifications
     * Sync tracking numbers and notes to WooCommerce
     */
    public function updated(Order $order): void
    {
        // Check if status changed
        if ($order->isDirty('status')) {
            $oldStatus = $order->getOriginal('status');
            $newStatus = $order->status;

            Log::info('Order status changed', [
                'order_id' => $order->id,
                'old_status' => $oldStatus,
                'new_status' => $newStatus,
            ]);

            // Send WhatsApp notification based on new status
            $this->sendStatusNotification($order, $newStatus);
        }

        // Check if AWB/tracking number was generated
        if ($order->isDirty('tracking_number') && $order->tracking_number) {
            Log::info('Tracking number generated', [
                'order_id' => $order->id,
                'tracking_number' => $order->tracking_number,
            ]);

            // Send AWB generated notification
            $this->sendAWBNotification($order);

            // SYNC TRACKING TO WOOCOMMERCE (Priority 1: Prevents duplicate processing)
            $this->syncTrackingToWooCommerce($order);
        }

        // Check if notes changed - sync to WooCommerce
        if ($order->isDirty('notes') && $order->notes) {
            Log::info('Order notes updated', [
                'order_id' => $order->id,
                'notes_preview' => substr($order->notes, 0, 50),
            ]);

            // Sync notes to WooCommerce
            $this->syncNotesToWooCommerce($order);
        }

        // Sync updated order to Google Sheets
        $this->syncToGoogleSheets($order);
    }

    /**
     * Send status change notification
     */
    protected function sendStatusNotification(Order $order, string $newStatus): void
    {
        try {
            $userId = $this->getUserIdForOrder($order);
            $whatsappService = new WhatsAppNotificationService($userId);

            // Map status to notification event
            $event = match($newStatus) {
                'processing' => 'order_processing',
                'completed' => 'order_completed',
                'cancelled' => 'order_cancelled',
                default => null,
            };

            if ($event) {
                $whatsappService->sendOrderNotification($order, $event);
            }

        } catch (\Exception $e) {
            Log::error('Failed to send status change notification', [
                'order_id' => $order->id,
                'status' => $newStatus,
                'error' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Send AWB generated notification
     *
     * DISABLED: WhatsApp notifications are now sent manually via "Send WhatsApp" button
     * This prevents automatic sending of non-trackable tracking numbers
     */
    protected function sendAWBNotification(Order $order): void
    {
        // Automatic WhatsApp notifications disabled
        // Seller/Admin must manually click "Send WhatsApp" button on orders page

        Log::info('AWB generated - Manual WhatsApp send required', [
            'order_id' => $order->id,
            'tracking_number' => $order->tracking_number,
        ]);

        // Still send to WhatsApp webhook sources if configured
        try {
            $this->sendToWhatsAppWebhooks($order, 'awb_generated');
        } catch (\Exception $e) {
            Log::error('Failed to send to WhatsApp webhooks', [
                'order_id' => $order->id,
                'error' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Send notification to all active WhatsApp webhooks
     */
    protected function sendToWhatsAppWebhooks(Order $order, string $event): void
    {
        try {
            // Get all active WhatsApp webhook sources
            $whatsappWebhooks = WebhookSource::where('type', 'whatsapp')
                ->where('is_active', true)
                ->get();

            if ($whatsappWebhooks->isEmpty()) {
                return;
            }

            // Prepare WhatsApp notification data
            $whatsappService = new WhatsAppNotificationService();
            $customerPhone = $this->getCustomerPhone($order);
            $message = $this->getMessageForEvent($order, $event);

            if (!$customerPhone || !$message) {
                return;
            }

            $payload = [
                'phone' => $customerPhone,
                'message' => $message,
                'order' => [
                    'id' => $order->id,
                    'order_number' => $order->order_number,
                    'global_order_id' => $order->global_order_id,
                    'status' => $order->status,
                    'total' => $order->total,
                    'currency' => $order->currency,
                    'tracking_number' => $order->tracking_number,
                    'courier' => $order->courier,
                ],
                'event' => $event,
                'timestamp' => now()->toIso8601String(),
            ];

            // Send to each webhook
            foreach ($whatsappWebhooks as $webhook) {
                try {
                    $response = Http::timeout(30)
                        ->post($webhook->webhook_url, $payload);

                    Log::info('WhatsApp webhook sent', [
                        'webhook_id' => $webhook->id,
                        'webhook_name' => $webhook->name,
                        'order_id' => $order->id,
                        'status' => $response->status(),
                    ]);

                } catch (\Exception $e) {
                    Log::error('WhatsApp webhook failed', [
                        'webhook_id' => $webhook->id,
                        'webhook_name' => $webhook->name,
                        'order_id' => $order->id,
                        'error' => $e->getMessage(),
                    ]);
                }
            }

        } catch (\Exception $e) {
            Log::error('Failed to send to WhatsApp webhooks', [
                'order_id' => $order->id,
                'error' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Get customer phone number from order
     */
    protected function getCustomerPhone(Order $order): ?string
    {
        $billing = $order->billing ?? [];
        $shipping = $order->shipping ?? [];
        return $billing['phone'] ?? $shipping['phone'] ?? null;
    }

    /**
     * Get formatted message for event
     */
    protected function getMessageForEvent(Order $order, string $event): ?string
    {
        $billing = $order->billing ?? [];
        $customerName = trim(($billing['first_name'] ?? '') . ' ' . ($billing['last_name'] ?? '')) ?: 'Customer';
        $storeName = $order->store->name ?? 'Our Store';

        $templates = [
            'order_created' => "Hello {$customerName}! 👋\n\nYour order #{$order->order_number} has been received.\n\nOrder Total: {$order->currency} {$order->total}\nStatus: " . ucfirst($order->status) . "\n\nThank you for shopping with us!\n\n- {$storeName}",

            'order_processing' => "Hello {$customerName}! 📦\n\nYour order #{$order->order_number} is now being processed.\n\nWe'll notify you once it's shipped!\n\n- {$storeName}",

            'order_completed' => "Hello {$customerName}! ✅\n\nYour order #{$order->order_number} has been completed.\n\nTracking Number: {$order->tracking_number}\nCourier: {$order->courier}\n\nThank you for your purchase!\n\n- {$storeName}",

            'order_cancelled' => "Hello {$customerName},\n\nYour order #{$order->order_number} has been cancelled.\n\nIf you have any questions, please contact us.\n\n- {$storeName}",

            'awb_generated' => "Hello {$customerName}! 🚚\n\nYour order #{$order->order_number} is ready for shipment!\n\nTracking Number: {$order->tracking_number}\nCourier: {$order->courier}\n\nYou can track your parcel here:\n" . $this->getTrackingUrl($order) . "\n\nThank you!\n- {$storeName}",
        ];

        return $templates[$event] ?? null;
    }

    /**
     * Get tracking URL
     */
    protected function getTrackingUrl(Order $order): string
    {
        if (!$order->tracking_number) {
            return '';
        }

        return match(strtolower($order->courier ?? 'pos laju')) {
            'j&t' => 'https://www.jtexpress.my/tracking?billno=' . $order->tracking_number,
            'dhl' => 'https://www.dhl.com/my-en/home/tracking.html?tracking-id=' . $order->tracking_number,
            'fedex' => 'https://www.fedex.com/fedextrack/?tracknumbers=' . $order->tracking_number,
            default => 'https://tracking.pos.com.my/tracking/' . $order->tracking_number,
        };
    }

    /**
     * Handle the Order "deleted" event.
     * Sync deletion to WooCommerce as "trash" status
     */
    public function deleted(Order $order): void
    {
        // Only sync to WooCommerce if this is a WooCommerce order
        if ($order->store && $order->woo_order_id) {
            try {
                $wooService = new \App\Services\WooCommerceService($order->store);
                $wooService->trashOrder($order->woo_order_id);

                Log::info('Order deleted - synced to WooCommerce trash', [
                    'order_id' => $order->id,
                    'woo_order_id' => $order->woo_order_id,
                    'store_id' => $order->store_id,
                ]);
            } catch (\Exception $e) {
                Log::error('Failed to sync order deletion to WooCommerce', [
                    'order_id' => $order->id,
                    'woo_order_id' => $order->woo_order_id,
                    'error' => $e->getMessage(),
                ]);
            }
        }
    }

    /**
     * Handle the Order "restored" event.
     */
    public function restored(Order $order): void
    {
        //
    }

    /**
     * Handle the Order "force deleted" event.
     */
    public function forceDeleted(Order $order): void
    {
        //
    }

    /**
     * Get the user ID for an order (respects each seller's settings)
     */
    protected function getUserIdForOrder(Order $order): ?int
    {
        // Priority 1: Manual order creator (checkout/sales page)
        if ($order->created_by) {
            return $order->created_by;
        }

        // Priority 2: Store owner (for WooCommerce/webhook orders)
        if ($order->store) {
            // Get first user from store's users relationship
            return $order->store->users()->first()?->id;
        }

        return null;
    }

    /**
     * Send PWA push notifications to relevant users based on their role
     */
    protected function sendPushNotifications(Order $order): void
    {
        try {
            $usersToNotify = $this->getUsersToNotifyForPush($order);

            foreach ($usersToNotify as $user) {
                try {
                    // Only notify if user has push subscriptions
                    if ($user->pushSubscriptions()->count() > 0) {
                        $user->notify(new OrderCreatedNotification($order));

                        Log::info('PWA push notification sent', [
                            'order_id' => $order->id,
                            'user_id' => $user->id,
                            'user_name' => $user->name,
                        ]);
                    }
                } catch (\Exception $e) {
                    // Log individual notification failures but continue with others
                    Log::error('Individual push notification failed', [
                        'order_id' => $order->id,
                        'user_id' => $user->id,
                        'error' => $e->getMessage(),
                    ]);
                }
            }

        } catch (\Exception $e) {
            Log::error('Send push notifications failed', [
                'order_id' => $order->id,
                'error' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Determine which users should receive push notifications for this order
     * Based on role hierarchy: Seller -> Manager -> Admin
     */
    protected function getUsersToNotifyForPush(Order $order)
    {
        $users = collect();

        // 1. Notify the seller who created this order
        if ($order->created_by) {
            $seller = User::find($order->created_by);
            if ($seller) {
                $users->push($seller);

                // 2. Notify the seller's manager (if exists and wants team notifications)
                if ($seller->manager_id) {
                    $manager = User::find($seller->manager_id);
                    if ($manager && $this->managerWantsTeamNotifications($manager)) {
                        $users->push($manager);
                    }
                }
            }
        }

        // 3. Notify admins who subscribed to all orders
        $admins = User::where('role', 'admin')
            ->whereHas('pushSubscriptions') // Only admins with push enabled
            ->get()
            ->filter(function($admin) {
                return $this->adminWantsAllOrderNotifications($admin);
            });

        $users = $users->merge($admins);

        // Remove duplicates and null values
        return $users->filter()->unique('id');
    }

    /**
     * Check if manager wants to receive team order notifications
     */
    protected function managerWantsTeamNotifications(User $manager): bool
    {
        // For now, managers get all team notifications
        // Later: check user preference in settings
        return true;
    }

    /**
     * Check if admin wants to receive all order notifications
     */
    protected function adminWantsAllOrderNotifications(User $admin): bool
    {
        // For now, all admins with push subscriptions get all notifications
        // Later: check user preference in settings
        return true;
    }

    /**
     * Sync tracking number to WooCommerce
     * CRITICAL: This allows tracking to persist when store is unlinked/re-linked
     * Prevents duplicate AWB generation
     */
    protected function syncTrackingToWooCommerce(Order $order): void
    {
        // Only sync if this is a WooCommerce order
        if (!$order->store || !$order->woo_order_id) {
            return;
        }

        try {
            $wooService = new \App\Services\WooCommerceService($order->store);

            // Get tracking URL
            $trackingUrl = $this->getTrackingUrl($order);

            // Sync tracking to WooCommerce meta data
            $wooService->updateOrderTracking(
                $order->woo_order_id,
                $order->tracking_number,
                $order->courier ?? 'Pos Laju',
                $trackingUrl
            );

            // Also add order note with tracking info
            $note = "📦 Tracking Number: {$order->tracking_number}\n";
            $note .= "🚚 Courier: " . ($order->courier ?? 'Pos Laju') . "\n";
            $note .= "🔗 Track: {$trackingUrl}\n";
            $note .= "\n✅ Generated via Omnia System";

            $wooService->addOrderNote($order->woo_order_id, $note, false);

            Log::info('Tracking synced to WooCommerce successfully', [
                'order_id' => $order->id,
                'woo_order_id' => $order->woo_order_id,
                'tracking_number' => $order->tracking_number,
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to sync tracking to WooCommerce', [
                'order_id' => $order->id,
                'woo_order_id' => $order->woo_order_id,
                'error' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Sync order notes to WooCommerce
     * Keeps WooCommerce source site updated with processing notes
     */
    protected function syncNotesToWooCommerce(Order $order): void
    {
        // Only sync if this is a WooCommerce order
        if (!$order->store || !$order->woo_order_id) {
            return;
        }

        try {
            $wooService = new \App\Services\WooCommerceService($order->store);

            // Add note to WooCommerce (as admin note, not customer-facing)
            $note = "📝 Internal Note (from Omnia):\n\n{$order->notes}";

            $wooService->addOrderNote($order->woo_order_id, $note, false);

            Log::info('Notes synced to WooCommerce successfully', [
                'order_id' => $order->id,
                'woo_order_id' => $order->woo_order_id,
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to sync notes to WooCommerce', [
                'order_id' => $order->id,
                'woo_order_id' => $order->woo_order_id,
                'error' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Sync order to Google Sheets (for sellers with Google Sheets enabled)
     */
    protected function syncToGoogleSheets(Order $order): void
    {
        try {
            // Get the seller who created this order
            $seller = null;

            if ($order->created_by) {
                $seller = User::find($order->created_by);
            } elseif ($order->store) {
                // Get first seller from store
                $seller = $order->store->users()->where('role', 'seller')->first();
            }

            if (!$seller) {
                return;
            }

            // Check if seller has Google Sheets sync enabled
            if (!$seller->google_sheets_sync_enabled || !$seller->google_sheet_id) {
                return;
            }

            // Dispatch sync job to queue
            \App\Jobs\SyncOrderToGoogleSheets::dispatch($seller, $order);

            Log::info('Google Sheets sync job queued', [
                'order_id' => $order->id,
                'seller_id' => $seller->id,
                'seller_name' => $seller->name,
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to queue Google Sheets sync', [
                'order_id' => $order->id,
                'error' => $e->getMessage(),
            ]);
        }
    }
}
