<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\MailMessage;
use NotificationChannels\WebPush\WebPushMessage;
use NotificationChannels\WebPush\WebPushChannel;
use App\Models\Order;
use App\Models\SystemSetting;

class OrderCreatedNotification extends Notification
{
    use Queueable;

    protected $order;

    /**
     * Create a new notification instance.
     */
    public function __construct(Order $order)
    {
        $this->order = $order;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via($notifiable)
    {
        return ['database', WebPushChannel::class];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail($notifiable)
    {
        return (new MailMessage)
            ->subject('New Order #' . $this->order->order_number)
            ->line('You have received a new order!')
            ->line('Order Number: ' . $this->order->order_number)
            ->line('Total: RM' . number_format($this->order->total, 2))
            ->action('View Order', url('/orders/' . $this->order->id))
            ->line('Thank you for using our application!');
    }

    /**
     * Get the Web Push representation of the notification.
     */
    public function toWebPush($notifiable, $notification)
    {
        // Determine which sound to use
        $sound = $this->getNotificationSound();

        // Build notification title
        $title = $this->getNotificationTitle();

        // Build notification body
        $body = $this->getNotificationBody();

        return (new WebPushMessage)
            ->title($title)
            ->icon('/icons/icon-192.png')
            ->badge('/icons/icon-72.png')
            ->body($body)
            ->action('View Order', 'view')
            ->data([
                'url' => url('/orders/' . $this->order->id),
                'sound' => $sound,
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'total' => $this->order->total,
            ])
            ->options([
                'TTL' => 3600, // 1 hour
                'urgency' => 'high',
                'topic' => 'orders',
            ]);
    }

    /**
     * Get the array representation of the notification (for database).
     */
    public function toArray($notifiable)
    {
        return [
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'total' => $this->order->total,
            'customer_name' => $this->getCustomerName(),
            'platform' => $this->getPlatformName(),
            'created_at' => $this->order->created_at,
        ];
    }

    /**
     * Determine which notification sound to use
     */
    protected function getNotificationSound()
    {
        // Check if it's first order of the day for this user
        // if ($this->isFirstOrderOfDay($notifiable)) {
        //     return config('pwa.sounds.first_of_day');
        // }

        // High value order - use threshold from settings
        $threshold = SystemSetting::getNotificationHighValueThreshold();
        if ($this->order->total >= $threshold) {
            return config('pwa.sounds.high_value');
        }

        // Default sound
        return config('pwa.sounds.default');
    }

    /**
     * Get notification title with emoji and order number
     */
    protected function getNotificationTitle()
    {
        // Get settings
        $threshold = SystemSetting::getNotificationHighValueThreshold();
        $regularEmoji = SystemSetting::getRegularOrderEmoji();
        $highValueEmoji = SystemSetting::getHighValueOrderEmoji();
        $template = SystemSetting::getNotificationTitleTemplate();

        // Determine which emoji to use
        $emoji = $this->order->total >= $threshold ? $highValueEmoji : $regularEmoji;

        // Replace template variables
        return str_replace(
            ['{emoji}', '{order_number}'],
            [$emoji, $this->order->order_number],
            $template
        );
    }

    /**
     * Get notification body
     */
    protected function getNotificationBody()
    {
        // Get template from settings
        $template = SystemSetting::getNotificationBodyTemplate();

        // Get data
        $platform = $this->getPlatformName();
        $total = 'RM' . number_format($this->order->total, 2);
        $customer = $this->getCustomerName();

        // Replace template variables
        return str_replace(
            ['{total}', '{customer}', '{platform}'],
            [$total, $customer, $platform],
            $template
        );
    }

    /**
     * Get platform name from order
     */
    protected function getPlatformName()
    {
        if (str_starts_with($this->order->order_number, 'CK-')) {
            return 'Checkout Form';
        } elseif (str_starts_with($this->order->order_number, 'SP-')) {
            return 'Sales Page';
        } elseif (str_starts_with($this->order->order_number, 'WOO-') || $this->order->store_id) {
            return 'WooCommerce';
        } elseif (str_starts_with($this->order->order_number, 'CH-')) {
            return 'Channel';
        } else {
            return 'Manual';
        }
    }

    /**
     * Get customer name from order
     */
    protected function getCustomerName()
    {
        if (is_array($this->order->billing) && isset($this->order->billing['first_name'])) {
            return trim(($this->order->billing['first_name'] ?? '') . ' ' . ($this->order->billing['last_name'] ?? ''));
        }
        return $this->order->customer_name ?? 'Customer';
    }

    /**
     * Check if this is the first order of the day for the user
     */
    protected function isFirstOrderOfDay($notifiable)
    {
        $today = now()->startOfDay();

        // Check if user has any orders created today before this one
        $previousOrdersToday = Order::where('created_by', $notifiable->id)
            ->whereBetween('created_at', [$today, $this->order->created_at])
            ->where('id', '!=', $this->order->id)
            ->count();

        return $previousOrdersToday === 0;
    }
}
