<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class TikTokAdsTransaction extends Model
{
    protected $table = 'tiktok_ads_transactions';

    protected $fillable = [
        'tiktok_shop_id',
        'uploaded_by',
        'transaction_id',
        'transaction_time',
        'payment_method',
        'status',
        'amount',
        'currency',
        'transaction_type',
        'import_batch_id',
    ];

    protected $casts = [
        'transaction_time' => 'datetime',
        'amount' => 'decimal:2',
    ];

    /**
     * Get the TikTok Shop this transaction belongs to
     */
    public function tiktokShop()
    {
        return $this->belongsTo(TikTokShop::class, 'tiktok_shop_id');
    }

    /**
     * Get the user who uploaded this transaction (audit trail)
     */
    public function uploader()
    {
        return $this->belongsTo(User::class, 'uploaded_by');
    }

    /**
     * Legacy: Keep user() relationship for backward compatibility
     * @deprecated Use uploader() instead
     */
    public function user()
    {
        return $this->uploader();
    }

    /**
     * Scope: Get only successful transactions
     */
    public function scopeSuccessful($query)
    {
        return $query->where('status', 'Success');
    }

    /**
     * Scope: Filter by date range
     */
    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('transaction_time', [$startDate, $endDate]);
    }

    /**
     * Get monthly spending summary
     */
    public static function getMonthlySummary($startDate = null, $endDate = null)
    {
        $query = self::successful();

        if ($startDate && $endDate) {
            $query->dateRange($startDate, $endDate);
        }

        return $query->selectRaw('
            DATE_FORMAT(transaction_time, "%Y-%m") as month,
            COUNT(*) as transaction_count,
            SUM(amount) as total_spent
        ')
        ->groupBy('month')
        ->orderBy('month', 'desc')
        ->get();
    }
}
