<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class SystemSetting extends Model
{
    protected $fillable = ['key', 'value', 'type', 'description'];

    /**
     * Get a setting value by key
     */
    public static function get(string $key, $default = null)
    {
        return Cache::remember("setting.{$key}", 3600, function () use ($key, $default) {
            $setting = self::where('key', $key)->first();

            if (!$setting) {
                return $default;
            }

            return self::castValue($setting->value, $setting->type);
        });
    }

    /**
     * Set a setting value
     */
    public static function set(string $key, $value, string $type = 'string'): void
    {
        self::updateOrCreate(
            ['key' => $key],
            [
                'value' => is_array($value) || is_object($value) ? json_encode($value) : $value,
                'type' => $type
            ]
        );

        Cache::forget("setting.{$key}");
    }

    /**
     * Cast value to appropriate type
     */
    protected static function castValue($value, string $type)
    {
        return match($type) {
            'boolean' => (bool) $value,
            'integer' => (int) $value,
            'json' => json_decode($value, true),
            default => $value,
        };
    }

    /**
     * Get company info source setting (admin or seller)
     */
    public static function getCompanyInfoSource(): string
    {
        return self::get('company_info_source', 'seller');
    }

    /**
     * Get global ID prefix (e.g., OMS, MMD)
     */
    public static function getGlobalIdPrefix(): string
    {
        return self::get('global_id_prefix', 'OMS');
    }

    /**
     * Get site title
     */
    public static function getSiteTitle(): string
    {
        return self::get('site_title', 'OMNIA');
    }

    /**
     * Get site logo URL
     */
    public static function getSiteLogo(): ?string
    {
        return self::get('site_logo', null);
    }

    /**
     * Get site favicon (use logo if no specific favicon is set)
     */
    public static function getFavicon(): ?string
    {
        $favicon = self::get('site_favicon', null);
        if ($favicon) {
            return $favicon;
        }

        // Fall back to logo if no favicon is set
        return self::getSiteLogo();
    }

    /**
     * Get site name
     */
    public static function getSiteName(): string
    {
        return self::get('site_name', 'Omni');
    }

    /**
     * Get login description
     */
    public static function getLoginDescription(): string
    {
        return self::get('login_description', 'Welcome to OMNIA');
    }

    /**
     * Get footer text
     */
    public static function getFooterText(): string
    {
        return self::get('footer_text', '© 2025 OMNIA. All rights reserved.');
    }

    /**
     * Check if registration is enabled
     */
    public static function isRegistrationEnabled(): bool
    {
        return self::get('registration_enabled', 'false') === 'true';
    }

    /**
     * Get Cloudflare API Token
     */
    public static function getCloudflareApiToken(): ?string
    {
        return self::get('cloudflare_api_token', null);
    }

    /**
     * Get Cloudflare Zone ID
     */
    public static function getCloudflareZoneId(): ?string
    {
        return self::get('cloudflare_zone_id', null);
    }

    /**
     * Get Cloudflare Account Email
     */
    public static function getCloudflareEmail(): ?string
    {
        return self::get('cloudflare_email', null);
    }

    /**
     * Check if Cloudflare integration is enabled
     */
    public static function isCloudflareEnabled(): bool
    {
        return !empty(self::getCloudflareApiToken()) && !empty(self::getCloudflareZoneId());
    }

    /**
     * Get Server IP Address
     */
    public static function getServerIp(): ?string
    {
        return self::get('server_ip', null);
    }

    /**
     * Check if automatic DNS creation is enabled
     */
    public static function isAutoDnsEnabled(): bool
    {
        return self::get('auto_dns_enabled', 'false') === 'true';
    }

    /**
     * Check if automatic domain verification is enabled
     */
    public static function isAutoVerificationEnabled(): bool
    {
        return self::get('auto_verification_enabled', 'false') === 'true';
    }

    /**
     * Get PWA notification title template
     */
    public static function getNotificationTitleTemplate(): string
    {
        return self::get('pwa_notification_title', '{emoji} Kaching! Pesanan #{order_number}');
    }

    /**
     * Get PWA notification body template
     */
    public static function getNotificationBodyTemplate(): string
    {
        return self::get('pwa_notification_body', '{total} • {customer} • {platform}');
    }

    /**
     * Get PWA high value threshold
     */
    public static function getNotificationHighValueThreshold(): int
    {
        return (int) self::get('pwa_high_value_threshold', 500);
    }

    /**
     * Get PWA notification volume (0-100)
     */
    public static function getNotificationVolume(): int
    {
        return (int) self::get('pwa_notification_volume', 80);
    }

    /**
     * Check if PWA notifications are enabled
     */
    public static function isPwaNotificationsEnabled(): bool
    {
        return self::get('pwa_notifications_enabled', 'true') === 'true';
    }

    /**
     * Get regular order emoji
     */
    public static function getRegularOrderEmoji(): string
    {
        return self::get('pwa_regular_emoji', '💰');
    }

    /**
     * Get high value order emoji
     */
    public static function getHighValueOrderEmoji(): string
    {
        return self::get('pwa_high_value_emoji', '🎉');
    }
}
