<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Support\Str;

class Store extends Model
{
    protected $fillable = [
        'name', 'url', 'store_prefix', 'consumer_key', 'consumer_secret',
        'is_active', 'settings', 'last_sync', 'webhook_secret', 'webhook_ids', 'webhooks_enabled'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'settings' => 'array',
        'last_sync' => 'datetime',
        'webhook_ids' => 'array',
        'webhooks_enabled' => 'boolean',
    ];

    protected static function booted()
    {
        // Auto-generate store prefix
        static::creating(function ($store) {
            if (empty($store->store_prefix)) {
                $store->store_prefix = $store->generateStorePrefix();
            }
        });
    }

    public function generateStorePrefix()
    {
        $base = strtoupper(substr(preg_replace('/[^A-Za-z0-9]/', '', $this->name), 0, 3));
        
        // Ensure minimum 2 characters
        if (strlen($base) < 2) {
            $base = 'ST';
        }
        
        // Check for uniqueness
        $prefix = $base;
        $counter = 1;
        
        while (self::where('store_prefix', $prefix)->exists()) {
            $prefix = $base . $counter;
            $counter++;
        }
        
        return $prefix;
    }

    protected function consumerKey(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => decrypt($value),
            set: fn ($value) => encrypt($value),
        );
    }

    protected function consumerSecret(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => decrypt($value),
            set: fn ($value) => encrypt($value),
        );
    }

    public function orders()
    {
        return $this->hasMany(Order::class);
    }

    public function processingOrders()
    {
        return $this->orders()->where('status', 'processing');
    }

    /**
     * Get the users (sellers) assigned to this store.
     */
    public function users()
    {
        return $this->belongsToMany(User::class, 'store_user')
                    ->withTimestamps();
    }

    /**
     * Get the primary owner/seller of this store (first assigned seller)
     */
    public function getOwner()
    {
        return $this->users()->where('role', 'seller')->first()
               ?? $this->users()->first()
               ?? User::where('role', 'admin')->first();
    }

    // Get formatted display name with prefix
    public function getDisplayNameAttribute()
    {
        return "[{$this->store_prefix}] {$this->name}";
    }
}