<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Models\User;

class SellerForecast extends Model
{
    protected $fillable = [
        'user_id',
        'year',
        'month',
        'actual_sales',
        'target_sales',
        'forecast_sales',
        'forecast_lower',
        'forecast_upper',
        'orders_count',
        'aov',
        'achievement_percentage',
        'performance_grade',
        'trend',
        'ai_suggestions',
    ];

    protected $casts = [
        'actual_sales' => 'decimal:2',
        'target_sales' => 'decimal:2',
        'forecast_sales' => 'decimal:2',
        'forecast_lower' => 'decimal:2',
        'forecast_upper' => 'decimal:2',
        'aov' => 'decimal:2',
        'achievement_percentage' => 'decimal:2',
    ];

    /**
     * Get the seller/manager user
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Calculate performance grade based on achievement percentage
     */
    public function calculateGrade(): string
    {
        $achievement = $this->achievement_percentage;

        if ($achievement >= 100) return 'A';
        if ($achievement >= 80) return 'B';
        if ($achievement >= 60) return 'C';
        if ($achievement >= 40) return 'D';
        return 'F';
    }

    /**
     * Calculate trend by comparing with previous months
     */
    public function calculateTrend(): string
    {
        // Get previous 3 months data
        $previousMonths = self::where('user_id', $this->user_id)
            ->where('year', '<=', $this->year)
            ->where(function ($query) {
                $query->where('year', '<', $this->year)
                    ->orWhere(function ($q) {
                        $q->where('year', '=', $this->year)
                          ->where('month', '<', $this->month);
                    });
            })
            ->orderByDesc('year')
            ->orderByDesc('month')
            ->take(3)
            ->pluck('actual_sales')
            ->toArray();

        if (count($previousMonths) < 2) {
            return 'stable'; // Not enough data
        }

        $avgPrevious = array_sum($previousMonths) / count($previousMonths);
        $current = $this->actual_sales;

        if ($current > $avgPrevious * 1.1) return 'growing'; // 10% increase
        if ($current < $avgPrevious * 0.9) return 'declining'; // 10% decrease
        return 'stable';
    }

    /**
     * Update achievement percentage and grade
     */
    public function updatePerformanceMetrics(): void
    {
        // Calculate achievement
        if ($this->target_sales > 0) {
            $this->achievement_percentage = ($this->actual_sales / $this->target_sales) * 100;
        } else {
            $this->achievement_percentage = 0;
        }

        // Calculate grade
        $this->performance_grade = $this->calculateGrade();

        // Calculate trend
        $this->trend = $this->calculateTrend();

        $this->save();
    }

    /**
     * Get month name
     */
    public function getMonthNameAttribute(): string
    {
        $months = [
            1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
            5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
            9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
        ];

        return $months[$this->month] ?? 'Unknown';
    }

    /**
     * Get short month name
     */
    public function getMonthShortAttribute(): string
    {
        return substr($this->month_name, 0, 3);
    }

    /**
     * Scope to filter by year
     */
    public function scopeForYear($query, $year)
    {
        return $query->where('year', $year);
    }

    /**
     * Scope to filter by grade
     */
    public function scopeByGrade($query, $grade)
    {
        return $query->where('performance_grade', $grade);
    }

    /**
     * Scope to filter by user (seller/manager)
     */
    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }
}
