<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class SellerDomain extends Model
{
    protected $fillable = [
        'user_id',
        'domain',
        'is_primary',
        'is_active',
        'verification_token',
        'verified_at',
        'default_sales_page_id',
        'cloudflare_record_id',
        'notes',
    ];

    protected $casts = [
        'is_primary' => 'boolean',
        'is_active' => 'boolean',
        'verified_at' => 'datetime',
    ];

    /**
     * Boot method to generate verification token
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($domain) {
            if (empty($domain->verification_token)) {
                $domain->verification_token = Str::random(32);
            }
        });
    }

    /**
     * Get the seller (user) who owns this domain
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the seller (alias for user)
     */
    public function seller()
    {
        return $this->user();
    }

    /**
     * Get the default sales page for this domain (root domain)
     */
    public function defaultSalesPage()
    {
        return $this->belongsTo(SalesPage::class, 'default_sales_page_id');
    }

    /**
     * Get all sales pages belonging to this domain's seller
     */
    public function salesPages()
    {
        return $this->hasMany(SalesPage::class, 'user_id', 'user_id');
    }

    /**
     * Check if domain is verified
     */
    public function isVerified(): bool
    {
        return $this->is_active && !is_null($this->verified_at);
    }

    /**
     * Mark domain as verified
     */
    public function markAsVerified()
    {
        $this->update([
            'is_active' => true,
            'verified_at' => now(),
        ]);
    }

    /**
     * Get the full URL for this domain
     */
    public function getUrlAttribute(): string
    {
        return 'https://' . $this->domain;
    }

    /**
     * Get domain with www prefix if needed
     */
    public function getWithWww(): string
    {
        return str_starts_with($this->domain, 'www.') ? $this->domain : 'www.' . $this->domain;
    }

    /**
     * Get domain without www prefix
     */
    public function getWithoutWww(): string
    {
        return str_starts_with($this->domain, 'www.') ? substr($this->domain, 4) : $this->domain;
    }

    /**
     * Scope: Active domains only
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope: Verified domains only
     */
    public function scopeVerified($query)
    {
        return $query->whereNotNull('verified_at');
    }

    /**
     * Scope: Primary domains only
     */
    public function scopePrimary($query)
    {
        return $query->where('is_primary', true);
    }
}
