<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class SalesPage extends Model
{
    protected $fillable = [
        'user_id',
        'name',
        'meta_description',
        'slug',
        'html_content',
        'thumbnail_path',
        'legal_content',
        'faq_items',
        'order_bump_enabled',
        'order_bump_product_id',
        'order_bump_text',
        'order_bump_bg_color',
        'order_bump_text_color',
        'order_bump_border_style',
        'order_bump_border_color',
        'available_products',
        'custom_domain',
        'custom_domain_verified',
        'facebook_pixel_id',
        'google_analytics_id',
        'tiktok_pixel_id',
        'logo_path',
        'show_logo',
        'favicon_path',
        'use_logo_as_favicon',
        'primary_color',
        'is_active',
        'total_views',
        'total_orders',
    ];

    protected $casts = [
        'available_products' => 'array',
        'faq_items' => 'array',
        'custom_domain_verified' => 'boolean',
        'show_logo' => 'boolean',
        'use_logo_as_favicon' => 'boolean',
        'order_bump_enabled' => 'boolean',
        'is_active' => 'boolean',
        'total_views' => 'integer',
        'total_orders' => 'integer',
    ];

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($salesPage) {
            if (empty($salesPage->slug)) {
                $salesPage->slug = Str::slug($salesPage->name);

                // Ensure slug is unique
                $count = 1;
                $originalSlug = $salesPage->slug;
                while (static::where('slug', $salesPage->slug)->exists()) {
                    $salesPage->slug = $originalSlug . '-' . $count;
                    $count++;
                }
            }
        });
    }

    /**
     * Get the user that owns the sales page
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get all analytics for this sales page
     */
    public function analytics(): HasMany
    {
        return $this->hasMany(SalesPageAnalytics::class);
    }

    /**
     * Get all orders from this sales page
     */
    public function salesPageOrders(): HasMany
    {
        return $this->hasMany(SalesPageOrder::class);
    }

    /**
     * Get available products for this sales page
     */
    public function getAvailableProductsModels()
    {
        if (!$this->available_products || !is_array($this->available_products)) {
            return collect();
        }

        return Product::whereIn('id', $this->available_products)->get();
    }

    /**
     * Get the public URL for this sales page
     */
    public function getPublicUrl(): string
    {
        if ($this->custom_domain && $this->custom_domain_verified) {
            return 'https://' . $this->custom_domain;
        }

        $sellerSlug = $this->user ? $this->user->getUrlSlug() : 'seller';
        return url('/p/' . $sellerSlug . '/' . $this->slug);
    }

    /**
     * Increment view count
     */
    public function incrementViews()
    {
        $this->increment('total_views');
    }

    /**
     * Increment order count
     */
    public function incrementOrders()
    {
        $this->increment('total_orders');
    }

    /**
     * Get conversion rate
     */
    public function getConversionRate(): float
    {
        if ($this->total_views == 0) {
            return 0;
        }

        return round(($this->total_orders / $this->total_views) * 100, 2);
    }

    /**
     * Get the favicon path (either custom favicon or logo)
     */
    public function getFaviconPath(): ?string
    {
        if ($this->use_logo_as_favicon && $this->logo_path) {
            return $this->logo_path;
        }

        return $this->favicon_path;
    }

    /**
     * Get the order bump product
     */
    public function orderBumpProduct(): BelongsTo
    {
        return $this->belongsTo(Product::class, 'order_bump_product_id');
    }
}
