<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Helpers\PhoneHelper;
use App\Models\SystemSetting;

class Order extends Model
{
    protected $fillable = [
        'store_id', 'webhook_source_id', 'woo_order_id', 'order_number', 'global_order_id', 'status',
        'total', 'subtotal', 'shipping_total', 'discount_total', 'fee_total',
        'currency', 'payment_method', 'payment_method_title',
        'billing', 'shipping', 'line_items', 'meta_data',
        'fee_lines', 'shipping_lines', 'shipping_method_title',
        'tracking_number', 'courier', 'poslaju_response', 'awb_generated_at', 'awb_status',
        'poslaju_tracking_events', 'last_tracking_status', 'last_tracking_update',
        'delivery_completed_at', 'auto_completed_by_webhook',
        'whatsapp_sent_at', 'is_manual', 'created_by', 'notes',
        'date_created', 'date_modified'
    ];

    protected $casts = [
        'billing' => 'array',
        'shipping' => 'array',
        'line_items' => 'array',
        'meta_data' => 'array',
        'fee_lines' => 'array',
        'shipping_lines' => 'array',
        'poslaju_response' => 'array',
        'poslaju_tracking_events' => 'array',
        'date_created' => 'datetime',
        'date_modified' => 'datetime',
        'awb_generated_at' => 'datetime',
        'whatsapp_sent_at' => 'datetime',
        'last_tracking_update' => 'datetime',
        'delivery_completed_at' => 'datetime',
        'auto_completed_by_webhook' => 'boolean',
        'total' => 'decimal:2',
        'subtotal' => 'decimal:2',
        'shipping_total' => 'decimal:2',
        'discount_total' => 'decimal:2',
        'fee_total' => 'decimal:2'
    ];

    protected static function booted()
    {
        // Auto-generate global_order_id and format phone numbers
        static::creating(function ($order) {
            if (empty($order->global_order_id)) {
                $order->global_order_id = $order->generateGlobalOrderId();
            }
            $order->formatPhoneNumbers();
        });

        static::updating(function ($order) {
            $order->formatPhoneNumbers();
        });
    }

    /**
     * Format phone numbers in billing and shipping data
     */
    protected function formatPhoneNumbers()
    {
        // Format billing phone
        if (!empty($this->billing['phone'])) {
            $billing = $this->billing;
            $billing['phone'] = PhoneHelper::formatMalaysian($billing['phone']);
            $this->billing = $billing;
        }

        // Format shipping phone if exists
        if (!empty($this->shipping['phone'])) {
            $shipping = $this->shipping;
            $shipping['phone'] = PhoneHelper::formatMalaysian($shipping['phone']);
            $this->shipping = $shipping;
        }
    }

    public function generateGlobalOrderId()
    {
        // Get the prefix from system settings (default: OMS)
        $prefix = SystemSetting::getGlobalIdPrefix();

        // Get the last order ID number
        $lastOrder = Order::orderBy('id', 'desc')->first();
        $nextNumber = $lastOrder ? $lastOrder->id + 1 : 1;

        // Format: {PREFIX}000001, {PREFIX}000002, etc. (NO DASHES)
        // Auto-adjusts padding based on number size
        if ($nextNumber < 10000000) {
            // 8 digits: PREFIX00000001 to PREFIX09999999
            return $prefix . str_pad($nextNumber, 8, '0', STR_PAD_LEFT);
        } else {
            // 9+ digits as needed
            return $prefix . $nextNumber;
        }
    }

    public function store()
    {
        return $this->belongsTo(Store::class);
    }

    public function webhookSource()
    {
        return $this->belongsTo(WebhookSource::class);
    }

    public function logs()
    {
        return $this->hasMany(OrderLog::class);
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the sales page order (if this order came from sales page)
     */
    public function salesPageOrder()
    {
        return $this->hasOne(SalesPageOrder::class, 'order_id');
    }

    // Display formatted order ID
    public function getDisplayOrderIdAttribute()
    {
        return $this->global_order_id ?: $this->generateGlobalOrderId();
    }

    // Get color coding for status
    public function getStatusColorAttribute()
    {
        return match($this->status) {
            'processing' => 'yellow',
            'approval' => 'purple',
            'printed' => 'blue',
            'completed' => 'green',
            'pending' => 'gray',
            'on-hold' => 'orange',
            'cancelled' => 'red',
            'refunded' => 'purple',
            'failed' => 'red',
            default => 'gray'
        };
    }

    public function getPaymentColorAttribute()
    {
        return match($this->payment_method) {
            'cod' => 'amber',
            'bacs' => 'blue',
            'stripe' => 'indigo',
            'paypal' => 'yellow',
            'bayarcash-wc' => 'green',        // Online banking - very light green
            'duitnowqr-wc' => 'sky',          // DuitNow QR - light blue
            'linecredit-wc' => 'purple',      // Line credit
            default => 'gray'
        };
    }

    public function isCOD()
    {
        return $this->payment_method === 'cod';
    }

    public function isLocalPickup()
    {
        return str_contains($this->shipping['method'] ?? '', 'local_pickup');
    }
    
    // Search scope for finding orders across stores
    public function scopeSearchGlobalId($query, $searchTerm)
    {
        return $query->where('global_order_id', 'like', '%' . $searchTerm . '%');
    }

    // Get currency symbol based on currency code
    public function getCurrencySymbolAttribute()
    {
        return match(strtoupper($this->currency)) {
            'MYR' => 'RM',
            'USD' => '$',
            'EUR' => '€',
            'GBP' => '£',
            'JPY' => '¥',
            'CNY' => '¥',
            'SGD' => 'S$',
            'THB' => '฿',
            'IDR' => 'Rp',
            'PHP' => '₱',
            'VND' => '₫',
            'KRW' => '₩',
            'INR' => '₹',
            'AUD' => 'A$',
            'CAD' => 'C$',
            'HKD' => 'HK$',
            'NZD' => 'NZ$',
            default => $this->currency . ' ',
        };
    }

    // Format currency with symbol
    public function formatCurrency($amount = null)
    {
        $amount = $amount ?? $this->total;
        return $this->currency_symbol . number_format($amount, 2);
    }

    // Calculate line items subtotal
    public function getCalculatedSubtotalAttribute()
    {
        if ($this->subtotal > 0) {
            return $this->subtotal;
        }

        if (!is_array($this->line_items)) {
            return 0;
        }

        return collect($this->line_items)->sum(function ($item) {
            return floatval($item['total'] ?? 0);
        });
    }

    // Calculate fees (difference between total and subtotal)
    public function getCalculatedFeeAttribute()
    {
        if ($this->fee_total > 0) {
            return $this->fee_total;
        }

        $subtotal = $this->calculated_subtotal;
        $total = floatval($this->total);
        $shipping = floatval($this->shipping_total);

        return max(0, $total - $subtotal - $shipping);
    }

    /**
     * Get company information based on system setting
     * Returns seller info or admin info based on company_info_source setting
     */
    public function getCompanyInfo()
    {
        $source = SystemSetting::getCompanyInfoSource();

        if ($source === 'admin') {
            // Use admin (first admin user) company information
            return User::where('role', 'admin')->first() ?? auth()->user();
        } else {
            // Use seller company information (default behavior)
            return $this->store ? $this->store->getOwner() : auth()->user();
        }
    }

    /**
     * Check if a line item is a bundle and get its composition
     * Returns the bundle product and its items if it's a bundle, null otherwise
     */
    public function getBundleItemDetails($lineItem)
    {
        $product = null;

        // Try to find product by SKU first
        if (isset($lineItem['sku']) && !empty($lineItem['sku'])) {
            $product = Product::where('sku', $lineItem['sku'])->first();
        }

        // If not found by SKU, try by product name
        if (!$product && isset($lineItem['name']) && !empty($lineItem['name'])) {
            $product = Product::where('name', $lineItem['name'])->first();
        }

        // If still not found or not a bundle, return null
        if (!$product || !$product->isBundle() || !$product->bundle_items) {
            return null;
        }

        // Enrich bundle items with product names if only product_id or sku is stored
        $enrichedItems = collect($product->bundle_items)->map(function($item) {
            // If we have product_id, fetch the latest product info
            if (isset($item['product_id'])) {
                $bundleProduct = Product::find($item['product_id']);
                if ($bundleProduct) {
                    return [
                        'product_id' => $item['product_id'],
                        'sku' => $bundleProduct->sku,
                        'name' => $bundleProduct->name,
                        'quantity' => $item['quantity']
                    ];
                }
            }

            // Otherwise, return as-is (backward compatibility)
            return [
                'sku' => $item['sku'] ?? 'N/A',
                'name' => $item['name'] ?? $item['sku'] ?? 'N/A',
                'quantity' => $item['quantity']
            ];
        })->toArray();

        return [
            'product' => $product,
            'items' => $enrichedItems
        ];
    }

    /**
     * Get line items with bundle details expanded
     * Returns array with line items and their bundle breakdowns if applicable
     */
    public function getExpandedLineItems()
    {
        if (!is_array($this->line_items)) {
            return [];
        }

        $expandedItems = [];

        foreach ($this->line_items as $item) {
            $bundleDetails = $this->getBundleItemDetails($item);

            $expandedItems[] = [
                'item' => $item,
                'is_bundle' => $bundleDetails !== null,
                'bundle_details' => $bundleDetails
            ];
        }

        return $expandedItems;
    }
}