<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Models\User;

class MarketingStrategyInput extends Model
{
    protected $fillable = [
        'year',
        'month',
        'total_marketing_spend',
        'brand_ambassador_spend',
        'ads_spend',
        'social_media_spend',
        'events_spend',
        'other_spend',
        // Ads breakdown
        'ads_facebook',
        'ads_instagram',
        'ads_tiktok',
        'ads_google',
        'ads_youtube',
        'ads_twitter',
        'ads_linkedin',
        'ads_other',
        // Social media breakdown
        'social_facebook',
        'social_instagram',
        'social_tiktok',
        'social_youtube',
        'social_twitter',
        'social_linkedin',
        'social_other',
        // Revenue breakdown
        'revenue_checkout_form',
        'revenue_sales_pages',
        'revenue_webstore',
        'revenue_marketplace',
        'revenue_direct',
        'revenue_other',
        // Conversion tracking
        'conversions_checkout_form',
        'conversions_sales_pages',
        'conversions_webstore',
        // Other fields
        'campaign_details',
        'target_audience',
        'channels_used',
        'channel_notes',
        'actual_revenue',
        'roi_percentage',
        'ai_analysis',
        'ai_recommendations',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'total_marketing_spend' => 'decimal:2',
        'brand_ambassador_spend' => 'decimal:2',
        'ads_spend' => 'decimal:2',
        'social_media_spend' => 'decimal:2',
        'events_spend' => 'decimal:2',
        'other_spend' => 'decimal:2',
        // Ads breakdown
        'ads_facebook' => 'decimal:2',
        'ads_instagram' => 'decimal:2',
        'ads_tiktok' => 'decimal:2',
        'ads_google' => 'decimal:2',
        'ads_youtube' => 'decimal:2',
        'ads_twitter' => 'decimal:2',
        'ads_linkedin' => 'decimal:2',
        'ads_other' => 'decimal:2',
        // Social media breakdown
        'social_facebook' => 'decimal:2',
        'social_instagram' => 'decimal:2',
        'social_tiktok' => 'decimal:2',
        'social_youtube' => 'decimal:2',
        'social_twitter' => 'decimal:2',
        'social_linkedin' => 'decimal:2',
        'social_other' => 'decimal:2',
        // Revenue breakdown
        'revenue_checkout_form' => 'decimal:2',
        'revenue_sales_pages' => 'decimal:2',
        'revenue_webstore' => 'decimal:2',
        'revenue_marketplace' => 'decimal:2',
        'revenue_direct' => 'decimal:2',
        'revenue_other' => 'decimal:2',
        // Other
        'actual_revenue' => 'decimal:2',
        'roi_percentage' => 'decimal:2',
        // Integers
        'conversions_checkout_form' => 'integer',
        'conversions_sales_pages' => 'integer',
        'conversions_webstore' => 'integer',
    ];

    /**
     * Get the user who created this record
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the user who last updated this record
     */
    public function updater(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Calculate ROI percentage
     */
    public function calculateROI(): float
    {
        if ($this->total_marketing_spend <= 0) {
            return 0;
        }

        // ROI = ((Revenue - Marketing Spend) / Marketing Spend) * 100
        $profit = $this->actual_revenue - $this->total_marketing_spend;
        return ($profit / $this->total_marketing_spend) * 100;
    }

    /**
     * Update ROI percentage
     */
    public function updateROI(): void
    {
        $this->roi_percentage = $this->calculateROI();
        $this->save();
    }

    /**
     * Get total marketing spend (auto-calculated)
     */
    public function getTotalSpendAttribute(): float
    {
        return $this->brand_ambassador_spend
             + $this->ads_spend
             + $this->social_media_spend
             + $this->events_spend
             + $this->other_spend;
    }

    /**
     * Get month name
     */
    public function getMonthNameAttribute(): string
    {
        $months = [
            1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
            5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
            9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
        ];

        return $months[$this->month] ?? 'Unknown';
    }

    /**
     * Get channel breakdown as array
     */
    public function getChannelBreakdownAttribute(): array
    {
        return [
            'Brand Ambassadors' => $this->brand_ambassador_spend,
            'Ads' => $this->ads_spend,
            'Social Media' => $this->social_media_spend,
            'Events' => $this->events_spend,
            'Other' => $this->other_spend,
        ];
    }

    /**
     * Get best performing channel
     */
    public function getBestChannelAttribute(): string
    {
        $breakdown = $this->channel_breakdown;
        arsort($breakdown);
        return array_key_first($breakdown);
    }

    /**
     * Scope to filter by year
     */
    public function scopeForYear($query, $year)
    {
        return $query->where('year', $year);
    }

    /**
     * Scope to filter by year and month
     */
    public function scopeForPeriod($query, $year, $month)
    {
        return $query->where('year', $year)->where('month', $month);
    }

    /**
     * Scope to get records with good ROI
     */
    public function scopeGoodROI($query, $threshold = 100)
    {
        return $query->where('roi_percentage', '>=', $threshold);
    }
}
