<?php

namespace App\Jobs;

use App\Models\User;
use App\Models\Order;
use App\Services\GoogleSheetsService;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\Middleware\RateLimited;
use Illuminate\Support\Facades\Log;

class SyncOrderToGoogleSheets implements ShouldQueue
{
    use Queueable;

    public $tries = 5;
    public $timeout = 120;
    public $backoff = [30, 60, 120, 240, 480]; // Exponential backoff in seconds

    protected $user;
    protected $order;
    protected $batchSync;
    protected $limit;

    /**
     * Create a new job instance.
     *
     * @param User $user The user whose Google Sheet to sync to
     * @param Order|null $order Single order to sync (null for batch sync)
     * @param bool $batchSync Whether to sync all orders
     * @param int|null $limit Limit for batch sync
     */
    public function __construct(User $user, Order $order = null, bool $batchSync = false, int $limit = null)
    {
        $this->user = $user;
        $this->order = $order;
        $this->batchSync = $batchSync;
        $this->limit = $limit;
    }

    /**
     * Get the middleware the job should pass through.
     *
     * Note: Rate limiting is handled by delays in the GoogleSheetsService
     * (1.2 seconds between requests) to stay under Google's 60 req/min limit
     */
    public function middleware(): array
    {
        return [];
    }

    /**
     * Execute the job.
     */
    public function handle(GoogleSheetsService $googleSheetsService): void
    {
        // Check if user has Google Sheets enabled
        if (!$this->user->google_sheets_sync_enabled || !$this->user->google_sheet_id) {
            Log::info('Google Sheets sync skipped - not enabled', [
                'user_id' => $this->user->id
            ]);
            return;
        }

        try {
            if ($this->batchSync) {
                // Batch sync all orders
                $result = $googleSheetsService->batchSyncOrders($this->user, $this->limit);

                Log::info('Google Sheets batch sync completed', [
                    'user_id' => $this->user->id,
                    'synced' => $result['synced'],
                    'failed' => $result['failed'],
                    'total' => $result['total']
                ]);
            } elseif ($this->order) {
                // Sync single order
                $success = $googleSheetsService->syncOrder($this->order, $this->user);

                if ($success) {
                    Log::info('Order synced to Google Sheets', [
                        'user_id' => $this->user->id,
                        'order_id' => $this->order->id,
                        'order_number' => $this->order->order_number
                    ]);
                } else {
                    Log::warning('Failed to sync order to Google Sheets', [
                        'user_id' => $this->user->id,
                        'order_id' => $this->order->id
                    ]);
                }
            }
        } catch (\Exception $e) {
            Log::error('Google Sheets sync job failed', [
                'user_id' => $this->user->id,
                'order_id' => $this->order ? $this->order->id : null,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            // Re-throw to trigger retry
            throw $e;
        }
    }

    /**
     * Handle job failure
     */
    public function failed(\Throwable $exception): void
    {
        Log::error('Google Sheets sync job failed permanently', [
            'user_id' => $this->user->id,
            'order_id' => $this->order ? $this->order->id : null,
            'error' => $exception->getMessage()
        ]);
    }
}
