<?php

namespace App\Http\Controllers;

use App\Models\Store;
use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class WebhookController extends Controller
{
    /**
     * Handle incoming WooCommerce webhook
     */
    public function handle(Request $request, $storeId)
    {
        try {
            $store = Store::findOrFail($storeId);

            // Verify webhook signature
            if (!$this->verifyWebhookSignature($request, $store)) {
                Log::warning("Invalid webhook signature for store: {$store->name}", [
                    'expected_signature' => $this->getExpectedSignature($request, $store),
                    'received_signature' => $request->header('X-WC-Webhook-Signature'),
                    'has_secret' => !empty($store->webhook_secret)
                ]);
                // TEMPORARY: Don't reject, just log
                // return response()->json(['error' => 'Invalid signature'], 401);
            }

            // Get the webhook topic from header
            $topic = $request->header('X-WC-Webhook-Topic');

            // Get order data from payload
            $orderData = $request->all();

            Log::info("Webhook received for store: {$store->name}", [
                'topic' => $topic,
                'order_id' => $orderData['id'] ?? 'unknown'
            ]);

            // Process based on webhook topic
            switch ($topic) {
                case 'order.created':
                case 'order.updated':
                    $this->syncOrder($store, $orderData);
                    break;

                case 'order.deleted':
                    $this->deleteOrder($store, $orderData['id']);
                    break;

                default:
                    Log::info("Unhandled webhook topic: {$topic}");
            }

            return response()->json(['success' => true], 200);

        } catch (\Exception $e) {
            Log::error("Webhook processing error: " . $e->getMessage(), [
                'store_id' => $storeId,
                'error' => $e->getMessage()
            ]);

            return response()->json(['error' => 'Processing failed'], 500);
        }
    }

    /**
     * Verify webhook signature
     */
    protected function verifyWebhookSignature(Request $request, Store $store)
    {
        if (!$store->webhook_secret) {
            return true; // Skip verification if no secret set
        }

        $signature = $request->header('X-WC-Webhook-Signature');
        $payload = $request->getContent();

        $expectedSignature = base64_encode(hash_hmac('sha256', $payload, $store->webhook_secret, true));

        return hash_equals($expectedSignature, $signature);
    }

    /**
     * Get expected signature for debugging
     */
    protected function getExpectedSignature(Request $request, Store $store)
    {
        if (!$store->webhook_secret) {
            return 'no-secret-set';
        }

        $payload = $request->getContent();
        return base64_encode(hash_hmac('sha256', $payload, $store->webhook_secret, true));
    }

    /**
     * Sync order from webhook data
     */
    protected function syncOrder(Store $store, array $orderData)
    {
        $order = Order::updateOrCreate(
            [
                'store_id' => $store->id,
                'woo_order_id' => $orderData['id']
            ],
            [
                'order_number' => $orderData['number'],
                // global_order_id will auto-generate from Order model
                'status' => $orderData['status'],
                'total' => $orderData['total'],
                'currency' => $orderData['currency'],
                'payment_method' => $orderData['payment_method'] ?? null,
                'payment_method_title' => $orderData['payment_method_title'] ?? null,
                'billing' => $orderData['billing'] ?? [],
                'shipping' => $orderData['shipping'] ?? [],
                'line_items' => $orderData['line_items'] ?? [],
                'meta_data' => $orderData['meta_data'] ?? [],
                'date_created' => $orderData['date_created'] ?? now(),
                'date_modified' => $orderData['date_modified'] ?? now(),
            ]
        );

        Log::info("Order synced via webhook", [
            'store' => $store->name,
            'order_id' => $orderData['id'],
            'order_number' => $orderData['number'],
            'global_order_id' => $order->global_order_id
        ]);
    }

    /**
     * Delete order from webhook
     */
    protected function deleteOrder(Store $store, $wooOrderId)
    {
        Order::where('store_id', $store->id)
            ->where('woo_order_id', $wooOrderId)
            ->delete();

        Log::info("Order deleted via webhook", [
            'store' => $store->name,
            'woo_order_id' => $wooOrderId
        ]);
    }
}
