<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Store;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    public function index(Request $request)
    {
        $user = $request->user();
        $query = User::withCount(['stores', 'tiktokShops']);

        // Filter based on user role
        if ($user->isManager()) {
            // Managers can only see sellers they manage
            $sellerIds = $user->getManagedSellerIds();
            $query->whereIn('id', $sellerIds)->where('role', 'seller');
        } elseif ($user->isAdmin()) {
            // Admins can see all users
            // Apply role filter if specified
            if ($request->filled('role')) {
                $query->where('role', $request->role);
            }
        } else {
            // Sellers shouldn't access this page, redirect to dashboard
            return redirect()->route('dashboard')
                ->with('error', 'You do not have permission to access user management.');
        }

        $users = $query->orderBy('created_at', 'desc')->paginate(15);
        return view('users.index', compact('users'));
    }

    public function create()
    {
        $stores = Store::orderBy('name')->get();
        return view('users.create', compact('stores'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8|confirmed',
            'role' => 'required|in:admin,seller,manager',
            'store_ids' => 'array',
            'store_ids.*' => 'exists:stores,id',
            'seller_ids' => 'array',
            'seller_ids.*' => 'exists:users,id',
            'company_name' => 'nullable|string|max:255',
            'company_registration' => 'nullable|string|max:255',
            'company_address' => 'nullable|string',
            'company_city' => 'nullable|string|max:255',
            'company_state' => 'nullable|string|max:255',
            'company_postcode' => 'nullable|string|max:20',
            'company_country' => 'nullable|string|max:255',
            'company_phone' => 'nullable|string|max:50',
            'company_email' => 'nullable|email|max:255',
            'company_website' => 'nullable|url|max:255',
        ]);

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'role' => $validated['role'],
            'company_name' => $validated['company_name'] ?? null,
            'company_registration' => $validated['company_registration'] ?? null,
            'company_address' => $validated['company_address'] ?? null,
            'company_city' => $validated['company_city'] ?? null,
            'company_state' => $validated['company_state'] ?? null,
            'company_postcode' => $validated['company_postcode'] ?? null,
            'company_country' => $validated['company_country'] ?? null,
            'company_phone' => $validated['company_phone'] ?? null,
            'company_email' => $validated['company_email'] ?? null,
            'company_website' => $validated['company_website'] ?? null,
        ]);

        // Attach stores to seller only
        if ($validated['role'] === 'seller' && !empty($validated['store_ids'])) {
            $user->stores()->attach($validated['store_ids']);
        }

        // Attach sellers to manager
        if ($validated['role'] === 'manager' && !empty($validated['seller_ids'])) {
            $user->managedSellers()->attach($validated['seller_ids']);
        }

        return redirect()->route('users.index')
            ->with('success', 'User created successfully!');
    }

    public function edit(User $user)
    {
        $stores = Store::orderBy('name')->get();
        return view('users.edit', compact('user', 'stores'));
    }

    public function update(Request $request, User $user)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'email', Rule::unique('users', 'email')->ignore($user->id)],
            'password' => 'nullable|string|min:8|confirmed',
            'role' => 'required|in:admin,seller,manager',
            'store_ids' => 'array',
            'store_ids.*' => 'exists:stores,id',
            'seller_ids' => 'array',
            'seller_ids.*' => 'exists:users,id',
            'tiktok_shop_ids' => 'array',
            'tiktok_shop_ids.*' => 'exists:tiktok_shops,id',
            'company_name' => 'nullable|string|max:255',
            'company_registration' => 'nullable|string|max:255',
            'company_address' => 'nullable|string',
            'company_city' => 'nullable|string|max:255',
            'company_state' => 'nullable|string|max:255',
            'company_postcode' => 'nullable|string|max:20',
            'company_country' => 'nullable|string|max:255',
            'company_phone' => 'nullable|string|max:50',
            'company_email' => 'nullable|email|max:255',
            'company_website' => 'nullable|url|max:255',
        ]);

        $updateData = [
            'name' => $validated['name'],
            'email' => $validated['email'],
            'role' => $validated['role'],
            'company_name' => $validated['company_name'] ?? null,
            'company_registration' => $validated['company_registration'] ?? null,
            'company_address' => $validated['company_address'] ?? null,
            'company_city' => $validated['company_city'] ?? null,
            'company_state' => $validated['company_state'] ?? null,
            'company_postcode' => $validated['company_postcode'] ?? null,
            'company_country' => $validated['company_country'] ?? null,
            'company_phone' => $validated['company_phone'] ?? null,
            'company_email' => $validated['company_email'] ?? null,
            'company_website' => $validated['company_website'] ?? null,
        ];

        // Only update password if provided
        if (!empty($validated['password'])) {
            $updateData['password'] = Hash::make($validated['password']);
        }

        $user->update($updateData);

        // Sync stores for seller
        if ($validated['role'] === 'seller') {
            $user->stores()->sync($validated['store_ids'] ?? []);
            $user->managedSellers()->detach(); // Remove any manager relationships if switching to seller
            $user->tiktokShops()->detach(); // Remove TikTok shop assignments if switching to seller
        } elseif ($validated['role'] === 'manager') {
            // Managers don't have direct store assignments - they access stores through assigned sellers
            $user->stores()->detach();

            // Sync seller assignments if provided
            if (isset($validated['seller_ids'])) {
                $user->managedSellers()->sync($validated['seller_ids']);
            }

            // Sync TikTok Shops for managers ONLY if tiktok_shop_ids field is present in request
            if (array_key_exists('tiktok_shop_ids', $validated)) {
                $user->tiktokShops()->sync($validated['tiktok_shop_ids'] ?? []);
            }
        } else {
            // If changed to admin, remove all store associations
            $user->stores()->detach();
            $user->managedSellers()->detach();
            $user->tiktokShops()->detach();
        }

        return redirect()->route('users.index')
            ->with('success', 'User updated successfully!');
    }

    public function destroy(User $user)
    {
        // Prevent deleting yourself
        if ($user->id === auth()->id()) {
            return redirect()->route('users.index')
                ->with('error', 'You cannot delete your own account!');
        }

        $user->delete();
        return redirect()->route('users.index')
            ->with('success', 'User deleted successfully!');
    }

    /**
     * Impersonate a user (Admin only)
     */
    public function impersonate(User $user)
    {
        // Only admins can impersonate
        if (!auth()->user()->isAdmin()) {
            abort(403, 'Only administrators can impersonate users.');
        }

        // Cannot impersonate yourself
        if ($user->id === auth()->id()) {
            return redirect()->route('users.index')
                ->with('error', 'You cannot impersonate yourself!');
        }

        // Store the original admin user ID in session
        session(['impersonate_admin_id' => auth()->id()]);

        // Login as the target user
        auth()->login($user);

        return redirect()->route('home')
            ->with('success', "Now viewing as {$user->name}. You can help them manage their account.");
    }

    /**
     * Toggle TikTok Shop access for a manager
     */
    public function toggleTikTokAccess(User $user)
    {
        // Only admins can toggle TikTok Shop access
        if (!auth()->user()->isAdmin()) {
            abort(403, 'Only administrators can manage TikTok Shop access.');
        }

        // Only managers can have TikTok Shop access
        if ($user->role !== 'manager') {
            return redirect()->back()
                ->with('error', 'Only managers can be assigned TikTok Shop access.');
        }

        // Toggle the access
        $user->can_manage_tiktok_shop = !$user->can_manage_tiktok_shop;
        $user->save();

        $message = $user->can_manage_tiktok_shop
            ? "TikTok Shop access enabled for {$user->name}"
            : "TikTok Shop access disabled for {$user->name}";

        return redirect()->back()->with('success', $message);
    }

    /**
     * Stop impersonating and return to admin account
     */
    public function stopImpersonating()
    {
        // Check if currently impersonating
        if (!session()->has('impersonate_admin_id')) {
            return redirect()->route('home')
                ->with('error', 'You are not currently impersonating anyone.');
        }

        // Get the admin user ID
        $adminId = session('impersonate_admin_id');
        $admin = User::findOrFail($adminId);

        // Clear the impersonation session
        session()->forget('impersonate_admin_id');

        // Login back as admin
        auth()->login($admin);

        return redirect()->route('users.index')
            ->with('success', 'Stopped impersonating. Welcome back!');
    }
}
