<?php

namespace App\Http\Controllers;

use App\Models\TikTokTransaction;
use App\Models\TikTokAdsTransaction;
use App\Services\TikTokImportService;
use App\Services\TikTokAdsImportService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class TikTokShopController extends Controller
{
    protected $importService;
    protected $adsImportService;

    public function __construct(TikTokImportService $importService, TikTokAdsImportService $adsImportService)
    {
        $this->importService = $importService;
        $this->adsImportService = $adsImportService;
    }

    /**
     * Display TikTok Shop management page
     */
    public function index(Request $request)
    {
        $user = auth()->user();

        // Check permission: Only admin or manager with assigned TikTok shops can access
        if (!$user->isAdmin() && !($user->isManager() && $user->tiktokShops()->exists())) {
            abort(403, 'You do not have permission to access TikTok Shop data. No shops assigned.');
        }

        // Get date range (default: this month)
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->endOfMonth()->format('Y-m-d'));

        // Get available TikTok Shops for this user
        if ($user->isAdmin()) {
            $availableShops = \App\Models\TikTokShop::where('is_active', true)->orderBy('name')->get();
        } else {
            $availableShops = $user->tiktokShops()->where('is_active', true)->orderBy('name')->get();
        }

        // Get selected shop filter
        $selectedShopId = $request->input('shop_id');

        // Get transactions with pagination (filter by settled_date for accurate revenue tracking)
        $transactionsQuery = TikTokTransaction::with(['tiktokShop:id,name', 'uploader:id,name'])
            ->whereBetween('settled_date', [$startDate, $endDate]);

        // Filter by selected shop if specified
        if ($selectedShopId) {
            $transactionsQuery->where('tiktok_shop_id', $selectedShopId);
        } elseif ($user->isManager()) {
            // If no shop selected and user is manager, show all their assigned shops
            $shopIds = $user->getTikTokShopIds();
            $transactionsQuery->whereIn('tiktok_shop_id', $shopIds);
        }

        $transactions = $transactionsQuery
            ->orderBy('settled_date', 'desc')
            ->orderBy('id', 'desc')
            ->paginate(50);

        // Get summary stats (use settled_date)
        $summaryQuery = TikTokTransaction::selectRaw('
                COUNT(CASE WHEN type = "Order" THEN 1 END) as total_orders,
                COUNT(CASE WHEN type = "Refund" THEN 1 END) as total_refunds,
                SUM(CASE WHEN type = "Order" THEN total_revenue ELSE 0 END) as total_revenue,
                SUM(CASE WHEN type = "Refund" THEN ABS(total_revenue) ELSE 0 END) as total_refund_amount,
                SUM(total_fees) as total_fees,
                SUM(net_settlement) as total_settlement,
                AVG(CASE WHEN type = "Order" THEN total_revenue END) as avg_order_value
            ')
            ->whereBetween('settled_date', [$startDate, $endDate]);

        // Apply same shop filter as transactions
        if ($selectedShopId) {
            $summaryQuery->where('tiktok_shop_id', $selectedShopId);
        } elseif ($user->isManager()) {
            $shopIds = $user->getTikTokShopIds();
            $summaryQuery->whereIn('tiktok_shop_id', $shopIds);
        }

        $summary = $summaryQuery->first();

        // Get per-shop summary if multiple shops available and no specific shop selected
        $shopSummaries = null;
        if ($availableShops->count() > 1 && !$selectedShopId) {
            $shopSummaries = TikTokTransaction::selectRaw('
                    tiktok_shop_id,
                    COUNT(CASE WHEN type = "Order" THEN 1 END) as total_orders,
                    SUM(CASE WHEN type = "Order" THEN total_revenue ELSE 0 END) as total_revenue,
                    SUM(net_settlement) as total_settlement
                ')
                ->whereBetween('settled_date', [$startDate, $endDate])
                ->whereIn('tiktok_shop_id', $availableShops->pluck('id'))
                ->groupBy('tiktok_shop_id')
                ->get()
                ->keyBy('tiktok_shop_id');
        }

        // Get ads spending summary
        $adsSpending = TikTokAdsTransaction::getMonthlySummary();

        return view('tiktok-shop.index', compact('transactions', 'summary', 'startDate', 'endDate', 'adsSpending', 'availableShops', 'selectedShopId', 'shopSummaries'));
    }

    /**
     * Show upload form
     */
    public function uploadForm()
    {
        $user = auth()->user();

        // Get TikTok Shops available to this user
        if ($user->isAdmin()) {
            $tiktokShops = \App\Models\TikTokShop::active()->with('users')->orderBy('name')->get();
        } elseif ($user->isManager()) {
            $tiktokShops = $user->tiktokShops()->where('is_active', true)->orderBy('name')->get();
        } else {
            abort(403, 'You do not have permission to upload TikTok Shop data.');
        }

        // Block if no shops assigned
        if ($tiktokShops->isEmpty()) {
            abort(403, 'No TikTok Shops assigned to you. Please contact administrator.');
        }

        // Get all managers with TikTok shop access for "Upload as" dropdown (admin only)
        $managers = collect();
        if ($user->isAdmin()) {
            $managers = \App\Models\User::where('role', 'manager')
                ->where('can_manage_tiktok_shop', 1)
                ->with('tiktokShops')
                ->orderBy('name')
                ->get();
        }

        return view('tiktok-shop.upload', compact('tiktokShops', 'managers'));
    }

    /**
     * Handle file upload and preview
     */
    public function upload(Request $request)
    {
        // Validate file and shop selection
        $request->validate([
            'file' => 'required|file|max:10240', // Max 10MB
            'tiktok_shop_id' => 'required|exists:tiktok_shops,id',
            'upload_as' => 'nullable|exists:users,id', // Optional: admin can upload on behalf of someone
        ]);

        $user = auth()->user();
        $tiktokShopId = $request->input('tiktok_shop_id');
        $uploadAsUserId = $request->input('upload_as');

        // Verify user has permission to upload to this shop
        if (!$user->isAdmin()) {
            $userShopIds = $user->getTikTokShopIds();
            if (!in_array($tiktokShopId, $userShopIds)) {
                abort(403, 'You do not have permission to upload to this TikTok Shop.');
            }
        }

        // Determine who gets credit for this upload
        $uploadedBy = $user->id; // Default to current user
        if ($user->isAdmin() && $uploadAsUserId) {
            // Admin can upload on behalf of a manager
            // Verify the selected user has access to this shop
            $selectedUser = \App\Models\User::find($uploadAsUserId);
            if ($selectedUser && in_array($tiktokShopId, $selectedUser->getTikTokShopIds())) {
                $uploadedBy = $uploadAsUserId;
            }
        }

        try {
            // Store uploaded file temporarily
            $file = $request->file('file');

            // Check file extension
            $extension = strtolower($file->getClientOriginalExtension());
            if (!in_array($extension, ['xlsx', 'xls'])) {
                return back()->withErrors(['file' => 'The file must be an Excel file (.xlsx or .xls)'])->withInput();
            }
            $path = $file->store('temp/tiktok-imports');
            $fullPath = Storage::path($path);

            // Parse the Excel file
            $result = $this->importService->parseExcelFile($fullPath);

            // Store parse result in session for confirmation
            session([
                'tiktok_import_file' => $path,
                'tiktok_import_result' => $result,
                'tiktok_import_shop_id' => $tiktokShopId,
                'tiktok_import_uploaded_by' => $uploadedBy, // Store who gets credit
            ]);

            return redirect()->route('tiktok-shop.preview');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to parse Excel file: ' . $e->getMessage());
        }
    }

    /**
     * Preview parsed data before import
     */
    public function preview()
    {
        $result = session('tiktok_import_result');

        if (!$result) {
            return redirect()->route('tiktok-shop.upload')->with('error', 'No data to preview');
        }

        return view('tiktok-shop.preview', compact('result'));
    }

    /**
     * Confirm and import to database
     */
    public function import(Request $request)
    {
        $result = session('tiktok_import_result');
        $tiktokShopId = session('tiktok_import_shop_id');
        $uploadedBy = session('tiktok_import_uploaded_by', auth()->id()); // Get from session or fallback to current user

        if (!$result || !$tiktokShopId) {
            return redirect()->route('tiktok-shop.upload')->with('error', 'No data to import');
        }

        try {
            // Import to database
            $importResult = $this->importService->importToDatabase(
                $result['transactions'],
                $tiktokShopId,
                $uploadedBy
            );

            // Clean up temporary file
            $filePath = session('tiktok_import_file');
            if ($filePath && Storage::exists($filePath)) {
                Storage::delete($filePath);
            }

            // Clear session
            session()->forget(['tiktok_import_file', 'tiktok_import_result', 'tiktok_import_shop_id', 'tiktok_import_uploaded_by']);

            if ($importResult['success']) {
                $shop = \App\Models\TikTokShop::find($tiktokShopId);
                return redirect()->route('tiktok-shop.index')->with('success',
                    "Successfully imported {$importResult['imported']} transactions to {$shop->name}. " .
                    "Skipped {$importResult['duplicates']} duplicates."
                );
            } else {
                return back()->with('error', 'Import failed: ' . $importResult['error']);
            }
        } catch (\Exception $e) {
            return back()->with('error', 'Import failed: ' . $e->getMessage());
        }
    }

    /**
     * Delete a transaction
     */
    public function destroy($id)
    {
        $transaction = TikTokTransaction::findOrFail($id);

        // Check permission (admin or owner)
        if (!auth()->user()->isAdmin() && $transaction->user_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }

        $transaction->delete();

        return back()->with('success', 'Transaction deleted successfully');
    }

    /**
     * Get daily stats for chart (AJAX)
     */
    public function dailyStats(Request $request)
    {
        $user = auth()->user();

        // Check permission: Only admin or manager with assigned TikTok shops can access
        if (!$user->isAdmin() && !($user->isManager() && $user->tiktokShops()->exists())) {
            abort(403, 'You do not have permission to access TikTok Shop data.');
        }

        $startDate = $request->input('start_date', now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));

        $query = TikTokTransaction::selectRaw('
                order_date as date,
                COUNT(*) as total_orders,
                SUM(CASE WHEN type = "Order" THEN total_revenue ELSE 0 END) as revenue,
                SUM(CASE WHEN type = "Refund" THEN ABS(total_revenue) ELSE 0 END) as refunds,
                SUM(total_fees) as fees,
                SUM(net_settlement) as settlement
            ')
            ->whereBetween('order_date', [$startDate, $endDate]);

        // Filter by assigned TikTok Shops if manager
        if ($user->isManager()) {
            $shopIds = $user->getTikTokShopIds();
            $query->whereIn('tiktok_shop_id', $shopIds);
        }

        $stats = $query
            ->groupBy('order_date')
            ->orderBy('order_date', 'desc')
            ->get();

        return response()->json($stats);
    }

    // ========== ADS SPENDING METHODS ==========

    /**
     * Handle ads spending file upload and preview
     */
    public function uploadAds(Request $request)
    {
        // Validate file and shop selection
        $request->validate([
            'ads_file' => 'required|file|max:10240', // Max 10MB
            'tiktok_shop_id' => 'required|exists:tiktok_shops,id',
            'upload_as' => 'nullable|exists:users,id', // Optional: admin can upload on behalf of someone
        ]);

        $user = auth()->user();
        $tiktokShopId = $request->input('tiktok_shop_id');
        $uploadAsUserId = $request->input('upload_as');

        // Verify user has permission to upload to this shop
        if (!$user->isAdmin()) {
            $userShopIds = $user->getTikTokShopIds();
            if (!in_array($tiktokShopId, $userShopIds)) {
                abort(403, 'You do not have permission to upload to this TikTok Shop.');
            }
        }

        // Determine who gets credit for this upload
        $uploadedBy = $user->id; // Default to current user
        if ($user->isAdmin() && $uploadAsUserId) {
            // Admin can upload on behalf of a manager
            // Verify the selected user has access to this shop
            $selectedUser = \App\Models\User::find($uploadAsUserId);
            if ($selectedUser && in_array($tiktokShopId, $selectedUser->getTikTokShopIds())) {
                $uploadedBy = $uploadAsUserId;
            }
        }

        try {
            // Store uploaded file temporarily
            $file = $request->file('ads_file');

            // Check file extension
            $extension = strtolower($file->getClientOriginalExtension());
            if (!in_array($extension, ['xlsx', 'xls'])) {
                return back()->withErrors(['ads_file' => 'The file must be an Excel file (.xlsx or .xls)'])->withInput();
            }

            $path = $file->store('temp/tiktok-ads-imports');
            $fullPath = Storage::path($path);

            // Parse the Excel file
            $result = $this->adsImportService->parseExcelFile($fullPath);

            // Store parse result in session for confirmation
            session([
                'tiktok_ads_import_file' => $path,
                'tiktok_ads_import_result' => $result,
                'tiktok_ads_import_shop_id' => $tiktokShopId,
                'tiktok_ads_import_uploaded_by' => $uploadedBy, // Store who gets credit
            ]);

            return redirect()->route('tiktok-shop.ads.preview');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to parse ads spending file: ' . $e->getMessage());
        }
    }

    /**
     * Preview ads spending data before import
     */
    public function previewAds()
    {
        $result = session('tiktok_ads_import_result');

        if (!$result) {
            return redirect()->route('tiktok-shop.index')->with('error', 'No ads data to preview');
        }

        return view('tiktok-shop.ads-preview', compact('result'));
    }

    /**
     * Confirm and import ads spending to database
     */
    public function importAds(Request $request)
    {
        $result = session('tiktok_ads_import_result');
        $tiktokShopId = session('tiktok_ads_import_shop_id');
        $uploadedBy = session('tiktok_ads_import_uploaded_by', auth()->id()); // Get from session or fallback to current user

        if (!$result || !$tiktokShopId) {
            return redirect()->route('tiktok-shop.index')->with('error', 'No ads data to import');
        }

        try {
            // Import to database
            $importResult = $this->adsImportService->importTransactions(
                $result['transactions'],
                $tiktokShopId,
                $uploadedBy
            );

            // Clean up temporary file
            $filePath = session('tiktok_ads_import_file');
            if ($filePath && Storage::exists($filePath)) {
                Storage::delete($filePath);
            }

            // Clear session
            session()->forget(['tiktok_ads_import_file', 'tiktok_ads_import_result', 'tiktok_ads_import_shop_id', 'tiktok_ads_import_uploaded_by']);

            if ($importResult['success']) {
                $shop = \App\Models\TikTokShop::find($tiktokShopId);
                return redirect()->route('tiktok-shop.index')->with('success',
                    "Successfully imported {$importResult['imported']} ads transactions to {$shop->name}. " .
                    "Skipped {$importResult['duplicates']} duplicates."
                );
            } else {
                return back()->with('error', 'Import failed');
            }
        } catch (\Exception $e) {
            return back()->with('error', 'Import failed: ' . $e->getMessage());
        }
    }

    /**
     * Delete an ads transaction
     */
    public function destroyAds($id)
    {
        $transaction = TikTokAdsTransaction::findOrFail($id);

        // Check permission (admin or owner)
        if (!auth()->user()->isAdmin() && $transaction->user_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }

        $transaction->delete();

        return back()->with('success', 'Ads transaction deleted successfully');
    }
}
