<?php

namespace App\Http\Controllers;

use App\Models\Store;
use App\Services\WooCommerceService;
use Illuminate\Http\Request;

class StoreController extends Controller
{
    public function index(Request $request)
    {
        $user = $request->user();

        $query = Store::withCount('orders');

        // Role-based filtering
        if ($user->isSeller()) {
            // Sellers can only see their assigned stores
            $storeIds = $user->getStoreIds();
            $query->whereIn('id', $storeIds);
        } elseif ($user->isManager()) {
            // Managers can see their own stores + their sellers' stores
            $storeIds = $user->getStoreIds();
            $query->whereIn('id', $storeIds);
        }
        // Admin sees all stores (no filter)

        $stores = $query->paginate(10);
        return view('stores.index', compact('stores'));
    }

    public function create(Request $request)
    {
        // Managers cannot create stores (they only view their sellers' stores)
        if ($request->user()->isManager()) {
            abort(403, 'Managers cannot create stores. Please contact an administrator.');
        }

        return view('stores.create');
    }

    public function store(Request $request)
    {
        // Managers cannot create stores (they only view their sellers' stores)
        if ($request->user()->isManager()) {
            abort(403, 'Managers cannot create stores. Please contact an administrator.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'url' => 'required|url',
            'store_prefix' => 'nullable|string|max:10|unique:stores,store_prefix|regex:/^[A-Z0-9]+$/i',
            'consumer_key' => 'required|string',
            'consumer_secret' => 'required|string',
        ], [
        'store_prefix.unique' => 'This prefix is already used by another store.',
        'store_prefix.regex' => 'Prefix can only contain letters and numbers.',
        ]);

        // If no prefix provided, it will be auto-generated
        if (empty($validated['store_prefix'])) {
        unset($validated['store_prefix']);
        }

        $store = Store::create($validated);

        // If seller creates a store, automatically assign it to them
        if ($request->user()->isSeller()) {
            $request->user()->stores()->attach($store->id);
        }

        return redirect()->route('stores.index')
            ->with('success', 'Store added successfully!');
    }

    public function show(Request $request, Store $store)
    {
        $user = $request->user();

        // Check access: Sellers and Managers can only access their assigned stores
        if (!$user->isAdmin() && !in_array($store->id, $user->getStoreIds())) {
            abort(403, 'You do not have access to this store.');
        }

        $store->loadCount('orders');
        $recentOrders = $store->orders()->latest()->limit(10)->get();
        return view('stores.show', compact('store', 'recentOrders'));
    }

    public function edit(Request $request, Store $store)
    {
        $user = $request->user();

        // Check access: Sellers and Managers can only access their assigned stores
        if (!$user->isAdmin() && !in_array($store->id, $user->getStoreIds())) {
            abort(403, 'You do not have access to this store.');
        }

        return view('stores.edit', compact('store'));
    }

    public function update(Request $request, Store $store)
    {
        $user = $request->user();

        // Check access: Sellers and Managers can only access their assigned stores
        if (!$user->isAdmin() && !in_array($store->id, $user->getStoreIds())) {
            abort(403, 'You do not have access to this store.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'url' => 'required|url',
            'consumer_key' => 'required|string',
            'consumer_secret' => 'required|string',
            'is_active' => 'boolean',
        ]);

        // Handle checkbox - if not present, set to false
        $validated['is_active'] = $request->has('is_active');

        $store->update($validated);

        return redirect()->route('stores.index')
            ->with('success', 'Store updated successfully!');
    }

    public function destroy(Store $store)
    {
        $store->delete();
        return redirect()->route('stores.index')
            ->with('success', 'Store deleted successfully!');
    }

    public function sync(Request $request, Store $store)
    {
        $user = $request->user();

        // Check access: Sellers and Managers can only access their assigned stores
        if (!$user->isAdmin() && !in_array($store->id, $user->getStoreIds())) {
            abort(403, 'You do not have access to this store.');
        }

        try {
            $service = new WooCommerceService($store);
            $syncType = $request->input('sync_type', 'recent');
            $count = 0;

            switch ($syncType) {
                case 'current_month':
                    // Sync only current month orders
                    $startOfMonth = now()->startOfMonth()->toISOString();
                    $count = $service->syncOrders(true, $startOfMonth);
                    $message = "Synced {$count} orders from current month ({$store->name})";
                    break;

                case 'recent':
                    // Sync last 30 days (default)
                    $count = $service->syncOrders(false);
                    $message = "Synced {$count} recent orders ({$store->name})";
                    break;

                case 'historical':
                    // Sync all historical orders (last 12 months)
                    $count = $service->syncHistoricalOrders(12);
                    $message = "Synced {$count} historical orders ({$store->name})";
                    break;

                default:
                    $count = $service->syncOrders(false);
                    $message = "Synced {$count} orders from {$store->name}";
            }

            return back()->with('success', $message);
        } catch (\Exception $e) {
            return back()->with('error', "Failed to sync orders: " . $e->getMessage());
        }
    }

    public function enableWebhooks(Store $store)
    {
        try {
            $service = new WooCommerceService($store);

            // Generate webhook secret
            $secret = bin2hex(random_bytes(32));

            // Get webhook URL
            $webhookUrl = route('webhook.handle', ['store' => $store->id]);

            // Create webhooks on WooCommerce
            $webhookIds = $service->createWebhooks($webhookUrl, $secret);

            if (empty($webhookIds)) {
                return back()->with('error', 'Failed to create webhooks. Check logs for details.');
            }

            // Save webhook data to store
            $store->update([
                'webhook_secret' => $secret,
                'webhook_ids' => $webhookIds,
                'webhooks_enabled' => true
            ]);

            return back()->with('success', "Webhooks enabled for {$store->name}. Real-time sync is now active!");
        } catch (\Exception $e) {
            return back()->with('error', "Failed to enable webhooks: " . $e->getMessage());
        }
    }

    public function disableWebhooks(Store $store)
    {
        try {
            if ($store->webhook_ids) {
                $service = new WooCommerceService($store);
                $service->deleteWebhooks($store->webhook_ids);
            }

            $store->update([
                'webhook_secret' => null,
                'webhook_ids' => null,
                'webhooks_enabled' => false
            ]);

            return back()->with('success', "Webhooks disabled for {$store->name}");
        } catch (\Exception $e) {
            return back()->with('error', "Failed to disable webhooks: " . $e->getMessage());
        }
    }
}