<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\SellerForecast;
use App\Models\User;
use App\Models\Order;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class SellerKpiController extends Controller
{
    /**
     * Display all sellers with their KPIs
     */
    public function index(Request $request)
    {
        $year = $request->input('year', date('Y'));
        $month = $request->input('month', date('n'));
        $grade = $request->input('grade');

        // Get all sellers/managers
        $query = SellerForecast::with('user')
            ->forYear($year)
            ->where('month', $month);

        if ($grade) {
            $query->byGrade($grade);
        }

        $forecasts = $query->get();

        // Enrich forecasts with cumulative team sales for managers (for display only)
        $enrichedForecasts = $forecasts->map(function($forecast) use ($year, $month) {
            if ($forecast->user && $forecast->user->isManager()) {
                // Get team members sales
                $teamMembers = $forecast->user->managedSellers;
                $teamSales = 0;

                foreach ($teamMembers as $member) {
                    // Get member's orders (created by member only)
                    $memberOrdersTotal = Order::where('created_by', $member->id)
                        ->whereNotIn('status', ['cancelled', 'failed'])
                        ->whereYear('created_at', $year)
                        ->whereMonth('created_at', $month)
                        ->sum('total');

                    // Get member's TikTok sales
                    $memberTiktokTotal = \App\Models\TikTokTransaction::where('uploaded_by', $member->id)
                        ->whereYear('settled_date', $year)
                        ->whereMonth('settled_date', $month)
                        ->where('type', 'Order')
                        ->sum('total_revenue');

                    $teamSales += ($memberOrdersTotal + $memberTiktokTotal);
                }

                // Add cumulative sales (manager's direct + team)
                $forecast->cumulative_sales = $forecast->actual_sales + $teamSales;
                $forecast->team_sales = $teamSales;

                // Recalculate achievement and grade based on cumulative
                if ($forecast->target_sales > 0) {
                    $forecast->cumulative_achievement = ($forecast->cumulative_sales / $forecast->target_sales) * 100;
                } else {
                    $forecast->cumulative_achievement = 0;
                }

                // Recalculate grade based on cumulative achievement
                $achievement = $forecast->cumulative_achievement;
                if ($achievement >= 100) {
                    $forecast->cumulative_grade = 'A';
                } elseif ($achievement >= 80) {
                    $forecast->cumulative_grade = 'B';
                } elseif ($achievement >= 60) {
                    $forecast->cumulative_grade = 'C';
                } elseif ($achievement >= 40) {
                    $forecast->cumulative_grade = 'D';
                } else {
                    $forecast->cumulative_grade = 'F';
                }
            } else {
                // For sellers, cumulative = actual (no team)
                $forecast->cumulative_sales = $forecast->actual_sales;
                $forecast->team_sales = 0;
                $forecast->cumulative_achievement = $forecast->achievement_percentage;
                $forecast->cumulative_grade = $forecast->performance_grade;
            }

            return $forecast;
        });

        // Sort by cumulative sales (display sales) descending
        $enrichedForecasts = $enrichedForecasts->sortByDesc('cumulative_sales');

        // Get performance summary
        // IMPORTANT: Total Sales = only direct sales (excludes manager team sales)
        $summary = [
            'total_sellers' => $enrichedForecasts->count(),
            'avg_achievement' => $enrichedForecasts->avg('cumulative_achievement'),
            'total_sales' => $enrichedForecasts->sum('actual_sales'), // Direct sales only!
            'total_target' => $enrichedForecasts->sum('target_sales'),
            'grades' => [
                'A' => $enrichedForecasts->where('cumulative_grade', 'A')->count(),
                'B' => $enrichedForecasts->where('cumulative_grade', 'B')->count(),
                'C' => $enrichedForecasts->where('cumulative_grade', 'C')->count(),
                'D' => $enrichedForecasts->where('cumulative_grade', 'D')->count(),
                'F' => $enrichedForecasts->where('cumulative_grade', 'F')->count(),
            ]
        ];

        return view('forecasting.seller-kpi.index', compact('enrichedForecasts', 'summary', 'year', 'month'));
    }

    /**
     * Show individual seller's KPI dashboard
     */
    public function show($userId, Request $request)
    {
        $user = User::findOrFail($userId);
        $year = $request->input('year', date('Y'));
        $selectedMonth = $request->input('month', date('n')); // For team breakdown filtering

        // Get all months data for the seller
        $forecasts = SellerForecast::forUser($userId)
            ->forYear($year)
            ->orderBy('month')
            ->get();

        // Get year-to-date stats
        $ytdStats = [
            'total_sales' => $forecasts->sum('actual_sales'),
            'total_target' => $forecasts->sum('target_sales'),
            'total_orders' => $forecasts->sum('orders_count'),
            'avg_aov' => $forecasts->where('orders_count', '>', 0)->avg('aov'),
            'avg_achievement' => $forecasts->avg('achievement_percentage'),
            'best_month' => $forecasts->sortByDesc('actual_sales')->first(),
            'worst_month' => $forecasts->where('actual_sales', '>', 0)->sortBy('actual_sales')->first(),
        ];

        // Get recent orders (created by user only)
        $recentOrders = Order::where('created_by', $userId)
            ->whereNotIn('status', ['cancelled', 'failed'])
            ->whereYear('created_at', $year)
            ->orderByDesc('created_at')
            ->take(10)
            ->get()
            ->map(function($order) {
                // Add total_price attribute for compatibility with view
                $order->total_price = $order->total;
                return $order;
            });

        // Get sales breakdown by channel for this user
        $salesBreakdown = $this->getSalesBreakdown($userId, $year);

        // Enhanced manager data
        $teamBreakdown = null;
        $teamMonthlyPerformance = null;
        $teamYtdSummary = null;
        $teamBreakdownByMonth = null;

        if ($user->isManager()) {
            // Get YTD team summary
            $teamYtdSummary = $this->getTeamYtdSummary($userId, $year);

            // Get monthly team performance (all months)
            $teamMonthlyPerformance = $this->getTeamMonthlyPerformance($userId, $year);

            // Get detailed team breakdown for selected month
            $teamBreakdownByMonth = $this->getTeamBreakdownByMonth($userId, $year, $selectedMonth);
        }

        return view('forecasting.seller-kpi.show', compact(
            'user',
            'forecasts',
            'ytdStats',
            'recentOrders',
            'year',
            'selectedMonth',
            'salesBreakdown',
            'teamYtdSummary',
            'teamMonthlyPerformance',
            'teamBreakdownByMonth'
        ));
    }

    /**
     * Get sales breakdown by channel (Orders table + TikTok)
     */
    private function getSalesBreakdown($userId, $year)
    {
        // Get orders from Orders table (created by user only)
        $ordersTotal = Order::where('created_by', $userId)
            ->whereNotIn('status', ['cancelled', 'failed'])
            ->whereYear('created_at', $year)
            ->sum('total');

        $ordersCount = Order::where('created_by', $userId)
            ->whereNotIn('status', ['cancelled', 'failed'])
            ->whereYear('created_at', $year)
            ->count();

        // Get TikTok Shop sales
        $tiktokTotal = \App\Models\TikTokTransaction::where('uploaded_by', $userId)
            ->whereYear('settled_date', $year)
            ->where('type', 'Order')
            ->sum('total_revenue');

        $tiktokCount = \App\Models\TikTokTransaction::where('uploaded_by', $userId)
            ->whereYear('settled_date', $year)
            ->where('type', 'Order')
            ->count();

        return [
            'orders' => [
                'total' => $ordersTotal,
                'count' => $ordersCount,
                'label' => 'Orders (Checkout/Sales Pages)'
            ],
            'tiktok' => [
                'total' => $tiktokTotal,
                'count' => $tiktokCount,
                'label' => 'TikTok Shop'
            ],
            'grand_total' => $ordersTotal + $tiktokTotal
        ];
    }

    /**
     * Get team YTD summary (for managers)
     */
    private function getTeamYtdSummary($managerId, $year)
    {
        $manager = User::find($managerId);
        $teamMembers = $manager->managedSellers;

        $totalTeamSales = 0;
        $totalTeamOrders = 0;
        $teamMemberStats = [];

        foreach ($teamMembers as $member) {
            // Get member's YTD sales (created by member only)
            $ordersTotal = Order::where('created_by', $member->id)
                ->whereNotIn('status', ['cancelled', 'failed'])
                ->whereYear('created_at', $year)
                ->sum('total');

            $ordersCount = Order::where('created_by', $member->id)
                ->whereNotIn('status', ['cancelled', 'failed'])
                ->whereYear('created_at', $year)
                ->count();

            $tiktokTotal = \App\Models\TikTokTransaction::where('uploaded_by', $member->id)
                ->whereYear('settled_date', $year)
                ->where('type', 'Order')
                ->sum('total_revenue');

            $tiktokCount = \App\Models\TikTokTransaction::where('uploaded_by', $member->id)
                ->whereYear('settled_date', $year)
                ->where('type', 'Order')
                ->count();

            $memberTotal = $ordersTotal + $tiktokTotal;
            $memberOrders = $ordersCount + $tiktokCount;

            $totalTeamSales += $memberTotal;
            $totalTeamOrders += $memberOrders;

            if ($memberTotal > 0) {
                $teamMemberStats[] = [
                    'name' => $member->name,
                    'total' => $memberTotal,
                    'orders' => $memberOrders
                ];
            }
        }

        // Find top performer
        $topPerformer = collect($teamMemberStats)->sortByDesc('total')->first();

        return [
            'total_sales' => $totalTeamSales,
            'total_orders' => $totalTeamOrders,
            'avg_order_value' => $totalTeamOrders > 0 ? ($totalTeamSales / $totalTeamOrders) : 0,
            'team_size' => count($teamMemberStats),
            'top_performer' => $topPerformer
        ];
    }

    /**
     * Get monthly team performance (for managers)
     */
    private function getTeamMonthlyPerformance($managerId, $year)
    {
        $manager = User::find($managerId);
        $teamMembers = $manager->managedSellers;
        $teamMemberIds = $teamMembers->pluck('id')->toArray();

        $monthlyData = [];

        for ($month = 1; $month <= 12; $month++) {
            // Get team orders for this month (created by team members only)
            $ordersTotal = Order::whereIn('created_by', $teamMemberIds)
                ->whereNotIn('status', ['cancelled', 'failed'])
                ->whereYear('created_at', $year)
                ->whereMonth('created_at', $month)
                ->sum('total');

            $ordersCount = Order::whereIn('created_by', $teamMemberIds)
                ->whereNotIn('status', ['cancelled', 'failed'])
                ->whereYear('created_at', $year)
                ->whereMonth('created_at', $month)
                ->count();

            // Get team TikTok sales for this month
            $tiktokTotal = \App\Models\TikTokTransaction::whereIn('uploaded_by', $teamMemberIds)
                ->whereYear('settled_date', $year)
                ->whereMonth('settled_date', $month)
                ->where('type', 'Order')
                ->sum('total_revenue');

            $tiktokCount = \App\Models\TikTokTransaction::whereIn('uploaded_by', $teamMemberIds)
                ->whereYear('settled_date', $year)
                ->whereMonth('settled_date', $month)
                ->where('type', 'Order')
                ->count();

            $monthTotal = $ordersTotal + $tiktokTotal;
            $monthOrders = $ordersCount + $tiktokCount;

            if ($monthTotal > 0 || $monthOrders > 0) {
                $monthlyData[] = [
                    'month' => $month,
                    'month_name' => date('F', mktime(0, 0, 0, $month, 1)),
                    'total_sales' => $monthTotal,
                    'total_orders' => $monthOrders,
                    'orders_sales' => $ordersTotal,
                    'tiktok_sales' => $tiktokTotal
                ];
            }
        }

        return $monthlyData;
    }

    /**
     * Get team breakdown by specific month (for managers)
     */
    private function getTeamBreakdownByMonth($managerId, $year, $month)
    {
        $manager = User::find($managerId);
        $teamMembers = $manager->managedSellers;

        $breakdown = [];

        foreach ($teamMembers as $member) {
            // Get member's orders for specific month (created by member only)
            $ordersTotal = Order::where('created_by', $member->id)
                ->whereNotIn('status', ['cancelled', 'failed'])
                ->whereYear('created_at', $year)
                ->whereMonth('created_at', $month)
                ->sum('total');

            $ordersCount = Order::where('created_by', $member->id)
                ->whereNotIn('status', ['cancelled', 'failed'])
                ->whereYear('created_at', $year)
                ->whereMonth('created_at', $month)
                ->count();

            // Get member's TikTok sales for specific month
            $tiktokTotal = \App\Models\TikTokTransaction::where('uploaded_by', $member->id)
                ->whereYear('settled_date', $year)
                ->whereMonth('settled_date', $month)
                ->where('type', 'Order')
                ->sum('total_revenue');

            $tiktokCount = \App\Models\TikTokTransaction::where('uploaded_by', $member->id)
                ->whereYear('settled_date', $year)
                ->whereMonth('settled_date', $month)
                ->where('type', 'Order')
                ->count();

            $memberTotal = $ordersTotal + $tiktokTotal;
            $memberOrders = $ordersCount + $tiktokCount;

            if ($memberTotal > 0 || $memberOrders > 0) {
                $breakdown[] = [
                    'name' => $member->name,
                    'orders_total' => $ordersTotal,
                    'orders_count' => $ordersCount,
                    'tiktok_total' => $tiktokTotal,
                    'tiktok_count' => $tiktokCount,
                    'total' => $memberTotal,
                    'total_orders' => $memberOrders
                ];
            }
        }

        // Sort by total descending
        usort($breakdown, function($a, $b) {
            return $b['total'] <=> $a['total'];
        });

        return $breakdown;
    }

    /**
     * Generate Prophet forecast for individual seller
     */
    public function generateForecast(Request $request)
    {
        $userId = $request->input('user_id');
        $periods = $request->input('periods', 6);

        // Get seller's historical data
        $historicalData = SellerForecast::forUser($userId)
            ->where('actual_sales', '>', 0)
            ->orderBy('year')
            ->orderBy('month')
            ->get()
            ->map(function ($record) {
                return [
                    'ds' => Carbon::create($record->year, $record->month, 1)->format('Y-m-d'),
                    'y' => (float) $record->actual_sales
                ];
            })
            ->toArray();

        if (count($historicalData) < 3) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient data. Need at least 3 months of sales history.'
            ], 400);
        }

        // Call Python Prophet script
        $pythonPath = 'C:\\laragon\\bin\\python\\python-3.13\\python.exe';
        $scriptPath = base_path('forecast_prophet.py');

        // Create temp file with data
        $tempFile = storage_path('app/forecast_seller_temp_' . $userId . '.json');
        file_put_contents($tempFile, json_encode($historicalData));

        // Execute Python script
        $command = $pythonPath . " " . escapeshellarg($scriptPath) . " " . $periods . " < " . escapeshellarg($tempFile) . " 2>&1";
        $output = shell_exec($command);

        // Clean up temp file
        @unlink($tempFile);

        // Parse output
        $result = json_decode($output, true);

        if (!$result || !isset($result['success']) || !$result['success']) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to generate forecast',
                'debug' => $output
            ], 500);
        }

        // Store forecasts in database
        foreach ($result['forecast'] as $prediction) {
            $date = Carbon::parse($prediction['ds']);

            SellerForecast::updateOrCreate(
                [
                    'user_id' => $userId,
                    'year' => $date->year,
                    'month' => $date->month,
                ],
                [
                    'forecast_sales' => $prediction['yhat'],
                    'forecast_lower' => $prediction['yhat_lower'],
                    'forecast_upper' => $prediction['yhat_upper'],
                    'trend' => $result['trend']
                ]
            );
        }

        return response()->json([
            'success' => true,
            'forecast' => $result['forecast'],
            'trend' => $result['trend'],
            'confidence' => $result['confidence']
        ]);
    }

    /**
     * Team performance dashboard (for managers)
     */
    public function teamPerformance(Request $request)
    {
        $year = $request->input('year', date('Y'));
        $month = $request->input('month', date('n'));
        $managerId = $request->user()->id;

        // Get team members (assuming there's a manager_id field in users table)
        // Adjust this query based on your actual team structure
        $teamMembers = User::where('manager_id', $managerId)
            ->orWhere('id', $managerId)
            ->pluck('id');

        // Get forecasts for team
        $teamForecasts = SellerForecast::whereIn('user_id', $teamMembers)
            ->forYear($year)
            ->where('month', $month)
            ->with('user')
            ->orderByDesc('actual_sales')
            ->get();

        // Calculate team summary
        $teamSummary = [
            'total_members' => $teamForecasts->count(),
            'total_sales' => $teamForecasts->sum('actual_sales'),
            'total_target' => $teamForecasts->sum('target_sales'),
            'team_achievement' => $teamForecasts->sum('target_sales') > 0
                ? ($teamForecasts->sum('actual_sales') / $teamForecasts->sum('target_sales')) * 100
                : 0,
            'top_performer' => $teamForecasts->sortByDesc('achievement_percentage')->first(),
            'needs_support' => $teamForecasts->where('performance_grade', 'F')->values(),
        ];

        return view('forecasting.seller-kpi.team', compact('teamForecasts', 'teamSummary', 'year', 'month'));
    }

    /**
     * Sync actual sales from orders table and TikTok Shop
     * For managers: Only count their DIRECT sales (not team sales)
     * For sellers: Count all their sales from all channels
     */
    public function syncActualSales(Request $request)
    {
        try {
            $year = $request->input('year', date('Y'));
            $month = $request->input('month', date('n'));

            // Get all users (sellers and managers)
            $users = User::whereIn('role', ['seller', 'manager'])->get();
            $syncedCount = 0;

            foreach ($users as $user) {
                $totalSales = 0;
                $totalOrders = 0;

                // Get user's own orders ONLY (created by this user)
                // Excludes WooCommerce store orders to avoid calculation issues
                $orderStats = Order::where('created_by', $user->id)
                    ->whereNotIn('status', ['cancelled', 'failed'])
                    ->whereYear('created_at', $year)
                    ->whereMonth('created_at', $month)
                    ->selectRaw('
                        SUM(total) as total_sales,
                        COUNT(*) as orders_count
                    ')
                    ->first();

            // Get user's own TikTok sales
            $tiktokStats = \App\Models\TikTokTransaction::where('uploaded_by', $user->id)
                ->whereYear('settled_date', $year)
                ->whereMonth('settled_date', $month)
                ->where('type', 'Order')
                ->selectRaw('
                    SUM(total_revenue) as total_sales,
                    COUNT(*) as orders_count
                ')
                ->first();

            $totalSales = ($orderStats->total_sales ?? 0) + ($tiktokStats->total_sales ?? 0);
            $totalOrders = ($orderStats->orders_count ?? 0) + ($tiktokStats->orders_count ?? 0);

            // ALWAYS update/create forecast record (even if zero sales)
            // This ensures managers with no direct sales get their KPI set to 0
            // Get or create forecast record
            $forecast = SellerForecast::firstOrNew([
                'user_id' => $user->id,
                'year' => $year,
                'month' => $month,
            ]);

            // Update with actual sales (could be 0)
            $forecast->actual_sales = $totalSales;
            $forecast->orders_count = $totalOrders;
            $forecast->aov = $totalOrders > 0 ? ($totalSales / $totalOrders) : 0;

            // Update performance metrics
            $forecast->save();
            $forecast->updatePerformanceMetrics();

                $syncedCount++;
            }

            return response()->json([
                'success' => true,
                'message' => "Synced sales data for {$syncedCount} sellers/managers (direct sales: checkout forms, sales pages, TikTok)",
                'count' => $syncedCount
            ]);

        } catch (\Exception $e) {
            \Log::error('Sync actual sales error', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error syncing sales data: ' . $e->getMessage(),
                'error' => config('app.debug') ? [
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'trace' => explode("\n", $e->getTraceAsString())
                ] : null
            ], 500);
        }
    }

    /**
     * Show targets setting page
     */
    public function showSetTargets(Request $request)
    {
        $year = $request->input('year', date('Y'));
        $month = $request->input('month', date('n'));

        // Get all sellers (you may want to filter by role)
        $sellers = User::where('role', 'seller')
            ->orWhere('role', 'manager')
            ->orderBy('name')
            ->get();

        // Get existing forecasts for this period
        $existingForecasts = SellerForecast::where('year', $year)
            ->where('month', $month)
            ->get();

        return view('forecasting.seller-kpi.set-targets', compact('sellers', 'existingForecasts', 'year', 'month'));
    }

    /**
     * Save targets for sellers
     */
    public function saveTargets(Request $request)
    {
        $request->validate([
            'year' => 'required|integer',
            'month' => 'required|integer|between:1,12',
            'targets' => 'required|array',
            'targets.*.user_id' => 'required|exists:users,id',
            'targets.*.target_sales' => 'required|numeric|min:0',
        ]);

        $year = $request->year;
        $month = $request->month;
        $updatedCount = 0;

        foreach ($request->targets as $targetData) {
            $forecast = SellerForecast::updateOrCreate(
                [
                    'user_id' => $targetData['user_id'],
                    'year' => $year,
                    'month' => $month,
                ],
                [
                    'target_sales' => $targetData['target_sales']
                ]
            );

            // Update performance metrics if actual sales exist
            if ($forecast->actual_sales > 0) {
                $forecast->updatePerformanceMetrics();
            }

            $updatedCount++;
        }

        return redirect()
            ->route('forecasting.seller-kpi.set-targets', ['year' => $year, 'month' => $month])
            ->with('success', "Successfully updated targets for {$updatedCount} sellers! Go back to Seller KPIs and sync to see updated grades.");
    }

    /**
     * Set targets for sellers (API endpoint)
     */
    public function setTargets(Request $request)
    {
        $request->validate([
            'targets' => 'required|array',
            'targets.*.user_id' => 'required|exists:users,id',
            'targets.*.year' => 'required|integer',
            'targets.*.month' => 'required|integer|between:1,12',
            'targets.*.target_sales' => 'required|numeric|min:0',
        ]);

        $updatedCount = 0;

        foreach ($request->targets as $targetData) {
            SellerForecast::updateOrCreate(
                [
                    'user_id' => $targetData['user_id'],
                    'year' => $targetData['year'],
                    'month' => $targetData['month'],
                ],
                [
                    'target_sales' => $targetData['target_sales']
                ]
            );

            $updatedCount++;
        }

        return response()->json([
            'success' => true,
            'message' => "Updated targets for {$updatedCount} sellers",
            'count' => $updatedCount
        ]);
    }

    /**
     * Update manual data (Actual Sales & Orders) for past data entry
     */
    public function updateManualData(Request $request)
    {
        $request->validate([
            'updates' => 'required|array',
            'updates.*.id' => 'required|exists:seller_forecasts,id',
            'updates.*.actual_sales' => 'nullable|numeric|min:0',
            'updates.*.orders_count' => 'nullable|integer|min:0',
        ]);

        $updatedCount = 0;

        foreach ($request->updates as $updateData) {
            $forecast = SellerForecast::find($updateData['id']);

            if ($forecast) {
                // Update actual sales if provided
                if (isset($updateData['actual_sales'])) {
                    $forecast->actual_sales = $updateData['actual_sales'];
                }

                // Update orders count if provided
                if (isset($updateData['orders_count'])) {
                    $forecast->orders_count = $updateData['orders_count'];
                }

                // Recalculate AOV (Average Order Value)
                if ($forecast->orders_count > 0) {
                    $forecast->aov = $forecast->actual_sales / $forecast->orders_count;
                } else {
                    $forecast->aov = 0;
                }

                // Save first
                $forecast->save();

                // Update performance metrics (achievement percentage and grade)
                $forecast->updatePerformanceMetrics();

                $updatedCount++;
            }
        }

        return response()->json([
            'success' => true,
            'message' => "Successfully updated {$updatedCount} records",
            'count' => $updatedCount
        ]);
    }
}
