<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\User;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\SalesReportExport;

class SalesReportController extends Controller
{
    /**
     * Display sales report
     */
    public function index(Request $request)
    {
        $user = $request->user();

        // Get time period
        $period = $request->input('period', 'today');
        $customStart = $request->input('start_date');
        $customEnd = $request->input('end_date');

        // Calculate date range
        $dateRange = $this->getDateRange($period, $customStart, $customEnd);

        // Get orders based on user role
        $orders = $this->getOrdersForUser($user, $dateRange['start'], $dateRange['end']);

        // Calculate statistics
        $statistics = $this->calculateStatistics($orders);

        // Platform breakdown
        $platformBreakdown = $this->getPlatformBreakdown($orders);

        // Payment method breakdown
        $paymentBreakdown = $this->getPaymentBreakdown($orders);

        // Status breakdown
        $statusBreakdown = $this->getStatusBreakdown($orders);

        // Get detailed orders with products
        $detailedOrders = $this->getDetailedOrders($orders);

        return view('reports.sales.index', compact(
            'user',
            'period',
            'dateRange',
            'statistics',
            'platformBreakdown',
            'paymentBreakdown',
            'statusBreakdown',
            'detailedOrders'
        ));
    }

    /**
     * Export to PDF
     */
    public function exportPdf(Request $request)
    {
        $user = $request->user();

        // Get time period
        $period = $request->input('period', 'today');
        $customStart = $request->input('start_date');
        $customEnd = $request->input('end_date');

        // Calculate date range
        $dateRange = $this->getDateRange($period, $customStart, $customEnd);

        // Get orders
        $orders = $this->getOrdersForUser($user, $dateRange['start'], $dateRange['end']);

        // Calculate statistics
        $statistics = $this->calculateStatistics($orders);
        $platformBreakdown = $this->getPlatformBreakdown($orders);
        $paymentBreakdown = $this->getPaymentBreakdown($orders);
        $statusBreakdown = $this->getStatusBreakdown($orders);
        $detailedOrders = $this->getDetailedOrders($orders);

        $pdf = Pdf::loadView('reports.sales.pdf', compact(
            'user',
            'dateRange',
            'statistics',
            'platformBreakdown',
            'paymentBreakdown',
            'statusBreakdown',
            'detailedOrders'
        ));

        $pdf->setPaper('a4', 'portrait');

        $filename = 'Sales_Report_' . $dateRange['label'] . '_' . date('Y-m-d') . '.pdf';

        return $pdf->download($filename);
    }

    /**
     * Export to Excel
     */
    public function exportExcel(Request $request)
    {
        $user = $request->user();

        // Get time period
        $period = $request->input('period', 'today');
        $customStart = $request->input('start_date');
        $customEnd = $request->input('end_date');

        // Calculate date range
        $dateRange = $this->getDateRange($period, $customStart, $customEnd);

        // Get orders
        $orders = $this->getOrdersForUser($user, $dateRange['start'], $dateRange['end']);

        $filename = 'Sales_Report_' . $dateRange['label'] . '_' . date('Y-m-d') . '.xlsx';

        return Excel::download(new SalesReportExport($orders, $dateRange, $user), $filename);
    }

    /**
     * Get date range based on period
     */
    private function getDateRange($period, $customStart = null, $customEnd = null)
    {
        $now = Carbon::now();

        switch ($period) {
            case 'today':
                return [
                    'start' => $now->copy()->startOfDay(),
                    'end' => $now->copy()->endOfDay(),
                    'label' => 'Today'
                ];

            case 'yesterday':
                return [
                    'start' => $now->copy()->subDay()->startOfDay(),
                    'end' => $now->copy()->subDay()->endOfDay(),
                    'label' => 'Yesterday'
                ];

            case 'this_week':
                return [
                    'start' => $now->copy()->startOfWeek(),
                    'end' => $now->copy()->endOfWeek(),
                    'label' => 'This Week'
                ];

            case 'this_month':
                return [
                    'start' => $now->copy()->startOfMonth(),
                    'end' => $now->copy()->endOfMonth(),
                    'label' => 'This Month'
                ];

            case 'last_month':
                return [
                    'start' => $now->copy()->subMonth()->startOfMonth(),
                    'end' => $now->copy()->subMonth()->endOfMonth(),
                    'label' => 'Last Month'
                ];

            case 'ytd':
                return [
                    'start' => $now->copy()->startOfYear(),
                    'end' => $now->copy()->endOfDay(),
                    'label' => 'Year to Date'
                ];

            case 'custom':
                $start = $customStart ? Carbon::parse($customStart)->startOfDay() : $now->copy()->startOfMonth();
                $end = $customEnd ? Carbon::parse($customEnd)->endOfDay() : $now->copy()->endOfDay();
                return [
                    'start' => $start,
                    'end' => $end,
                    'label' => $start->format('M d') . ' - ' . $end->format('M d, Y')
                ];

            default:
                return [
                    'start' => $now->copy()->startOfDay(),
                    'end' => $now->copy()->endOfDay(),
                    'label' => 'Today'
                ];
        }
    }

    /**
     * Get orders based on user role
     */
    private function getOrdersForUser($user, $start, $end)
    {
        $query = Order::whereBetween('created_at', [$start, $end])
            ->whereNotIn('status', ['cancelled', 'failed']);

        if ($user->isSeller()) {
            // Sellers see only their own orders
            $query->where('created_by', $user->id);
        } elseif ($user->isManager()) {
            // Managers see team orders
            $teamMemberIds = $user->managedSellers->pluck('id')->toArray();
            $teamMemberIds[] = $user->id; // Include manager's own orders
            $query->whereIn('created_by', $teamMemberIds);
        }
        // Admin sees all orders (no filter)

        return $query->orderBy('created_at', 'desc')->get();
    }

    /**
     * Calculate statistics
     */
    private function calculateStatistics($orders)
    {
        $totalOrders = $orders->count();
        $totalRevenue = $orders->sum('total');
        $avgOrderValue = $totalOrders > 0 ? $totalRevenue / $totalOrders : 0;

        // Find top platform
        $platformCounts = $orders->groupBy(function($order) {
            return $this->getPlatformName($order);
        })->map->count();

        $topPlatform = $platformCounts->sortDesc()->keys()->first() ?? 'N/A';

        return [
            'total_orders' => $totalOrders,
            'total_revenue' => $totalRevenue,
            'avg_order_value' => $avgOrderValue,
            'top_platform' => $topPlatform
        ];
    }

    /**
     * Get platform breakdown
     */
    private function getPlatformBreakdown($orders)
    {
        $breakdown = $orders->groupBy(function($order) {
            return $this->getPlatformName($order);
        })->map(function($platformOrders) use ($orders) {
            $count = $platformOrders->count();
            $revenue = $platformOrders->sum('total');
            $percentage = $orders->count() > 0 ? ($count / $orders->count()) * 100 : 0;

            return [
                'count' => $count,
                'revenue' => $revenue,
                'percentage' => $percentage
            ];
        })->sortByDesc('revenue');

        return $breakdown;
    }

    /**
     * Get payment method breakdown
     */
    private function getPaymentBreakdown($orders)
    {
        $breakdown = $orders->groupBy(function($order) {
            return $this->getPaymentMethodName($order->payment_method);
        })->map(function($paymentOrders) {
            return [
                'count' => $paymentOrders->count(),
                'revenue' => $paymentOrders->sum('total')
            ];
        })->sortByDesc('revenue');

        return $breakdown;
    }

    /**
     * Get status breakdown
     */
    private function getStatusBreakdown($orders)
    {
        $breakdown = $orders->groupBy('status')->map(function($statusOrders) {
            return [
                'count' => $statusOrders->count(),
                'revenue' => $statusOrders->sum('total')
            ];
        })->sortByDesc('count');

        return $breakdown;
    }

    /**
     * Get detailed orders with products
     */
    private function getDetailedOrders($orders)
    {
        return $orders->map(function($order) {
            return [
                'id' => $order->id,
                'date' => Carbon::parse($order->created_at)->format('M d, Y'),
                'time' => Carbon::parse($order->created_at)->format('h:i A'),
                'order_number' => $order->order_number,
                'platform' => $this->getPlatformName($order),
                'customer_name' => $this->getCustomerName($order),
                'phone' => $this->getCustomerPhone($order),
                'address' => $this->getCustomerAddress($order),
                'products' => $this->getProductsList($order),
                'payment_method' => $this->getPaymentMethodName($order->payment_method),
                'status' => ucfirst($order->status),
                'total' => $order->total,
                'currency' => $order->currency ?? 'MYR'
            ];
        });
    }

    /**
     * Get platform name from order
     */
    private function getPlatformName($order)
    {
        // Check order number prefix
        if (str_starts_with($order->order_number, 'CK-')) {
            return 'Checkout Form';
        } elseif (str_starts_with($order->order_number, 'SP-')) {
            return 'Sales Page';
        } elseif (str_starts_with($order->order_number, 'WOO-') || $order->store_id) {
            return 'WooCommerce';
        } elseif (str_starts_with($order->order_number, 'CH-')) {
            return 'Channel';
        } else {
            return 'Manual';
        }
    }

    /**
     * Get payment method name
     */
    private function getPaymentMethodName($method)
    {
        if (str_contains(strtolower($method ?? ''), 'cod')) {
            return 'COD';
        } elseif (in_array(strtolower($method ?? ''), ['bayarcash', 'bayarcash-wc', 'online', 'fpx'])) {
            return 'Online Payment';
        } else {
            return ucfirst($method ?? 'N/A');
        }
    }

    /**
     * Get customer name
     */
    private function getCustomerName($order)
    {
        if (is_array($order->billing) && isset($order->billing['first_name'])) {
            return trim(($order->billing['first_name'] ?? '') . ' ' . ($order->billing['last_name'] ?? ''));
        }
        return $order->customer_name ?? 'N/A';
    }

    /**
     * Get customer phone
     */
    private function getCustomerPhone($order)
    {
        if (is_array($order->billing) && isset($order->billing['phone'])) {
            return $order->billing['phone'];
        }
        return $order->customer_phone ?? 'N/A';
    }

    /**
     * Get customer address
     */
    private function getCustomerAddress($order)
    {
        if (is_array($order->billing)) {
            $parts = array_filter([
                $order->billing['address_1'] ?? '',
                $order->billing['address_2'] ?? '',
                $order->billing['city'] ?? '',
                $order->billing['state'] ?? '',
                $order->billing['postcode'] ?? ''
            ]);
            return implode(', ', $parts) ?: 'N/A';
        }
        return $order->customer_address ?? 'N/A';
    }

    /**
     * Get products list with quantities
     */
    private function getProductsList($order)
    {
        $products = [];

        if (is_array($order->line_items) && count($order->line_items) > 0) {
            foreach ($order->line_items as $item) {
                $products[] = [
                    'name' => $item['name'] ?? 'Product',
                    'quantity' => $item['quantity'] ?? 1,
                    'price' => $item['total'] ?? 0
                ];
            }
        } else {
            // Fallback if no line items
            $products[] = [
                'name' => 'Order Items',
                'quantity' => 1,
                'price' => $order->total
            ];
        }

        return $products;
    }
}
